/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 14.11.2008 - [HM] - creation
 * 10.09.2012 - [JR] - #596: checked classes instead of instances
 */
package javax.rad.genui;

import javax.rad.ui.IFactory;

/**
 * Platform and technology independent UIFactoryManager.
 * This is a static provider for the current IFactory.
 * 
 * @author Martin Handsteiner
 */
public class UIFactoryManager
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The global factory instance. */
	private static IFactory factory = null;
	
	/** The global factory class. */
	private static Class factoryClass = null;
	
	/** The thread local factory instance. */
	private static ThreadLocal<IFactory> threadLocal = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creation of <code>UIFactoryManager</code> is not allowed.
	 */
	protected UIFactoryManager()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * Gets the global {@link IFactory} singleton instance.
	 * 
	 * @return the global {@link IFactory} singleton instance.
	 * @see IFactory
	 */
	public static IFactory getFactory()
	{
		if (threadLocal != null)
		{
			IFactory tempFactory = threadLocal.get();
			if (tempFactory != null)
			{
				return tempFactory;
			}
		}
		return factory;
	}
	
	/**
	 * Sets the global {@link IFactory} singleton instance.
	 * 
	 * @param pFactory the factory
	 */
	public static void setFactoryInstance(IFactory pFactory)
	{
		if (factory == null)
		{
			factory = pFactory;
			factoryClass = pFactory.getClass();
			
			UIImage.setDefaults();
		}
		else if (pFactory == null || factory.getClass() != pFactory.getClass())
		{
			throw new IllegalStateException("Only one factory class is supported per virtual machine instance!");
		}
	}
	
	/**
	 * Creates the global {@link IFactory} singleton instance.
	 * 
	 * @param pFactoryClass the factory class.
	 * @return the singleton {@link IFactory} instance.
	 * @see IFactory
	 */
	public static IFactory getFactoryInstance(Class pFactoryClass)
	{
		try
		{
			if (factory == null)
			{
				if (threadLocal != null)
				{
					throw new IllegalStateException("A static factory instance can not be created in threaded factory mode!");
				}

				factory = (IFactory)pFactoryClass.newInstance();
				
				factoryClass = pFactoryClass;

				UIImage.setDefaults();
			}
			else if (factoryClass != pFactoryClass)
			{
				throw new IllegalStateException("Only one factory class is supported per virtual machine instance!");
			}
		}
		catch (Exception ex)
		{
			throw new IllegalArgumentException("It is impossible to instanciate an IFactory with the given class!", ex);
		}
		return factory;
	}
	
	/**
	 * Registers the {@link IFactory} instance for the current Thread.
	 * 
	 * @param pFactory the factory.
	 * @see IFactory
	 */
	public static void registerThreadFactoryInstance(IFactory pFactory)
	{
		try
		{
			boolean firstRegistration = false;

			if (factory != null)
			{
				throw new IllegalStateException("The threaded factory registration can not be used in static factory mode!");
			}
			else if (factoryClass == null)
			{
				factoryClass = pFactory.getClass();
				
				threadLocal = new ThreadLocal<IFactory>();
				
				firstRegistration = true;
			}
			else if (factoryClass != pFactory.getClass())
			{
				throw new IllegalStateException("Only one factory class is supported in per virtual machine instance!");
			}
			
			threadLocal.set(pFactory);

			if (firstRegistration)
			{
				UIImage.setDefaults();
			}
		}
		catch (Exception pEx)
		{
			throw new IllegalArgumentException("It is impossible to register the given IFactory!", pEx);
		}
	}
	
	/**
	 * Unregisters the {@link IFactory} instance for the current Thread.
	 * If the factory is not unregistered, it is overwritten in the next registration of any Factory
	 * in this thread.
	 * The factories are stored in a weak hash map. But anyway it is the best way to finally unregister
	 * the Factory if it should not be used anymore, to get deterministic behaviour. 
	 * 
	 * @see IFactory
	 */
	public static void unregisterThreadFactoryInstance()
	{
		if (threadLocal != null)
		{
			threadLocal.set(null);
		}
	}
	
}	// UIFactoryManager
