/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.11.2008 - [HM] - creation
 * 23.07.2009 - [JR] - checkAdd: allow sub toolbar
 * 04.10.2011 - [JR] - #477: beforeAddNotify handling    
 * 22.10.2011 - [JR] - setZOrder: used ((UIContainer)this) because jdk 1.5 <= 06 can not compile the code!                 
 */
package javax.rad.genui;

import java.util.ArrayList;
import java.util.List;

import javax.rad.genui.container.UITabsetPanel;
import javax.rad.ui.IComponent;
import javax.rad.ui.IContainer;
import javax.rad.ui.IImage;
import javax.rad.ui.ILayout;
import javax.rad.ui.container.IToolBar;
import javax.rad.ui.menu.IMenuBar;

/**
 * Platform and technology independent Container.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF,... .
 * 
 * @author Martin Handsteiner
 * 
 * @param <C> instance of IContainer
 */
public abstract class UIContainer<C extends IContainer> extends UIComponent<C> 
                                               implements IContainer
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The UIParent of this UICompoennt. */
	private ILayout uiLayout = null;
	
	/** List of subcomponents. */
	protected List<IComponent> components = new ArrayList<IComponent>(4);
	
	/** the flag indicates that addNotify is active. */
	private boolean bAddNotify = false;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>UIContainer</code>.
     *
     * @param pContainer the Container.
     * @see IContainer
     */
	protected UIContainer(C pContainer)
	{
		super(pContainer);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface Implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public ILayout getLayout()
    {
    	return uiLayout;
    }

	/**
	 * {@inheritDoc}
	 */
	public void setLayout(ILayout pLayout)
    {
		// ensure that the factory gets his own resource, to prevent exception on factory internal casts!
		if (pLayout instanceof UILayout)
		{
	    	uiResource.setLayout(((UILayout<ILayout, ?>)pLayout).uiResource);
		}
		else
		{
	    	uiResource.setLayout(pLayout);
		}
		uiLayout = pLayout;
    }

	/**
	 * {@inheritDoc}
	 */
	public void add(IComponent pComponent)
	{
		add(pComponent, null, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	public void add(IComponent pComponent, Object pConstraints)
	{
		add(pComponent, pConstraints, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	public void add(IComponent pComponent, int pIndex)
	{
		add(pComponent, null, pIndex);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void add(IComponent pComponent, Object pConstraints, int pIndex)
    {
		checkAdd(pComponent, pConstraints, pIndex);
		
		// Remove from UIContainer or from IContainer	
		if (pComponent.getParent() != null)
		{
			pComponent.getParent().remove(pComponent);
		}
		
		// Support both, UIComponents and IComponents
		IComponent component;
		if (pComponent instanceof UIComponent)
		{
			component = ((UIComponent<IComponent>)pComponent).getComponentUIResource();
		}
		else
		{
			component = pComponent;
		}
		Object constraints;
		if (pConstraints instanceof UIResource)
		{
			constraints = ((UIResource)pConstraints).getUIResource();
		}
		else
		{
			constraints = pConstraints;
		}

		if (pComponent instanceof UIComponent)
		{
			//send beforeAddNotify if the container is already added
			if (isNotified() && !((UIComponent)pComponent).isBeforeNotified())
			{
				((UIComponent)pComponent).beforeAddNotify(this);
			}
		}		
		
		uiResource.add(component, constraints, pIndex);

		if (pIndex < 0)
		{
			components.add(pComponent);
		}
		else
		{
			components.add(pIndex, pComponent);
		}
		
		if (pComponent instanceof UIComponent)
		{
			pComponent.setParent(this);

			//send addNotify if the container is already added
			if (isNotified() && !((UIComponent)pComponent).isNotified())
			{
				((UIComponent)pComponent).addNotify();
			}
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public void remove(int pIndex)
    {
		IComponent comp = components.get(pIndex);
		
		if (comp instanceof UIComponent)
		{
			uiResource.remove((IComponent)((UIComponent)comp).getComponentUIResource());
			
			if (((UIComponent)comp).isNotified())
			{
				((UIComponent)comp).removeNotify();
			}
		}
		else
		{
			uiResource.remove(comp);
		}

		IComponent component = components.remove(pIndex);
		
		if (component instanceof UIComponent)
		{
			component.setParent(null);
		}
    }

	/**
	 * {@inheritDoc}
	 */
	public void remove(IComponent pComponent)
	{
		if (pComponent.getParent() == this)
		{
			remove(components.indexOf(pComponent));
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void removeAll()
	{
		while (components.size() > 0)
		{
			remove(components.size() - 1);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public int getComponentCount()
	{
		return components.size();
	}

	/**
	 * {@inheritDoc}
	 */
	public IComponent getComponent(int pIndex)
	{
		return components.get(pIndex);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int indexOf(IComponent pComponent)
	{
		return components.indexOf(pComponent);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void updateTranslation()
	{
		super.updateTranslation();
		
		//updateTranslation will be called through addNotify -> don't call it more than once!
		if (bAddNotify)
		{
			return;
		}
		
		IComponent comp;
		
		//Update the translation for all sub components
		for (int i = 0, anz = components.size(); i < anz; i++)
		{
			comp = components.get(i);
			
			if (comp instanceof UIComponent)
			{
				((UIComponent)comp).updateTranslation();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void beforeAddNotify(IComponent pParent)
	{
		if (bAddNotify)
		{
			return;
		}
		
		bAddNotify = true;
		
		try
		{
			super.beforeAddNotify(pParent);
		}
		finally
		{
			bAddNotify = false;
		}
	
		IComponent comp;
		
		
		for (int i = 0, anz = components.size(); i < anz; i++)
		{
			comp = components.get(i);
			
			if (comp instanceof UIComponent && !((UIComponent)comp).isBeforeNotified())
			{
				((UIComponent)comp).beforeAddNotify(this);
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addNotify()
	{
		if (bAddNotify)
		{
			return;
		}
		
		bAddNotify = true;
		
		try
		{
			super.addNotify();
		}
		finally
		{
			bAddNotify = false;
		}
	
		IComponent comp;
		
		
		for (int i = 0, anz = components.size(); i < anz; i++)
		{
			comp = components.get(i);
			
			if (comp instanceof UIComponent && !((UIComponent)comp).isNotified())
			{
				((UIComponent)comp).addNotify();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeNotify()
	{
		super.removeNotify();

		IComponent comp;
		
		for (int i = 0, anz = components.size(); i < anz; i++)
		{
			comp = components.get(i);
			
			if (comp instanceof UIComponent && ((UIComponent)comp).isNotified())
			{
				((UIComponent)comp).removeNotify();
			}
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Checks if it's allowed to add a specific component to this container.
	 * 
	 * @param pComponent the component to be added
	 * @param pConstraints an object expressing layout constraints
	 * @param pIndex the position in the container's list at which to insert the IComponent; -1 means insert at the end component
	 */
	protected void checkAdd(IComponent pComponent, Object pConstraints, int pIndex)
	{
		if (!(this instanceof IToolBar) && pComponent instanceof IToolBar)
		{
			throw new IllegalArgumentException("It's not supported to 'add' an IToolBar. Use 'addToolBar'!"); 
		}
		
		if (pComponent instanceof IMenuBar)
		{
			throw new IllegalArgumentException("It's not supported to 'add' an IMenuBar. Use 'setMenuBar'!");
		}
	}
	
	/**
	 * Sets the order of the given component.
	 * @param pComponent the component
	 * @param pZOrder the zOrder.
	 */
	public void setZOrder(IComponent pComponent, int pZOrder)
	{
		int index = indexOf(pComponent);
		if (index >= 0)
		{
			Object constraints = null;
			IImage image = null;
			if (getLayout() != null)
			{
				constraints = getLayout().getConstraints(pComponent);
			}
			else if (((UIContainer)this) instanceof UITabsetPanel)
			{
				constraints = ((UITabsetPanel)((UIContainer)this)).getTextAt(index);
				image = ((UITabsetPanel)((UIContainer)this)).getIconAt(index);
			}
			
			if (pComponent instanceof UIComponent)
			{
				uiResource.remove((IComponent)((UIComponent)pComponent).getComponentUIResource());
			}
			else
			{
				uiResource.remove(pComponent);
			}
			components.remove(index);
			if (pZOrder > components.size())
			{
				pZOrder = components.size();
			}
			components.add(pZOrder, pComponent);

			if (pComponent instanceof UIComponent)
			{
				pComponent = ((UIComponent<IComponent>)pComponent).getComponentUIResource();
			}
			if (constraints instanceof UIResource)
			{
				constraints = ((UIResource)constraints).getUIResource();
			}
			uiResource.add(pComponent, constraints, pZOrder);
			
			if (image != null)
			{
				((UITabsetPanel)((UIContainer)this)).setIconAt(index, image);
			}
		}
	}
	
}	// UIContainer
