/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 26.10.2009 - [HM] - creation
 * 13.02.2010 - [JR] - format with Date and long parameter implemented
 * 08.03.2012 - [JR] - #556: getListSeparator implemented
 */
package com.sibvisions.util.type;

import java.text.DecimalFormatSymbols;
import java.util.Locale;

import com.sibvisions.util.Internalize;

/**
 * The <code>DateUtil</code> is a utility class for date conversion and for formatting dates
 * as string.
 *  
 * @author Martin Handsteiner
 */
public final class LocaleUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class Members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The default locale. */
	private static Locale defaultLocale;
	
	/** The default thread locale. */
	private static ThreadLocal<Locale> threadDefaultLocale;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creating a LocaleUtil is not allowed.
	 */
	private LocaleUtil()
	{
		// Utility Class
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Get a locale parsing language, country and variant.
	 * 
	 * @param pLanguageTag the language tag.
	 * @return the Locale.
	 */
	@SuppressWarnings("unused")
	public static Locale forLanguageTag(String pLanguageTag)
	{
		// Default Locale schould be base of intern functionality.
		Locale internLocale = Internalize.intern(Locale.getDefault());
		if (pLanguageTag == null || pLanguageTag.length() == 0)
		{
			return Locale.getDefault();
		}
		else
		{
			int langEnd = pLanguageTag.indexOf('_');
			
			if (langEnd < 0)
			{
				return Internalize.intern(new Locale(pLanguageTag));
			}
			else
			{
				int countryEnd = pLanguageTag.indexOf('_', langEnd + 1);
				
				if (countryEnd < 0)
				{
					return Internalize.intern(new Locale(pLanguageTag.substring(0, langEnd), pLanguageTag.substring(langEnd + 1)));
				}
				else
				{
					return Internalize.intern(new Locale(
							pLanguageTag.substring(0, langEnd), 
							pLanguageTag.substring(langEnd + 1, countryEnd), 
							pLanguageTag.substring(countryEnd + 1)));
				}
			}
		}
	}
	
	/**
	 * Gets the default locale. If no locale is set, the default local is returned.
	 * 
	 * @return the default locale.
	 */
	public static Locale getDefault()
	{
		if (threadDefaultLocale != null)
		{
			Locale locale = threadDefaultLocale.get();
			if (locale != null)
			{
				return locale;
			}
		}
		if (defaultLocale != null)
		{
			return defaultLocale; 
		}
		return Locale.getDefault();
	}
	
	/**
	 * Gets the default locale. If no locale is set, the default local is returned.
	 * 
	 * @param pLocale the default locale.
	 */
	public static void setDefault(Locale pLocale)
	{
		defaultLocale = pLocale;
	}
	
	/**
	 * Gets the default locale. If no locale is set, the default local is returned.
	 * 
	 * @param pLocale the default locale.
	 */
	public static void setThreadDefault(Locale pLocale)
	{
		if (threadDefaultLocale == null)
		{
			threadDefaultLocale = new ThreadLocal<Locale>();
		}
		threadDefaultLocale.set(pLocale);
	}

	/**
	 * Gets the list separator for the current locale.
	 * 
	 * @return the list separator e.g. <code>;</code> in Europe or <code>,</code> in US
	 * @see #getListSeparator(Locale)
	 */
	public static String getListSeparator()
	{
		return getListSeparator(getDefault());
	}
	
	/**
	 * Gets the list separator for the given locale.
	 *
	 * @param pLocale the locale to use
	 * @return the list separator e.g. <code>;</code> in Europe or <code>,</code> in US
	 */
	public static String getListSeparator(Locale pLocale)
	{
		DecimalFormatSymbols dfs = new DecimalFormatSymbols(pLocale);
		
		
		if (dfs.getDecimalSeparator() == '.')
		{
			return ",";
		}
		else
		{
			return ";";
		}
	}
	
}	// LocaleUtil
