/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 08.04.2009 - [JR] - creation
 * 30.11.2010 - [JR] - equals implemented
 * 11.02.2011 - [JR] - getFirstCause implemented
 * 29.07.2011 - [JR] - getCauseList implemented
 * 12.09.2011 - [JR] - getFreePort implemented
 * 07.12.2012 - [JR] - isReachable implemented
 */
package com.sibvisions.util.type;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketAddress;
import java.util.List;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>CommonUtil</code> contains utility methods for handling
 * type independent operations.
 * 
 * @author Ren Jahn
 */
public final class CommonUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor because <code>CommonUtil</code> is a utility
	 * class.
	 */
	private CommonUtil()
	{
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Writes the stack trace of an exception/throwable object into a string.
	 * 
	 * @param pCause Exception/Throwable (StackTrace)
	 * @param pDeep <code>true</code> to dump the trace with all causes; <code>false</code> to dump
	 *        only the exception
	 * @return stack trace  
	 */
	public static String dump(Throwable pCause, boolean pDeep)
	{
		if (pCause == null)
		{
			return "";
		}
		else
		{
	    	StringWriter sw = new StringWriter(512);
	    	PrintWriter pw = new PrintWriter(sw);
	        
	    	if (pDeep)
	    	{
	    		pCause.printStackTrace(pw);
	    	}
	    	else
	    	{
	    		pw.println(pCause);
	    		
	            StackTraceElement[] trace = pCause.getStackTrace();
	            for (int i = 0, anz = trace.length; i < anz; i++)
	            {
	                pw.println("\tat " + trace[i]);
	            }
	    	}
	        
	        pw.close();
	        
	        return sw.toString();
		}
	}

	/**
	 * Gets the first cause.
	 * 
	 * @param pCause the origin cause
	 * @return the first cause
	 */
	public static Throwable getFirstCause(Throwable pCause)
	{
		if (pCause == null)
		{
			return null;
		}
		
		Throwable th = pCause;
		
		while (th.getCause() != null)
		{
			th = th.getCause();
		}
		
		return th;
	}
	
	/**
	 * Gets the list of all available exceptions.
	 * 
	 * @param pCause the start exception
	 * @return the list of all causes and the exception itself. The start exception is
	 *         the first entry and the cause of the start exception is the second entry
	 *         and so on.
	 */
	public static List<Throwable> getCauseList(Throwable pCause)
	{
		ArrayUtil<Throwable> auThrowable = new ArrayUtil<Throwable>();
		
		if (pCause == null)
		{
			return auThrowable;
		}
		
		Throwable th = pCause;
		
		while (th.getCause() != null)
		{
			auThrowable.add(th);
			
			th = th.getCause();
		}

		//that means: pCause has no cause
		if (th != null)
		{
			auThrowable.add(th);
		}

		return auThrowable;
	}
	
	/**
	 * Gets an alternative value for a <code>null</code> object.
	 * 
	 * @param <T> parameter type
	 * @param pValue desired value
	 * @param pNvlValue alternative value 
	 * @return <code>pValue</code> or <code>pNvlValue</code> if <code>pValue == null</code>
	 */
	public static <T> T nvl(T pValue, T pNvlValue)
	{
		if (pValue == null)
		{
			return pNvlValue;
		}
		
		return pValue;
	}
	
	/**
	 * Indicates whether two object are "equal". Two objects are equals if both are <code>null</code> or
	 * the {@link Object#equals(Object)} returns <code>true</code>.
	 *   
	 * @param pFirst an object
	 * @param pSecond another object
	 * @return <code>true</code> if both objects are equal
	 */
	public static boolean equals(Object pFirst, Object pSecond)
	{
		return pFirst == pSecond || (pFirst != null && pFirst.equals(pSecond));
	}

    /**
     * Searchs a free port in the given range.
     * 
     * @param pMin the min port
     * @param pMax the max port
     * @return -1 if no free port is available in the given range
     */
    public static int getFreePort(int pMin, int pMax)
    {
    	ServerSocket ssok = null;
    	
    	
    	for (int i = pMin; i <= pMax; i++)
    	{
    		try
    		{
    			ssok = new ServerSocket(i);
    			ssok.setReuseAddress(true);
    			
    			return i;
    		}
    		catch (IOException ioe)
    		{
    			//try next port
    		}
    		finally
    		{
    			if (ssok != null)
    			{
    				try
    				{
    					ssok.close();
    				}
    				catch (Exception e)
    				{
    					//nothing to be done
    				}
    			}

    		}
    	}
    	
    	return -1;
    }	
    
    /**
     * Tests if the given port is reachable on the given host.
     * 
     * @param pHost the hostname or IP
     * @param pPort th port number
     * @return <code>true</code> if the port is reachable, <code>false</code> otherwise
     */
    public static boolean isReachable(String pHost, int pPort)
    {
    	Socket sok = null;
    	
		try
		{
			sok = new Socket();
	        sok.setReuseAddress(true);
	        sok.setSoTimeout(3000);
	        
	        SocketAddress sa = new InetSocketAddress(pHost, pPort);
	        sok.connect(sa, 3000);
	        
	        return true;
		}
		catch (Exception e)
		{
			return false;
		}
		finally
		{
			if (sok != null)
			{
				try
				{
			        sok.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
		}
    }
	
}	// CommonUtil
