/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 18.03.2009 - [JR] - creation
 * 10.10.2011 - [JR] - #479: open files on linux (kde, gnome)
 */
package com.sibvisions.util;

import java.io.File;
import java.io.IOException;

/**
 * The <code>FileViewer</code> opens files/documents with the platform dependent viewer
 * application.
 * 
 * @author Ren Jahn
 */
public final class FileViewer
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Invisible constructor, because the <code>FileViewer</code> class is a utility class.
	 */
	private FileViewer()
	{
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Opens a document with the platform dependent viewer.
	 * 
	 * @param pDocument the document as File reference.
	 * @throws IOException if the viewer can not open the document
	 */
	public static void open(File pDocument) throws IOException
	{
		open('\"' + pDocument.getAbsolutePath() + '\"');
	}
	
	/**
	 * Opens a document with the platform dependent viewer.
	 * 
	 * @param pDocumentName the absolute path to the document
	 * @throws IOException if the viewer can not open the document
	 */
	public static void open(String pDocumentName) throws IOException
	{
		String sOs = System.getProperty("os.name").toLowerCase(); 
		
		if (sOs.startsWith("windows"))
		{
			Runtime.getRuntime().exec(getCommandArray("rundll32 url.dll,FileProtocolHandler " + pDocumentName));
		}
		else if (sOs.indexOf("linux") >= 0)
		{
			String sWM = System.getenv("GDMSESSION");
			
			if (sWM == null)
			{
				sWM = System.getenv("DESKTOP_SESSION");
			}
			
			if (sWM == null)
			{
				throw new IOException("WindowManager detection failed!");
			}
			
			sWM = sWM.toLowerCase();
			
			int iExitValue = -1;
			
			if (sWM.indexOf("gnome") >= 0)
			{
				Process proc = Runtime.getRuntime().exec(getCommandArray("gnome-open " + pDocumentName));
				
				try
				{
					iExitValue = proc.waitFor();
				}
				catch (InterruptedException ie)
				{
					iExitValue = -1;
				}
			}
			else if (sWM.indexOf("kde") >= 0)
			{
				Process proc = Runtime.getRuntime().exec(getCommandArray("kde-open " + pDocumentName));

				try
				{
					iExitValue = proc.waitFor();
				}
				catch (InterruptedException ie)
				{
					iExitValue = -1;
				}
			}
			else
			{
				iExitValue = -1;
			}
			
			if (iExitValue != 0)
			{
				Runtime.getRuntime().exec(getCommandArray("xdg-open " + pDocumentName));
			}
		}
		else if (sOs.indexOf("mac") >= 0)
		{
			Runtime.getRuntime().exec(getCommandArray("open " + pDocumentName));
		}
		else
		{
			throw new IOException("The document [" + pDocumentName + "] can not be opened!");
		}
	}
	
	/**
	 * Returns a protected operating system command as String[].
	 * 
	 * @param pCommand		the command line to protect
	 * @return a protected operating system command line as String[].
	 */
	private static String[] getCommandArray(String pCommand)
	{
		ArrayUtil<String> auCmds = new ArrayUtil<String>();
		StringBuffer sbCmd = new StringBuffer();
		
		boolean bIgnore = false;
		int i = 0;
		while (i < pCommand.length())
		{
			char ch = pCommand.charAt(i);
			if (bIgnore || " \t\n\r\f".indexOf(ch) < 0)
			{
				if (ch == '\"')
				{
					bIgnore = !bIgnore;
				}
				sbCmd.append(ch);
			}
			else if (sbCmd.length() > 0)
			{
				auCmds.add(sbCmd.toString());
				sbCmd.setLength(0);
			}
			i++;
		}
		
		if (sbCmd.length() > 0)
		{
			auCmds.add(sbCmd.toString());
		}
		
		return auCmds.toArray(new String[auCmds.size()]);
	} 
	
}	//FileViewer
