/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 14.11.2008 - [HM] - creation
 * 25.06.2009 - [JR] - setNumberFormat: forward the pattern to the formatter [BUGFIX]
 * 24.03.2011 - [JR] - #317: cancelEditing checks parents enabled state
 * 25.08.2011 - [JR] - #465: install actions
 */
package com.sibvisions.rad.ui.swing.ext.celleditor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.text.ParseException;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.ICellEditorHandler;
import javax.rad.model.ui.ICellEditorListener;
import javax.rad.model.ui.ICellRenderer;
import javax.rad.ui.IColor;
import javax.rad.ui.celleditor.INumberCellEditor;
import javax.swing.JComponent;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.table.DefaultTableCellRenderer;

import com.sibvisions.rad.ui.swing.ext.ICellFormatterEditorListener;
import com.sibvisions.rad.ui.swing.ext.JVxUtil;
import com.sibvisions.rad.ui.swing.ext.format.CellFormat;
import com.sibvisions.rad.ui.swing.ext.text.NumberFormatter;
import com.sibvisions.rad.ui.swing.impl.SwingFactory;

/**
 * The <code>JVxTextCellEditor</code> provides the generation of the 
 * physical number editor component, handles correct all events, and 
 * gives standard access to edited values.
 * 
 * @author Martin Handsteiner
 */
public class JVxNumberCellEditor extends JVxStyledCellEditor 
                                 implements INumberCellEditor, 
											ICellRenderer<Component>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the content type. */
	private String numberFormat;
	
	/** The cell renderer. */
	private JLabel cellRenderer = null;
	
	/** The cell renderer. */
	private NumberFormatter numberFormatter = new NumberFormatter();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new JVxTextCellEditor.
	 */
	public JVxNumberCellEditor()
	{
		this(null);
	}
	
	/**
	 * Constructs a new JVxNumberCellEditor with the given number format.
	 * @param pNumberFormat the number format.
	 */
	public JVxNumberCellEditor(String pNumberFormat)
	{
		setNumberFormat(pNumberFormat);
		setHorizontalAlignment(ALIGN_RIGHT);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public ICellEditorHandler<JComponent> createCellEditorHandler(ICellEditorListener pCellEditorListener, 
			                                                      IDataRow pDataRow, String pColumnName)
	{
		return new CellEditorHandler(this, (ICellFormatterEditorListener)pCellEditorListener, pDataRow, pColumnName);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDirectCellEditor()
	{
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public String getNumberFormat()
	{
		return numberFormat;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setNumberFormat(String pNumberFormat)
	{
		numberFormat = pNumberFormat;
		
		numberFormatter.setNumberPattern(pNumberFormat);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Component getCellRendererComponent(Component pParentComponent,
								              IDataPage pDataPage,
									          int       pRowNumber,
									          IDataRow  pDataRow,
									          String    pColumnName,
									          boolean   pIsSelected,
									          boolean   pHasFocus)
	{
		if (cellRenderer == null)
		{
			cellRenderer = new DefaultTableCellRenderer();
		}
		cellRenderer.setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(getHorizontalAlignment()));
		cellRenderer.setVerticalAlignment(SwingFactory.getVerticalSwingAlignment(getVerticalAlignment()));
		try
		{
			cellRenderer.setText(numberFormatter.valueToString(pDataRow.getValue(pColumnName)));
		}
		catch (Exception pException)
		{
			cellRenderer.setText(null);
		}
		return cellRenderer;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
     * Sets the internal changed Flag, and informs the CellEditorListener 
     * if editing is completed.
     * 
     * @author Martin Handsteiner
     */
    public static class CellEditorHandler implements ICellEditorHandler<JComponent>,
                                                     DocumentListener, 
                                                     FocusListener, 
                                                     KeyListener
    {
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Class members
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/** The CellEditor, that created this handler. */
    	private JVxNumberCellEditor cellEditor;
    	
    	/** The CellEditorListener to inform, if editing is started or completed. */
    	private ICellFormatterEditorListener cellEditorListener;
    	
    	/** The data row that is edited. */
    	private IDataRow dataRow;
    	
    	/** The column name of the edited column. */
    	private String columnName;

    	/** The physical component that is added to the parent container. */
    	private JFormattedTextField cellEditorComponent;
    	
    	/** True, the Event should be ignored. */
    	private boolean ignoreEvent;
    	
    	/** True, if it's the first editing started event. */
    	private boolean firstEditingStarted = true;
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Initialization
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/**
    	 * Constructs a new CellEditorHandler.
    	 * 
    	 * @param pCellEditor the CellEditor that created this handler.
    	 * @param pCellEditorListener CellEditorListener to inform, if editing is started or completed.
    	 * @param pDataRow the data row that is edited.
    	 * @param pColumnName the column name of the edited column.
    	 */
    	public CellEditorHandler(JVxNumberCellEditor pCellEditor, ICellFormatterEditorListener pCellEditorListener,
    			                 IDataRow pDataRow, String pColumnName)
    	{
    		cellEditor = pCellEditor;
    		cellEditorListener = pCellEditorListener;
    		dataRow = pDataRow;
    		columnName = pColumnName;
    		
    		NumberFormatter numberFormatter = new NumberFormatter();
    		numberFormatter.setNumberPattern(cellEditor.getNumberFormat());
    		
    		try
    		{
    			numberFormatter.setDataType(dataRow.getRowDefinition().getColumnDefinition(columnName).getDataType());
    		}
    		catch (ModelException me)
    		{
    			//nothing to be done
    		}

    		cellEditorComponent = new JFormattedTextField(numberFormatter);
    		cellEditorComponent.setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(cellEditor.getHorizontalAlignment()));
    		cellEditorComponent.setColumns(5);
    		
    		cellEditorComponent.getDocument().addDocumentListener(this);
    		cellEditorComponent.setFocusTraversalKeysEnabled(false);
    		cellEditorComponent.addFocusListener(this);
    		cellEditorComponent.addKeyListener(this);
    		
    		JVxUtil.installActions(cellEditorComponent);
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Interface implementation
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	// ICellEditorHandler
    	
    	/**
    	 * {@inheritDoc}
    	 */
    	public void uninstallEditor()
    	{
    		cellEditorComponent.getDocument().removeDocumentListener(this);
    		cellEditorComponent.removeFocusListener(this);
    		cellEditorComponent.removeKeyListener(this);
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void updateEditor()
    	{
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditor getCellEditor()
    	{
    		return cellEditor;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditorListener getCellEditorListener()
    	{
    		return cellEditorListener;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public IDataRow getDataRow()
    	{
    		return dataRow;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public String getColumnName()
    	{
    		return columnName;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public JComponent getCellEditorComponent()
    	{
    		return cellEditorComponent;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void saveEditing() throws ModelException
    	{
    		try
    		{
				cellEditorComponent.commitEdit();
    		}
    		catch (ParseException pEx)
    		{
    			// try to store
    		}
       		dataRow.setValue(columnName, cellEditorComponent.getValue());
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void cancelEditing() throws ModelException
    	{
    		if (!ignoreEvent)
    		{
	    		ignoreEvent = true;
	
	    		try
	    		{
	    			ColumnDefinition columnDef = dataRow.getRowDefinition().getColumnDefinition(columnName);
	    			
					cellEditorComponent.setValue(dataRow.getValue(columnName));
	
					CellFormat cellFormat = null;
					
					Container conParent = cellEditorComponent.getParent();
					
					boolean bParentEnabled = conParent == null || conParent.isEnabled();
					
		    		if (dataRow instanceof IDataBook)
		    		{
		    			IDataBook dataBook = (IDataBook)dataRow;
		    			cellEditorComponent.setEditable(bParentEnabled
		    					                        && dataBook.isUpdateEnabled() 
		    											&& !columnDef.isReadOnly()
		    											&& dataBook.getSelectedRow() >= 0);

		    			if (cellEditorListener.getCellFormatter() != null)
						{
							try
							{
								cellFormat = cellEditorListener.getCellFormatter().getCellFormat(
										dataBook, dataBook.getDataPage(), dataBook, columnName, dataBook.getSelectedRow(), -1);
							}
							catch (ModelException pModelException)
							{
								// Do nothing
							}
						}
		    		}
					else
					{
						cellEditorComponent.setEditable(bParentEnabled && !columnDef.isReadOnly());
					}
					Color background;
					Color foreground;
					Font font;
					if (cellFormat == null)
					{
						background = null;
						foreground = null;
						font = null;
					}
					else
					{
						background = cellFormat.getBackground();
						foreground = cellFormat.getForeground();
						font = cellFormat.getFont();
					}
					if (font == null)
					{
						font = ((Component)cellEditorListener).getFont();
					}
					if (foreground == null && ((Component)cellEditorListener).isForegroundSet())
					{
						foreground = ((Component)cellEditorListener).getForeground();
					}
					if (background == null && ((Component)cellEditorListener).isBackgroundSet())
					{
						background = ((Component)cellEditorListener).getBackground();
					}

					cellEditorComponent.setFont(font);
		    		if (cellEditorComponent.isEditable())
		    		{
		    			if (background == null)
		    			{
			    			if (columnDef.isNullable())
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_BACKGROUND);
			    			}
			    			else
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_MANDATORY_BACKGROUND);
			    			}
		    			}
						if (cellEditorComponent.hasFocus())
						{
							cellEditorComponent.selectAll();
						}
						else
						{
							cellEditorComponent.select(0, 0);
						}
		    		}
		    		else
		    		{
		    			background = JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND);
						cellEditorComponent.select(0, 0);
		    		}
    				cellEditorComponent.setBackground(background);
    				cellEditorComponent.setForeground(foreground);
	    		}
	    		catch (Exception pException)
	    		{
	    			cellEditorComponent.setValue(null);
	    			cellEditorComponent.setEditable(false);
	    			cellEditorComponent.setBackground(JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND));
	    			
	    			throw new ModelException("Editor cannot be restored!", pException);
	    		}
	    		finally
	    		{
	    			firstEditingStarted = true;
	    			ignoreEvent = false;
	    		}
    		}
    	}
    	
    	// DocumentListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
        public void insertUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       		}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void removeUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       		}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void changedUpdate(DocumentEvent pDocumentEvent) 
        {
        }

    	// FocusListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void focusGained(FocusEvent pFocusEvent)
		{
			if (cellEditorComponent.isEditable())
			{
				// Sets JFormattedTextFields edited true, this prevents setFormatter and therefore document changes on focusGained.
				cellEditorComponent.setText(cellEditorComponent.getText());
				
	    		cellEditorComponent.selectAll();
			}
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void focusLost(FocusEvent pFocusEvent)
		{
			if (!pFocusEvent.isTemporary())
			{
				fireEditingComplete(ICellEditorListener.FOCUS_LOST);
			}
		}

    	// KeyListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyPressed(KeyEvent pKeyEvent)
		{
			if (!pKeyEvent.isConsumed())
			{
				switch (pKeyEvent.getKeyCode())
				{
					case KeyEvent.VK_ESCAPE: 
						pKeyEvent.consume(); 
				        fireEditingComplete(ICellEditorListener.ESCAPE_KEY);
				        break;
					case KeyEvent.VK_ENTER: 
						pKeyEvent.consume();
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_ENTER_KEY);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.ENTER_KEY);
						}
				        break;
					case KeyEvent.VK_TAB: 
						pKeyEvent.consume(); 
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_TAB_KEY);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.TAB_KEY);
						}
				        break;
				    default:
				    	// Nothing to do
				}
			}
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyReleased(KeyEvent pKeyEvent)
		{
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyTyped(KeyEvent pKeyEvent)
		{
		}
        
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// User-defined methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that the event occurs only one time.
		 */
		protected void fireEditingStarted()
		{
        	if (firstEditingStarted && !ignoreEvent && cellEditorListener != null)
        	{
           		firstEditingStarted = false;
           		cellEditorListener.editingStarted();
        	}
		}
		
		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that editing started will be called before,
		 * if it is not called until jet.
		 * 
		 * @param pCompleteType the editing complete type.
		 */
		protected void fireEditingComplete(String pCompleteType)
		{
			if (!ignoreEvent && cellEditorListener != null)
			{
				cellEditorListener.editingComplete(pCompleteType);
			}
		}
		
    }	// CellEditorHandler
	
}	// JVxTextCellEditor
