/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 03.11.2008 - [HM] - creation
 * 07.04.2009 - [JR] - cancelEditing: setEditorEditable instead of setEditable 
 * 24.03.2011 - [JR] - #317: cancelEditing checks parents enabled state
 * 31.03.2011 - [JR] - #161: forward translation change to combobase
 */
package com.sibvisions.rad.ui.swing.ext.celleditor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.ICellEditorHandler;
import javax.rad.model.ui.ICellEditorListener;
import javax.rad.model.ui.ICellRenderer;
import javax.rad.model.ui.IControl;
import javax.rad.ui.IColor;
import javax.rad.ui.celleditor.IDateCellEditor;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.table.DefaultTableCellRenderer;

import com.sibvisions.rad.ui.swing.ext.ICellFormatterEditorListener;
import com.sibvisions.rad.ui.swing.ext.JVxDateCombo;
import com.sibvisions.rad.ui.swing.ext.JVxUtil;
import com.sibvisions.rad.ui.swing.ext.format.CellFormat;
import com.sibvisions.rad.ui.swing.ext.text.DateFormatter;
import com.sibvisions.rad.ui.swing.impl.SwingFactory;

/**
 * The <code>JVxDateCellEditor</code> provides the generation of the 
 * physical Date editor component, handles correct all events, and 
 * gives standard access to edited values.
 * 
 * @author Martin Handsteiner
 */
public class JVxDateCellEditor extends JVxStyledCellEditor 
                               implements IDateCellEditor, 
										  ICellRenderer<Component>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** The cell renderer. */
	private JLabel cellRenderer = null;
	
	/** The cell renderer. */
	private DateFormatter dateFormatter = new DateFormatter();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Constructs a new JVxDateCellEditor.
	 */
	public JVxDateCellEditor()
	{
		this(null);
	}
	
	/**
	 * Constructs a new JVxDateCellEditor with the given date format.
	 * @param pDateFormat the date format.
	 */
	public JVxDateCellEditor(String pDateFormat)
	{
		setDateFormat(pDateFormat);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public ICellEditorHandler<JComponent> createCellEditorHandler(ICellEditorListener pCellEditorListener, 
			                                                      IDataRow pDataRow, String pColumnName)
	{
		return new CellEditorHandler(this, (ICellFormatterEditorListener)pCellEditorListener, pDataRow, pColumnName);
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDirectCellEditor()
	{
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	public String getDateFormat()
	{
		return dateFormatter.getDatePattern();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDateFormat(String pDateFormat)
	{
		dateFormatter.setDatePattern(pDateFormat);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Component getCellRendererComponent(Component pParentComponent,
								              IDataPage pDataPage,
									          int       pRowNumber,
									          IDataRow  pDataRow,
									          String    pColumnName,
									          boolean   pIsSelected,
									          boolean   pHasFocus)
	{
		if (cellRenderer == null)
		{
			cellRenderer = new DefaultTableCellRenderer();
			cellRenderer.setOpaque(true);
		}
		cellRenderer.setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(getHorizontalAlignment()));
		cellRenderer.setVerticalAlignment(SwingFactory.getVerticalSwingAlignment(getVerticalAlignment()));
		try
		{
			cellRenderer.setText(dateFormatter.valueToString(pDataRow.getValue(pColumnName)));
		}
		catch (Exception pException)
		{
			cellRenderer.setText(null);
		}
		return cellRenderer;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

    /**
     * Sets the internal changed flag, and informs the CellEditorListener 
     * if editing is completed.
     * 
     * @author Martin Handsteiner
     */
    public static class CellEditorHandler implements ICellEditorHandler<JComponent>,
                                                     DocumentListener, 
                                                     KeyListener,
                                                     PopupMenuListener,
                                                     FocusListener
    {
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Class members
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/** The CellEditor, that created this handler. */
    	private JVxDateCellEditor cellEditor;
    	
    	/** The CellEditorListener to inform, if editing is started or completed. */
    	private ICellFormatterEditorListener cellEditorListener;
    	
    	/** The data row that is edited. */
    	private IDataRow dataRow;
    	
    	/** The column name of the edited column. */
    	private String columnName;

    	/** The physical component that is added to the parent container. */
    	private JVxDateCombo cellEditorComponent;
    	
    	/** Tells the listener to ignore the events. */
    	private boolean ignoreEvent = false;
    	
    	/** True, if it's the first editing started event. */
    	private boolean firstEditingStarted = true;
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Initialization
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/**
    	 * Constructs a new CellEditorHandler.
    	 * 
    	 * @param pCellEditor the CellEditor that created this handler.
    	 * @param pCellEditorListener CellEditorListener to inform, if editing is started or completed.
    	 * @param pDataRow the data row that is edited.
    	 * @param pColumnName the column name of the edited column.
    	 */
    	public CellEditorHandler(JVxDateCellEditor pCellEditor, ICellFormatterEditorListener pCellEditorListener, 
                				 IDataRow pDataRow, String pColumnName)
    	{
    		cellEditor = pCellEditor;
    		cellEditorListener = pCellEditorListener;
    		dataRow = pDataRow;
    		columnName = pColumnName;
    		
    		cellEditorComponent = new JVxDateCombo(cellEditor.getDateFormat());
    		if (cellEditorComponent.getEditorComponent() instanceof JTextField)
    		{
    			((JTextField)cellEditorComponent.getEditorComponent()).setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(cellEditor.getHorizontalAlignment()));
    		}

    		cellEditorComponent.getEditorComponent().getDocument().addDocumentListener(this);
    		cellEditorComponent.getEditorComponent().addKeyListener(this);
    		cellEditorComponent.getEditorComponent().addFocusListener(this);
    		cellEditorComponent.getEditorComponent().setFocusTraversalKeysEnabled(false);
    		cellEditorComponent.addPopupMenuListener(this);
    		
    		updateEditor();
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Interface implementation
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	// ICellEditorHandler
    	
    	/**
    	 * {@inheritDoc}
    	 */
    	public void uninstallEditor()
    	{
    		cellEditorComponent.getEditorComponent().getDocument().removeDocumentListener(this);
    		cellEditorComponent.getEditorComponent().removeKeyListener(this);
    		cellEditorComponent.getEditorComponent().removeFocusListener(this);
    		cellEditorComponent.removePopupMenuListener(this);
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void updateEditor()
    	{
    		cellEditorComponent.setTranslation(((IControl)cellEditorListener).getTranslation());
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditor getCellEditor()
    	{
    		return cellEditor;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditorListener getCellEditorListener()
    	{
    		return cellEditorListener;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public IDataRow getDataRow()
    	{
    		return dataRow;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public String getColumnName()
    	{
    		return columnName;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public JComponent getCellEditorComponent()
    	{
    		return cellEditorComponent;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void saveEditing() throws ModelException
    	{
    		ignoreEvent = true;
			cellEditorComponent.setSelectedItem(cellEditorComponent.getEditor().getItem());
    		ignoreEvent = false;
   			dataRow.setValue(columnName, cellEditorComponent.getSelectedItem());
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void cancelEditing() throws ModelException
    	{
    		if (!ignoreEvent)
    		{
	    		ignoreEvent = true;

	    		try
	    		{
	    			ColumnDefinition columnDef = dataRow.getRowDefinition().getColumnDefinition(columnName);
	    			
	    			cellEditorComponent.setSelectedItem(dataRow.getValue(columnName));

					CellFormat cellFormat = null;

					Container conParent = cellEditorComponent.getParent();
					
					boolean bParentEnabled = conParent == null || conParent.isEnabled();
					
					if (dataRow instanceof IDataBook)
		    		{
		    			IDataBook dataBook = (IDataBook)dataRow;
		    			cellEditorComponent.setEditorEditable(bParentEnabled 
		    					                              && dataBook.isUpdateEnabled() 
				        				                      && !columnDef.isReadOnly()
				        				                      && dataBook.getSelectedRow() >= 0);

		    			if (cellEditorListener.getCellFormatter() != null)
						{
							try
							{
								cellFormat = cellEditorListener.getCellFormatter().getCellFormat(
										dataBook, dataBook.getDataPage(), dataBook, columnName, dataBook.getSelectedRow(), -1);
							}
							catch (ModelException pModelException)
							{
								// Do nothing
							}
						}
		    		}
					else
					{
						cellEditorComponent.setEditorEditable(bParentEnabled && !columnDef.isReadOnly());
					}
					
					Color background;
					Color foreground;
					Font  font;
					if (cellFormat == null)
					{
						background = null;
						foreground = null;
						font = null;
					}
					else
					{
						background = cellFormat.getBackground();
						foreground = cellFormat.getForeground();
						font = cellFormat.getFont();
					}
					if (font == null)
					{
						font = ((Component)cellEditorListener).getFont();
					}
					if (foreground == null && ((Component)cellEditorListener).isForegroundSet())
					{
						foreground = ((Component)cellEditorListener).getForeground();
					}
					if (background == null && ((Component)cellEditorListener).isBackgroundSet())
					{
						background = ((Component)cellEditorListener).getBackground();
					}

					cellEditorComponent.getEditorComponent().setFont(font);
		    		if (cellEditorComponent.isEditorEditable())
		    		{
		    			if (background == null)
		    			{
			    			if (columnDef.isNullable())
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_BACKGROUND);
			    			}
			    			else
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_MANDATORY_BACKGROUND);
			    			}
		    			}
		    			if (cellEditorComponent.getEditorComponent().hasFocus())
						{
							cellEditorComponent.getEditorComponent().selectAll();
						}
						else
						{
							cellEditorComponent.getEditorComponent().select(0, 0);
						}
		    		}
		    		else
		    		{
		    			background = JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND);
						cellEditorComponent.getEditorComponent().select(0, 0);
		    		}
    				cellEditorComponent.getEditorComponent().setBackground(background);
    				cellEditorComponent.getEditorComponent().setForeground(foreground);
	    		}
	    		catch (Exception pException)
	    		{
	    			cellEditorComponent.setSelectedItem(null);
	    			cellEditorComponent.setEditorEditable(false);
	    			cellEditorComponent.getEditorComponent().setBackground(JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND));
	    			
	    			throw new ModelException("Editor cannot be restored!", pException);
	    		}
	    		finally
	    		{
	    			firstEditingStarted = true;
	    			ignoreEvent = false;
	    		}
    		}
    	}
    	
    	// DocumentListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
        public void insertUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       		}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void removeUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       		}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void changedUpdate(DocumentEvent pDocumentEvent) 
        {
        }

    	// KeyListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyPressed(KeyEvent pKeyEvent)
		{
			if (!pKeyEvent.isConsumed())
			{
				switch (pKeyEvent.getKeyCode())
				{
					case KeyEvent.VK_ESCAPE: 
						pKeyEvent.consume(); 
				        fireEditingComplete(ICellEditorListener.ESCAPE_KEY, true);
				        break;
					case KeyEvent.VK_ENTER: 
						pKeyEvent.consume();
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_ENTER_KEY, true);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.ENTER_KEY, true);
						}
				        break;
					case KeyEvent.VK_TAB: 
						pKeyEvent.consume(); 
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_TAB_KEY, true);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.TAB_KEY, true);
						}
				        break;
				    default:
				    	// Nothing to do
				}
			}
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyReleased(KeyEvent pKeyEvent)
		{
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyTyped(KeyEvent pKeyEvent)
		{
		}

    	// PopupMenuListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuCanceled(PopupMenuEvent pPopupMenuEvent)
		{
			fireEditingComplete(ICellEditorListener.ESCAPE_KEY, false);
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuWillBecomeInvisible(PopupMenuEvent pPopupMenuEvent)
		{
			if (!cellEditorComponent.isPopupCanceled())
			{
				fireEditingStarted();
				fireEditingComplete(ICellEditorListener.FOCUS_LOST, false);
			}
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuWillBecomeVisible(PopupMenuEvent pPopupMenuEvent)
		{
		}

    	// FocusListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void focusGained(FocusEvent pFocusEvent)
		{
			if (cellEditorComponent.isEditorEditable())
			{
				// Sets JFormattedTextFields edited true, this prevents setFormatter and therefore document changes on focusGained.
				cellEditorComponent.getEditorComponent().setText(cellEditorComponent.getEditorComponent().getText());
				
	    		cellEditorComponent.getEditorComponent().selectAll();
			}
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void focusLost(FocusEvent pFocusEvent)
		{
			if (!pFocusEvent.isTemporary() && !cellEditorComponent.isPopupFocusEvent(pFocusEvent))
			{
				fireEditingComplete(ICellEditorListener.FOCUS_LOST, true);
			}
		}
		
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// User-defined methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that the event occurs only one time.
		 */
		protected void fireEditingStarted()
		{
        	if (firstEditingStarted && !ignoreEvent && cellEditorListener != null)
        	{
           		firstEditingStarted = false;
           		cellEditorListener.editingStarted();
        	}
		}
		
		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that editing started will be called before,
		 * if it is not called until jet.
		 * 
		 * @param pCompleteType the editing complete type.
		 * @param pClosePopup try closing the popup.
		 */
		protected void fireEditingComplete(String pCompleteType, boolean pClosePopup)
		{
			if (!ignoreEvent && cellEditorListener != null)
			{
				if (pClosePopup && cellEditorComponent.isPopupVisible())
				{
					ignoreEvent = true;
					cellEditorComponent.setPopupCanceled(true);
					cellEditorComponent.setPopupVisible(false);
					ignoreEvent = false;
				}
				cellEditorListener.editingComplete(pCompleteType);
			}
		}
		
    }	// CellEditorHandler
	
}	// JVxDateCellEditor
