/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 04.10.2009 - [JR] - creation
 * 14.11.2009 - [JR] - #7
 *                     * validatePassword implemented
 *                     * IPasswordValidator implemented   
 * 06.06.2010 - [JR] - #132: password encryption support
 * 31.07.2011 - [JR] - #261: createSecurityManager() implemented
 *                   - #16: prepareException implemented   
 * 27.09.2011 - [JR] - add/remove hidden package names                                                                
 */
package com.sibvisions.rad.server.security;

import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;

import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.security.validation.IPasswordValidator;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.SecureHash;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.log.ILogger.LogLevel;

/**
 * The <code>AbstractSecurityManager</code> is the base class for {@link ISecurityManager} implementations
 * but it does not implement the security methods.
 * It supports security managers with important and usable methods.
 *  
 * @author Ren Jahn
 */
public abstract class AbstractSecurityManager implements ISecurityManager,
                                                         IPasswordValidator
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the list of additional hidden packages. */
	private static ArrayUtil<String> auHiddenPackages = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void checkPassword(ISession pSession, String pPassword)
	{
		if (pPassword == null || pPassword.trim().length() == 0)
		{
			throw new SecurityException("The new password is empty");
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Compares two passwords to be identical.
	 * 
	 * @param pConfig the application configuration
	 * @param pPassword base password (plain text)
	 * @param pConfirmPassword confirmation password (encrypted or plain text)
	 * @return <code>true</code> if the passwords are identical, otherwise <code>false</code>
	 * @throws Exception if the password encryption causes an error  
	 */
	protected boolean comparePassword(IConfiguration pConfig, String pPassword, String pConfirmPassword) throws Exception
	{
		if (pPassword == null && pConfirmPassword == null)
		{
			return true;
		}
		else if (pPassword == null || pConfirmPassword == null)
		{
			return false;
		}
		
		//NO ENCRYPTION:
		//
		//PWD  | CONFIRM
		//--------------
		//pass | pass (OK)
		//0101 | pass (ERROR)
		
		//ENCRYPTION:
		//
		//PWD  | CONFIRM
		//--------------
		//pass  | !0101 (OK -> default mode)
		//!0101 | !0101 (ERROR)
		//pass  | pass  (OK -> compatibility mode)
		//!0101 | pass  (ERROR)
		
		String sEncPwd = getEncryptedPassword(pConfig, pPassword);
		String sEncConfirm;

		if (isPasswordEncryptionEnabled(pConfig) && pConfirmPassword.charAt(0) != (char)127)
		{
			sEncConfirm = getEncryptedPassword(pConfig, pConfirmPassword);
		}
		else
		{
			sEncConfirm = pConfirmPassword;
		}
		
		return sEncPwd.equals(sEncConfirm);
	}
	
	/**
	 * Gets the password validator from an application configuration.
	 * 
	 * @param pConfig the application configuration
	 * @return the {@link IPasswordValidator} or <code>null</code> if no validator is specified
	 */
	protected IPasswordValidator getPasswordValidator(IConfiguration pConfig)
	{
		String sValidator = pConfig.getProperty("/application/securitymanager/passwordvalidator/class");
		
		if (sValidator != null && sValidator.trim().length() > 0)
		{
			try
			{
				return (IPasswordValidator)Class.forName(sValidator).newInstance();
			}
			catch (ClassNotFoundException cnfe)
			{
				throw new SecurityException("Password validator '" + sValidator + "' was not found!");
			}
			catch (InstantiationException ie)
			{
				throw new SecurityException("Can't instantiate password validator '" + sValidator + "'!");
			}
			catch (IllegalAccessException iae)
			{
				throw new SecurityException("Password validator '" + sValidator + "' not accessible!");
			}
		}	
		
		return null;
	}
	
	/**
	 * Validates a new password against an old password an uses a preconfigured password validator for checking
	 * the strength of the new password.
	 * 
	 * @param pSession the session which changes the password
	 * @param pOldPassword the old/current password
	 * @param pNewPassword the new password
	 * @throws Exception if the password validation failed, e.g. old = new, new is not strength enough, ...
	 */
	protected void validatePassword(ISession pSession, String pOldPassword, String pNewPassword) throws Exception
	{
		if (pOldPassword != null && pOldPassword.equals(pNewPassword))
		{
			throw new SecurityException("The old and new password are the same");
		}

		IPasswordValidator pwdval = getPasswordValidator(pSession.getConfig());
		
		if (pwdval != null)
		{
			pwdval.checkPassword(pSession, pNewPassword);
		}
		else
		{
			checkPassword(pSession, pNewPassword);
		}
	}
	
	/**
	 * Gets the password, encrypted with the algorithm specified in an application configuration.
	 * 
	 * @param pConfig the application configuration
	 * @param pPassword the plain text password
	 * @return the encrypted password 
	 * @throws Exception if the encryption fails
	 */
	public static String getEncryptedPassword(IConfiguration pConfig, String pPassword) throws Exception
	{
		String sAlgorithm = pConfig.getProperty("/application/securitymanager/passwordalgorithm");
		
		if (sAlgorithm != null && sAlgorithm.length() > 0)
		{
			if (!sAlgorithm.equalsIgnoreCase("PLAIN"))
			{
				//special char: 127 (DEL) is a valid XML char too (http://www.w3.org/TR/2000/REC-xml-20001006#NT-Char)
				return ((char)127) + SecureHash.getHash(sAlgorithm, pPassword.getBytes());
			}
		}
		
		return pPassword;
	}
	
	/**
	 * Checks if the password encryption is enabled. That means that the config parameter
	 * <code>/application/securitymanager/passwordalgorithm</code> contains an algorithm.
	 * PLAIN is not interpreted as algorithm.
	 * 
	 * @param pConfig the application configuration
	 * @return <code>true</code> if the password should be encrypted
	 */
	public static boolean isPasswordEncryptionEnabled(IConfiguration pConfig)
	{
		String sAlgorithm = pConfig.getProperty("/application/securitymanager/passwordalgorithm");
		
		return sAlgorithm != null && sAlgorithm.length() > 0 && !sAlgorithm.equalsIgnoreCase("PLAIN");
	}

	/**
	 * Creates a new {@link ISecurityManager} for the given session.
	 * 
	 * @param pSession the session
	 * @return the security manager for the application
	 * @throws Exception if the security manager is not set, the class was not found or the application is invalid
	 */
	public static ISecurityManager createSecurityManager(ISession pSession) throws Exception
	{
		String sSecManClass = pSession.getConfig().getProperty("/application/securitymanager/class");
		
		if (sSecManClass == null)
		{
			throw new SecurityException("Security manager is not set!");
		}
		
		return (ISecurityManager)Class.forName(sSecManClass).newInstance();
	}

	/**
	 * Creates a new {@link ISecurityManager} for the given application.
	 * 
	 * @param pApplicationName the name of the application
	 * @return the security manager for the application
	 * @throws Exception if the security manager is not set, the class was not found or the application is invalid
	 */
	public static ISecurityManager createSecurityManager(String pApplicationName) throws Exception
	{
		String sSecManClass = Configuration.getApplicationZone(pApplicationName).getProperty("/application/securitymanager/class");
		
		if (sSecManClass == null)
		{
			throw new SecurityException("Security manager is not set!");
		}
		
		return (ISecurityManager)Class.forName(sSecManClass).newInstance();
	}
	
	/**
	 * Hides the StackTraceElements of "com.sibvisions.rad.*" when the given exception is a
	 * {@link SecurityException}. If {@link LogLevel#DEBUG} is enabled, the stack won't be
	 * changed.
	 * 
	 * @param pException the occured exception
	 * @return the changed exception
	 */
	public static Throwable prepareException(Throwable pException)
	{
		return prepareException(pException, false);
	}
	
	/**
	 * Hides the StackTraceElements of "com.sibvisions.rad.*" when the given exception is a
	 * {@link SecurityException}. If {@link LogLevel#DEBUG} is enabled, the stack won't be
	 * changed, but it's possible to force changing.
	 * 
	 * @param pException the occured exception
	 * @param pForce force exception hiding
	 * @return the changed exception
	 */
	public static Throwable prepareException(Throwable pException, boolean pForce)
	{
		if (pException instanceof SecurityException)
		{
			if (pForce || !LoggerFactory.getInstance(AbstractSecurityManager.class).isEnabled(LogLevel.DEBUG))
			{
				ArrayUtil<StackTraceElement> auStack = new ArrayUtil<StackTraceElement>();
				
				StackTraceElement[] stack = pException.getStackTrace();
				
				if (stack != null)
				{
					for (int i = 0; i < stack.length; i++)
					{
						if (!stack[i].getClassName().startsWith("com.sibvisions.rad")
							&& !isHiddenPackage(stack[i].getClassName()))
						{
							auStack.add(stack[i]);
						}
					}
				}

				stack = auStack.toArray(new StackTraceElement[auStack.size()]);
				
				pException.setStackTrace(stack);
			}
		}
		
		return pException;
	}
	
	/**
	 * Adds a package name to the hidden package list.
	 * 
	 * @param pPackage the full qualified java package name e.g. com.sibvisions
	 */
	public static void addHiddenPackage(String pPackage)
	{
		if (pPackage != null)
		{
			if (auHiddenPackages == null)
			{
				auHiddenPackages = new ArrayUtil<String>();
			}
			
			auHiddenPackages.add(pPackage);
		}
	}
	
	/**
	 * Removes a package name from the hidden package list.
	 * 
	 * @param pPackage the full qualified java package naem e.g. com.sibvisions
	 */
	public static void removeHiddenPackage(String pPackage)
	{
		if (pPackage != null && auHiddenPackages != null)
		{
			auHiddenPackages.remove(pPackage);
			
			if (auHiddenPackages.size() == 0)
			{
				auHiddenPackages = null;
			}
		}
	}
	
	/**
	 * Checks if a class or package name is excluded through the hidden package list.
	 * 
	 * @param pJavaName the full qualified java class or package name e.g. com.sibvisions.rad.IPackageSetup
	 * @return <code>true</code> if the name contains a hidden package name
	 */
	public static boolean isHiddenPackage(String pJavaName)
	{
		if (pJavaName == null || auHiddenPackages == null)
		{
			return false;
		}
		
		for (int i = 0, anz = auHiddenPackages.size(); i < anz; i++)
		{
			if (pJavaName.startsWith(auHiddenPackages.get(i)))
			{
				return true;
			}
		}
		
		return false;
	}
	
}	// AbstractSecurityManager
