/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 12.05.2009 - [RH] - creation.
 * 23.11.2009 - [RH] - ColumnMetaData && PrimaryKey Column is replaced with MetaData class
 * 02.03.2010 - [RH] - reorganized MetaData -> ServerMetaData, ColumnMetaData -> ServerColumnMetaData
 * 10.03.2010 - [JR] - set NLS_COMP='BINARY'
 * 27.03.2010 - [JR] - #92: default value support    
 * 28.03.2010 - [JR] - #47: getAllowedValues implemented
 * 06.04.2010 - [JR] - #115: getUKs: prepared statement closed  
 * 06.05.2010 - [JR] - open: close statement(s)     
 * 09.10.2010 - [JR] - #114: used CheckConstraintSupport to detect allowed values       
 * 19.11.2010 - [RH] - getUKs, getPKs return Type changed to a <code>Key</code> based result.         
 * 29.11.2010 - [RH] - getUKs Oracle select statement fixed, that it only returns UKs - no PKs. 
 * 01.12.2010 - [RH] - getFKs Oracle select statement fixed, that also returns FKs over UKs.        
 * 14.12.2010 - [RH] - getUKS is solved in DBAccess.
 * 23.12.2010 - [RH] - #227: getFKs returned PK <-> FK columns wrong related, wrong sort fixed!
 * 28.12.2010 - [RH] - #230: quoting of all DB objects like columns, tables, views. 
 * 03.01.2011 - [RH] - schema detecting made better in getColumnMetaData()
 * 06.01.2011 - [JR] - #234: used ColumnMetaDataInfo
 * 24.02.2011 - [RH] - #295 - just return the PK, UKs and FKs for the table && schema.
 * 11.03.2011 - [RH] - #308 - DB specific automatic quoting implemented          
 * 19.07.2011 - [RH] - #432 - OracleDBAccess return list of UKs wrong.   
 * 21.07.2011 - [RH] - #436 - OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert                      
 * 18.11.2011 - [RH] - #510: All XXDBAccess should provide a SQLException format method 
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Hashtable;
import java.util.List;

import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.ColumnMetaData;
import javax.rad.persist.DataSourceException;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>OracleDBAccess</code> is the implementation for Oracle databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class OracleDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The base select statement to get the Unique keys in Oracle. */
	private static String sUKSelectBase = "SELECT cons.owner table_schem, cols.table_name, cons.constraint_name, cols.column_name " +
									      "FROM all_constraints cons, all_cons_columns cols " +
									      "WHERE cols.table_name = ? " + 
	                                      "AND cons.constraint_type = 'U' " +
	                                      "AND cons.constraint_name = cols.constraint_name " + 
	                                      "AND cons.owner = cols.owner ";
			
	/** The select statement with schema to get the Unique keys in Oracle. */
    private static String sUKSelectWithSchema = sUKSelectBase + "AND cons.owner = ? " + "ORDER BY cols.table_name, cols.constraint_name, cols.position";
	
	/** the select statement to get the Check constraints in Oracle. */
	private static String sCheckSelect = "select search_condition " +
	                                       "from user_constraints " +
	                                      "where constraint_type = 'C' " +
	                                        "and generated = 'USER NAME' " +
	                                        "and status = 'ENABLED' " +
	                                        "and table_name = ?";
	
	/** The base select statement to get the Foreign keys in Oracle. */
	private static String sFKSelectBase = "SELECT c.owner           fktable_schem " +
			                                    ",c.constraint_name fk_name " +
			                                    ",c.table_name      fktable_name " +
			                                    ",cc.column_name    fkcolumn_name " +
			                                    ",r_owner           pktable_schem " +
			                                    ",r_constraint_name pk_name " +
			                                    ",ccr.table_name    pktable_name " +
			                                    ",ccr.column_name   pkcolumn_name " +
			                                "FROM all_constraints c " +
			                                    ",all_cons_columns cc " +
			                                    ",all_cons_columns ccr " +
			                               "WHERE cc.owner = c.owner " +
			                                 "AND cc.constraint_name = c.constraint_name " +
			                                 "AND ccr.owner = c.r_owner " +
			                                 "AND ccr.constraint_name = c.r_constraint_name " +
			                                 "AND ccr.position = cc.position " +
			                                 "AND c.constraint_type = 'R' " +
			                                 "AND c.table_name = ? ";

	/** The select statement to get the Foreign keys in Oracle. */
	private static String sFKSelectOrder = "ORDER BY c.owner, c.table_name, c.constraint_name, cc.position";

	/** The select statement with schema to get the Foreign keys in Oracle. */
    private static String sFKSelectWithSchema = sFKSelectBase + " AND c.owner = ? " + sFKSelectOrder;
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new OracleDBAccess Object.
	 */
	public OracleDBAccess()
	{
		super();		
		
		setDriver("oracle.jdbc.OracleDriver");	
	}
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatabaseSpecificLockStatement(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{
		return super.getDatabaseSpecificLockStatement(pWritebackTable, pServerMetaData, pPKFilter) + " NOWAIT";											
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] insertDatabaseSpecific(String pWritenbackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
                                           Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
    {
		return insertOracle(pWritenbackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws DataSourceException
	{
		super.open();

		Statement stmt = null;
		
		try
		{
			stmt = getConnection().createStatement();
			
			stmt.executeUpdate("ALTER SESSION SET NLS_COMP='BINARY'");			
			stmt.executeUpdate("ALTER SESSION SET NLS_SORT='BINARY'");
		}
		catch (SQLException ex)
		{
			// Try silent to change nls_sort, nls_comp
		}
		finally
		{
			if (stmt != null)
			{
				try
				{
					stmt.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<Key> getUKs(String pCatalog, 
					  			 String pSchema, 
					  			 String pTable) throws DataSourceException
	{
		ResultSet rsResultSet = null;
		try
		{	
			ArrayUtil<Key>  auResult           = new ArrayUtil<Key>();
			ArrayUtil<Name> auUniqueKeyColumns = new ArrayUtil<Name>();
			
			long lMillis = System.currentTimeMillis();
			
			PreparedStatement psResult = getPreparedStatement(sUKSelectWithSchema, false);
			
			try
			{
				psResult.setString(1, pTable);
				psResult.setString(2, pSchema);
				rsResultSet = psResult.executeQuery();
				if (!rsResultSet.next())
				{
					rsResultSet.close();
					psResult.close();
					
					return auResult;
				}
				
				String sUKName = null;
				do
				{
					if (rsResultSet.getString("COLUMN_NAME") != null)
					{
						if (sUKName != null && !rsResultSet.getString("CONSTRAINT_NAME").equals(sUKName))
						{
							
							Key uk = new Key(sUKName);
							uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
							auResult.add(uk);
							auUniqueKeyColumns.clear();
						}
						sUKName = rsResultSet.getString("CONSTRAINT_NAME");
						
						auUniqueKeyColumns.add(new Name(rsResultSet.getString("COLUMN_NAME"), quote(rsResultSet.getString("COLUMN_NAME"))));
					}
				}
				while (rsResultSet.next());

				if (auUniqueKeyColumns.size() > 0)
				{
					Key uk = new Key(sUKName);
					uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
					auResult.add(uk);					
				}
				
				logger.debug("getUKs(", pTable, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");							
				
				return auResult;
			}
			finally
			{
				if (psResult != null)
				{
					try
					{
						psResult.close();
					}
					catch (Exception e)
					{
						//nothing to be done
					}
				}
			}
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Unique Keys couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}		
		finally
		{
			if (rsResultSet != null)
			{
				try
				{
					rsResultSet.close();
				}
				catch (SQLException e)
				{
					//nothing to be done
				}						
			}
		}			
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ForeignKey> getFKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		PreparedStatement psFKs = null;
		ResultSet         rsFKs = null;

		try
		{
			ArrayUtil<ForeignKey> auForeignKeys = new ArrayUtil<ForeignKey>();

			long lMillis = System.currentTimeMillis();

			String sCatalog = getConnection().getCatalog();
			
			psFKs = getPreparedStatement(sFKSelectWithSchema, false);
			psFKs.setString(1, pTable);
			psFKs.setString(2, pSchema);

			rsFKs = psFKs.executeQuery();			
			if (!rsFKs.next())
			{
				rsFKs.close();

				// #295 - just return the FKs for the table && schema.
				
				// find no FKs -> From Clause can't determined
				return auForeignKeys;
			}

			String sTempKeyName = null;
			String sKeyName = null;
			ArrayUtil<Name> auPKColumns = new ArrayUtil<Name>();
			ArrayUtil<Name> auFKColumns = new ArrayUtil<Name>();

			ForeignKey fkForeignKey = new ForeignKey(
					new Name(rsFKs.getString("PKTABLE_NAME"), quote(rsFKs.getString("PKTABLE_NAME"))), 
					new Name(sCatalog, quote(sCatalog)), 
					new Name(rsFKs.getString("PKTABLE_SCHEM"), quote(rsFKs.getString("PKTABLE_SCHEM"))));

			do
			{
				sTempKeyName = rsFKs.getString("FK_NAME");
				if (sTempKeyName == null || sTempKeyName.length() == 0)
				{
					sTempKeyName = rsFKs.getString("PK_NAME");
					if (sTempKeyName == null || sTempKeyName.length() == 0)
					{
						throw new DataSourceException("Database/jdbc driver didn't support FK, PK names!");
					}
				}
				if (sKeyName != null && !sKeyName.equals(sTempKeyName))
				{
					fkForeignKey.setFKColumns(auFKColumns.toArray(new Name[auFKColumns.size()]));
					fkForeignKey.setPKColumns(auPKColumns.toArray(new Name[auPKColumns.size()]));
					auForeignKeys.add(fkForeignKey);

					auPKColumns.clear();
					auFKColumns.clear();

					fkForeignKey = new ForeignKey(
							new Name(rsFKs.getString("PKTABLE_NAME"), quote(rsFKs.getString("PKTABLE_NAME"))), 
							new Name(sCatalog, quote(sCatalog)), 
							new Name(rsFKs.getString("PKTABLE_SCHEM"), quote(rsFKs.getString("PKTABLE_SCHEM"))));
				}
				sKeyName = sTempKeyName;

				auPKColumns.add(new Name(rsFKs.getString("PKCOLUMN_NAME"), quote(rsFKs.getString("PKCOLUMN_NAME"))));
				auFKColumns.add(new Name(rsFKs.getString("FKCOLUMN_NAME"), quote(rsFKs.getString("FKCOLUMN_NAME"))));

				fkForeignKey.setFKName(rsFKs.getString("FK_NAME"));
			}
			while (rsFKs.next());

			fkForeignKey.setFKColumns(auFKColumns.toArray(new Name[auFKColumns.size()]));
			fkForeignKey.setPKColumns(auPKColumns.toArray(new Name[auPKColumns.size()]));
			auForeignKeys.add(fkForeignKey);

			logger.debug("getFKs(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");

			return auForeignKeys;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Foreign Keys couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}
		finally
		{
			try
			{
				if (psFKs != null)
				{
					psFKs.close();
				}
				if (rsFKs != null)
				{
					rsFKs.close();
				}
			}
			catch (SQLException e)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return super.getDefaultValues(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		//Oracle JDBC returns 'value'\n
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Hashtable<String, Object[]> getAllowedValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		PreparedStatement psCheck = null;
		
		ResultSet resCheck = null;
		
		Hashtable<String, Object[]> htAllowed;
		
		Hashtable<String, List<String>> htFoundValues = null;
		
		
		try
		{
			psCheck = getPreparedStatement(sCheckSelect, false);
			psCheck.setString(1, pTable);
			
			resCheck = psCheck.executeQuery();
			
			//detect all possible values
			
			while (resCheck.next())
			{
				htFoundValues = CheckConstraintSupport.parseCondition(resCheck.getString(1), htFoundValues, true);
			}			
			
			//interpret values
			
			htAllowed = CheckConstraintSupport.translateValues(this, pCatalog, pSchema, pTable, htFoundValues);
		}
		catch (SQLException sqle)
		{
			throw new DataSourceException("Can't access check constraints for: '" + pTable + "'", formatSQLException(sqle));
		}
		finally
		{
			if (resCheck != null)
			{
				try
				{
					resCheck.close();
				}
				catch (SQLException e)
				{
					//nothing to be done
				}
			}
			
			if (psCheck != null)
			{
				try
				{
					psCheck.close();
				}
				catch (SQLException e)
				{
					//nothing to be done
				}
			}
		}
		
		return htAllowed;
	}
  
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the newly inserted row from an Oracle Database. <br>
	 * It uses RETURNING .. INTO to get the primary key values back from the database. 
	 * 
	 * @param pTablename		the table to use for the insert
	 * @param pInsertStatement	the SQL Statement to use for the insert
	 * @param pServerMetaData	the meta data to use.
	 * @param pNewDataRow		the new IDataRow with the values to insert
	 * @param pDummyColumn		true, if all writeable columns are null, but for a correct INSERT it have
	 *                          to be minimum one column to use in the syntax.
	 * @return the newly inserted row from an Oracle Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert the <code>IDataRow</code> 
	 *             to the storage
	 */	
	private Object[] insertOracle(String pTablename, String pInsertStatement, ServerMetaData pServerMetaData, 
			                      Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
	{
		StringBuffer sInsertStatement = new StringBuffer("BEGIN " + pInsertStatement);
		
		// use RETURNING to get all PK column values filled in in from the trigger
		sInsertStatement.append(" RETURNING ");
		
		String[] pPKColumns = pServerMetaData.getPrimaryKeyQuotedColumnNames();					
		for (int i = 0; pPKColumns != null && i < pPKColumns.length; i++)
		{
			if (i > 0)
			{
				sInsertStatement.append(", ");
			}
			
			sInsertStatement.append(pPKColumns[i]);
		}
		sInsertStatement.append(" INTO ");
		
		for (int i = 0; pPKColumns != null && i < pPKColumns.length; i++)
		{
			if (i > 0)
			{
				sInsertStatement.append(", ");
			}
			sInsertStatement.append("?");
		}
	
		sInsertStatement.append("; END;");

		CallableStatement csInsert = null;
		try
		{
			try
			{				
				// #436 - OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert
				String sSQL = translateQuotes(sInsertStatement.toString());
				logger.debug("executeSQL->" + sSQL);
				csInsert = getConnection().prepareCall(sSQL);
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("Insert failed! - " + sInsertStatement, formatSQLException(sqlException));
			}						
		
			ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
			int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
			
			int iLastIndex = 0;
			if (pDummyColumn == null)
			{
				iLastIndex = setColumnsToStore(csInsert, cmdServerColumnMetaData, iaWriteables, pNewDataRow, null);
			}
			else
			{
				try
				{
					for (int i = 0; i < cmdServerColumnMetaData.length; i++)
					{
						if (cmdServerColumnMetaData[i].getColumnName().getQuotedName().equals(pDummyColumn))
						{
							csInsert.setObject(1, null, cmdServerColumnMetaData[i].getSQLType());
							break;
						}
					}					
					iLastIndex = 1;
				}
				catch (SQLException sqlException)
				{			
					throw new DataSourceException("Insert failed! - " + sInsertStatement, formatSQLException(sqlException));
				}				
			}
			// use RETURNING to get the PK column values filled in by the trigger
			// set the out parameters
			int[] iaPKColumns = pServerMetaData.getPrimaryKeyColumnIndices();
			
			for (int i = 0; iaPKColumns != null && i < iaPKColumns.length; i++)
			{
				try
				{	
					int iSQLType;

					IDataType dtDataType = ColumnMetaData.createDataType(cmdServerColumnMetaData[iaPKColumns[i]].getColumnMetaData());
					if (dtDataType instanceof BigDecimalDataType)
					{
						iSQLType = java.sql.Types.NUMERIC;
					}
					else if (dtDataType instanceof TimestampDataType)
					{
						iSQLType = java.sql.Types.TIMESTAMP;
					}
					else
					{
						iSQLType = java.sql.Types.CHAR;
					}
										
					csInsert.registerOutParameter(iLastIndex + i + 1, iSQLType);
				}
				catch (SQLException sqlException)
				{			
					throw new DataSourceException("Insert failed! - " + sInsertStatement, formatSQLException(sqlException));
				}				
			}
		
			if (executeUpdate(csInsert) == 1)
			{
				// use RETURNING to get the PK column values filled in by the trigger
				// get the out parameters, and set the PK columns
				
				for (int i = 0; iaPKColumns != null && i < iaPKColumns.length; i++)
				{
					try
					{
						pNewDataRow[iaPKColumns[i]] = csInsert.getObject(iLastIndex + i + 1);
					}
					catch (SQLException sqlException)
					{
						throw new DataSourceException("Insert failed! - " + sInsertStatement, formatSQLException(sqlException));
					}				
				}
				return pNewDataRow;
			}
			throw new DataSourceException("Insert failed ! - Result row count != 1 ! - " +  sInsertStatement);
		}
		finally
		{
			try
			{
				if (csInsert != null)
				{
					csInsert.close();
				}
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(sqlException));
			}			
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
		 							     	    String[] pQueryColumns,
										     	String pBeforeQueryColumns, 
										     	String pWhereClause, 
										     	String pAfterWhereClause,
										     	String pWritebackTable,
										     	String[] pWritebackColumns) throws DataSourceException
    {
		ColumnMetaDataInfo mdInfo = super.getColumnMetaData(pFromClause, pQueryColumns, pBeforeQueryColumns, pWhereClause, 
				                                      	    pAfterWhereClause, pWritebackTable, pWritebackColumns);
		
		// schema is null, so try to set the schema.
		if (mdInfo.getSchema() == null)
		{
			mdInfo.setSchema(getUsername().toUpperCase());
		}
		String table = mdInfo.getTable();
		if (table != null && !table.startsWith(QUOTE) && !table.endsWith(QUOTE))
		{
			mdInfo.setTable(table.toUpperCase());
		}
		
		return mdInfo;
    }
	
} 	// OracleDBAccess
