/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation.
 * 02.11.2008 - [RH] - extends now MemDataSource 
 * 12.11.2008 - [RH] - getPrimaryKey bug with Oracle fixed. (Databases without Catalogs)
 * 17.11.2008 - [RH] - setFilterParameters changes to allow null Parameters
 * 19.11.2008 - [RH] - filter redesign
 * 23.11.2008 - [RH] - Statement cache disabled, Oracle Insert with RETURNING added
 * 26.11.2008 - [RH] - lockAndRefetch added()
 * 27.11.2008 - [RH] - throw Exceptions optimized
 * 30.01.2009 - [RH] - fetch bug with SelectColumns fixed
 * 13.05.2009 - [RH] - reviewed, separated into one DBAcces for each different Database
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 14.07.2009 - [JR] - getSelectStatement: missing from clause exception throwed
 *                     insert, update, delete: missing writebackTable exception throwed
 *                     getDatabaseSpecificLockStatement: missing writebackTable exception throwed
 * 06.10.2009 - [RH] - supportsGetGeneratedKeys add, used in DerbyDBAccess to set it to true, because of the derby bug
 * 14.10.2009 - [RH] - sql to java type conversion added (->serializeable!)
 *                     bug fix CLOB, -> CLOB is interpreted as a StringType instead of BinaryTyp.
 *                     CLOB, BLOB size is default Integer.MAX_VALUE
 * 15.10.2009 - [JR] - fetch: pMinimumRowCount < 0                     
 * 23.10.2009 - [HM] - Properties changes to get/set username, url, driver, password                      
 * 23.10.2009 - [HM] - static getDBAccess added
 *            - [JR] - getDBAccess: checked null  
 * 27.10.2009 - [RH] - setNull, for empty insert/update [BUGFIX]
 * 23.11.2009 - [RH] - use of MetaData instead of MetaDataColumn[] and PK column String[].
 *                     code optimization      
 * 04.12.2009 - [RH] - QueryColumns with alias bug fixed            
 *                     getUKs, getPK return null if none is found. Exceptions will be logged!
 * 09.12.2009 - [JR] - set/getDBProperty implemented
 *                   - open, toString: hide username password form the property output
 * 02.03.2010 - [RH] - reorganized MetaData -> ServerMetaData, ColumnMetaData -> ServerColumnMetaData
 * 08.03.2010 - [RH] - if in update() is determined, that no values are changed, then it returns instead of null, 
 *                     the old value.
 * 10.03.2010 - [RH] - Exception didn't thrown with duplicate column names in getColumnMetaData()
 *                     Ticket #81 fixed
 * 13.03.2010 - [JR] - #90: fetch: handled different Number instances as BigDecimal          
 * 19.03.2010 - [RH] - if in update() no rows changed, then try to find that row, if that doesn't exist, then insert it.
 * 25.03.2010 - [JR] - #92: getDefaultValues, translateDefaultValue implemented            
 * 28.03.2010 - [JR] - #47: getAllowedValues implemented  
 * 30.03.2010 - [RH] - updateAnsiSQL() + updateDatabaseSpecific add for DB specific support.
 * 06.04.2010 - [JR] - #115: closed all ResultSets
 * 21.04.2010 - [JR] - createDataType replaced with (ServerColumnMetaData).getDataType()   
 * 18.06.2010 - [JR] - fetch: optimized type detection and reduced method calls in for loop        
 * 11.10.2010 - [RH] - #91: the automatic link column name is now build independent (foregnkey column name) and with an optional TranslationMap.
 * 20.10.2010 - [JR] - #188: UK detection returns an empty row if no UK was found     
 * 25.10.2010 - [JR] - #196: setConnection implemented and used in open/close
 *                   - #195: getDBAccess(java.sql.Connection) implemented
 * 19.11.2010 - [RH] - getUKs, getPKs return Type changed to a <code>Key</code> based result.     
 * 22.11.2010 - [RH] - executeSQL added.          
 * 23.11.2010 - [RH] - setSQLTypeName on ServerColumnMetaData used.   
 * 01.12.2010 - [RH] - getPKs set PKName corrected.
 * 14.12.2010 - [RH] - #222: getUKs return no PK anymore, just UKs
 * 15.12.2010 - [RH] - better logging in executeXXX methods.
 * 23.12.2010 - [RH] - #224: wrong AutomaticLinkColumnName fixed (if PK with one column, and FK with more the one column used, + all _ID are remove)
 *                                                                 duplicate use of ForeignKeys over partly the same columns fixed.   
 * 28.12.2010 - [RH] - #230: quoting of all DB objects like columns, tables, views. 
 * 03.01.2011 - [RH] - Schema detecting made better in getColumnMetaData()
 *                   - #232: get/setQuoting() on/off , 
 *                   - #136: Mysql PK refetch not working -> extract the plan table without schema.
 * 06.01.2011 - [JR] - #234: introduced ColumnMetaDataInfo 
 * 29.01.2011 - [JR] - #211: getDefaultAllowedValues implemented
 *                   - translateValue: check null explicitly
 * 16.02.2011 - [JR] - getQueryColumns: code review      
 * 11.03.2011 - [RH] - #308: DB specific automatic quoting implemented       
 * 25.03.2011 - [RH] - removeDBSpecificquotes added.   
 * 28.04.2011 - [RH] - #341:  LikeReverse Condition, LikeReverseIgnoreCase Condition   
 *                     createReplace for DB added. Standard implementation is ANSI SQL.
 * 06.06.2011 - [JR] - #381: fixed column name usage      
 * 12.07.2011 - [RH] - #420: DBStorage creates too long alias names for oracle: ORA-00972: Bezeichner ist zu lang      
 * 22.07.2011 - [RH] - #440: DBAccess.insert return null if no auto increment column in table
 * 31.07.2011 - [RH] - #447: if getMaxColumnNameLength() return 0, we should interprete it as unlimmited
 * 14.09.2011 - [JR] - #470: splitSchemaTable, set/getDefaultSchema implemented
 * 18.11.2011 - [RH] - #510: All XXDBAccess should provide a SQLException format method 
 * 22.11.2011 - [JR] - #515: open now checks the transaction isolation level
 * 14.12.2011 - [JR] - insert/update logging
 * 17.12.2011 - [JR] - #528: createStorage implemented
 * 30.01.2012 - [RH] - #544 - In databases with default lower case column names, the default labels in the RowDefinition are wrong  -> setLabel removed!
 * 05.03.2012 - [HM] - #552:  Reverted createStorage, added functionality in DBAccess
 * 13.03.2012 - [RH] - #562: DBAccess getColumnMetaData() should determine writeable columns case insensitive
 * 08.05.2012 - [JR] - #575: convert value(s) to database specific value(s) 
 * 20.11.2012 - [JR] - #589: use JNDI for connection/dbaccess detection
 *                   - open now checks isOpen
 */
package com.sibvisions.rad.persist.jdbc;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.sql.Types;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.naming.InitialContext;
import javax.rad.model.ModelException;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.And;
import javax.rad.model.condition.CompareCondition;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.Filter;
import javax.rad.model.condition.Greater;
import javax.rad.model.condition.GreaterEquals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.condition.Less;
import javax.rad.model.condition.LessEquals;
import javax.rad.model.condition.Like;
import javax.rad.model.condition.LikeIgnoreCase;
import javax.rad.model.condition.LikeReverse;
import javax.rad.model.condition.LikeReverseIgnoreCase;
import javax.rad.model.condition.Not;
import javax.rad.model.condition.OperatorCondition;
import javax.rad.model.condition.Or;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.BinaryDataType;
import javax.rad.model.datatype.BooleanDataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.DataSourceException;
import javax.rad.util.TranslationMap;

import com.sibvisions.rad.persist.jdbc.param.AbstractParam;
import com.sibvisions.rad.persist.jdbc.param.AbstractParam.ParameterType;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.ILogger.LogLevel;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.BeanUtil;
import com.sibvisions.util.type.DateUtil;
import com.sibvisions.util.type.StringUtil;
import com.sibvisions.util.type.StringUtil.CaseSensitiveType;

/**
 * The <code>DBAccess</code> is the implementation for most used SQL databases.<br>
 * Standard ANSI SQL Databases are anyways supported.<br>
 * Its used to read/write data between the storage and the <code>DataBook/DataPage</code>.
 * It has database type specific implementations.<br>
 * 
 * <br><br>Example:
 * <pre>
 * <code>
 * DBAccess dba = new DBAccess();
 * 
 * // set connect properties
 * dba.setDriver("org.hsqldb.jdbcDriver");
 * dba.setUrl("jdbc:hsqldb:file:testdbs/test/testdb");
 * dba.setUsername("sa");
 * dba.setPassword("");
 * 
 * Properties pDBProperties = new Properties();
 * pDBProperties.put("shutdown", true);
 * dba.setDBProperties(pDBProperties);
 * 
 * // open
 * dba.open();
 * 
 * To get Database independent DBAccess, it is better to use:
 * 
 * DBAccess dba = DBAccess.getDBAccess("jdbc:hsqldb:file:testdbs/test/testdb");
 * 
 * // insert data into test table
 * PreparedStatement psPreparedStatement = dba.getPreparedStatement(
 * 	             "insert into test (id, name) values(?,?)", false);
 * psPreparedStatement.setInt(1, 1);
 * psPreparedStatement.setString(2, "projectX");
 * dba.executeUpdate(psPreparedStatement);
 * 
 * // select data from test table
 * psPreparedStatement = dba.getPreparedStatement("select * from test", false);
 * ResultSet rs = dba.executeQuery(psPreparedStatement);
 * 
 * while (rs.next())
 * {
 *    System.out.println(rs.getInt("id") + "," + rs.getString("name"));
 * }
 * </code>
 * </pre>
 *  
 * @see com.sibvisions.rad.persist.jdbc.IDBAccess
 * @see com.sibvisions.rad.model.remote.RemoteDataBook
 * 
 * @author Roland Hrmann
 */
public class DBAccess implements IDBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	// jdk 1.6 jdbc types
    /** A jdbc VARCHAR DB column data type constant. */
    public static final int NCHAR 			= -15;
    
    /** A jdbc VARCHAR DB column data type constant. */
    public static final int NVARCHAR 		= -9;
        
    /** A jdbc VARCHAR DB column data type constant. */
    public static final int LONGNVARCHAR 	= -16;
    
    /** A jdbc XML DB column data type constant. */
    public static final int SQLXML 			= 2009;

    /** A jdbc BLOB DB column data type constant. */ 
    public static final int NCLOB 			= 2011;
    
	/** JVx generell DB quote character. will be translated in the DB specific. */
    public static final String QUOTE = "`";
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// static 
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /** The Default translation Map for the automatic link column name for the specified ForeignKey and FK Column in the master table.*/
    private static TranslationMap tmpAutoLinkColumnNames  = new TranslationMap(); 
    
	/** Stores suitable DBAccess classes for jdbc urls. */
    private static Hashtable<String, Class<? extends DBAccess>> dbAccessClasses = new Hashtable<String, Class<? extends DBAccess>>();
    
    /** The logger for protocol the performance. */
	protected static ILogger logger = LoggerFactory.getInstance(DBAccess.class);

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The maximum time in milliseconds to use, to try to fetch all rows. reduce open cursors, and increase performance. */
	private int 									iMaxTime = 100;
	
	/** The jdbc connection to the database. */
	private Connection								cConnection;

	/** The jdbc connection string. */
	private String									sUrl;

	/** Name of the jdbc driver <code>Class</code>. */
	private String									sDriver;

	/** User name to connect to the database. */
	private String									sUsername;

	/** Password to connect to the database. */
	private String									sPassword;

    /** The open quote character for database operations. */
    private String 									sOpenQuote;

    /** The close quote character for database operations. */
    private String 									sCloseQuote;
    
    /** Sets the default scheman for the case that schema is not automatically detected. */
    private String									sDefaultSchema;
	
    /** Database specific properties. */
	private Properties								properties = new Properties();

	/** Query time out, which is used as limit for fetching data. */
	private int 									iQueryTimeOut = 0;

	/** Cache of <code>ResultSet</code>'s to reuse in the next fetch. */
	private Hashtable<Select, ResultSet>			htFetchResultSetCache = new Hashtable<Select, ResultSet>();
	
	/** the date util for string to date conversion. */
	private DateUtil 								dateUtil = new DateUtil(new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"));
	
	/** whether the connection is an external connection. */
	private boolean									bExternalConnection = false;
	
	/** stores the max. column length in this database. */
	private int 									iMaxColumnLength;
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	static
	{
		registerDBAccessClass("jdbc:oracle:", OracleDBAccess.class);
		registerDBAccessClass("jdbc:db2:", DB2DBAccess.class);
		registerDBAccessClass("jdbc:derby:", DerbyDBAccess.class);
		registerDBAccessClass("jdbc:jtds:sqlserver:", MSSQLDBAccess.class);
		registerDBAccessClass("jdbc:mysql:", MySQLDBAccess.class);
		registerDBAccessClass("jdbc:postgresql:", PostgreSQLDBAccess.class);
		registerDBAccessClass("jdbc:hsqldb:", HSQLDBAccess.class);
		
		tmpAutoLinkColumnNames.put("*_id*", "*0*1");
		tmpAutoLinkColumnNames.put("*_ID*", "*0*1");
	}
	
	/**
	 * Constructs a new DBAccess Object.
	 */
	public DBAccess()
	{
		super();
		
		setQuoteCharacters("\"", "\"");
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the suitable DBAcces for the given JdbcUrl.
	 *  
	 * @param pJdbcUrl the JdbcUrl.
	 * @return the suitable DBAccess.
	 * @throws DataSourceException if the instance name is unknown or the DBAccess object cannot be created.
	 */
	public static DBAccess getDBAccess(String pJdbcUrl) throws DataSourceException
	{
		if (pJdbcUrl == null)
		{
			return null;
		}
		
		//#589 check JNDI if the URL is not a jdbc url
		if (!pJdbcUrl.toLowerCase().startsWith("jdbc:"))
		{
			try
			{
				InitialContext cxt = new InitialContext();
				
				if (cxt != null) 
				{
					Object objInstance = cxt.lookup(pJdbcUrl);
					
					if (objInstance instanceof DBAccess)
					{
						return (DBAccess)objInstance;
					}
					else if (objInstance instanceof Connection)
					{
						return DBAccess.getDBAccess((Connection)objInstance);
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
				
				logger.debug("Connection is not configured as JNDI resource", ex);
			}
		}
		
		Class<? extends DBAccess> clazz = getDBAccessClass(pJdbcUrl);
		
		try
		{
			DBAccess dbAccess = clazz.newInstance();
			dbAccess.setUrl(pJdbcUrl);
			
			return dbAccess;
		}
		catch (Exception e)
		{
			throw new DataSourceException("Instantion of '" + clazz + "' failed!", e);
		}
	}
	
	/**
	 * Gets the suitable DBAccess for the given {@link DBCredentials}.
	 * 
	 * @param pCredentials the database credentials.
	 * @return the suitable DBAccess.
	 * @throws DataSourceException if the instance name is unknown or the DBAccess object cannot be created.
	 */
	public static DBAccess getDBAccess(DBCredentials pCredentials) throws DataSourceException
	{
		if (pCredentials == null)
		{
			return null;
		}
		
		Class<? extends DBAccess> clazz = getDBAccessClass(pCredentials.getUrl());
		
		try
		{
			DBAccess dbAccess = clazz.newInstance();
			dbAccess.setUrl(pCredentials.getUrl());
			dbAccess.setUsername(pCredentials.getUserName());
			dbAccess.setPassword(pCredentials.getPassword());
			
			return dbAccess;
		}
		catch (Exception e)
		{
			throw new DataSourceException("Instantion of '" + clazz + "' failed!", e);
		}
	}
	
	/**
	 * Gets the suitable DBAccess for the given connection.
	 * 
	 * @param pConnection the database connection.
	 * @return the suitable DBAccess.
	 * @throws DataSourceException if the instance name is unknown or the DBAccess object cannot be created.
	 */
	public static DBAccess getDBAccess(Connection pConnection) throws DataSourceException
	{
		if (pConnection == null)
		{
			throw new DataSourceException("Connection is null");
		}
		
		try
		{
			Class<? extends DBAccess> clazz = getDBAccessClass(pConnection.getMetaData().getURL());
			
			DBAccess dbAccess = clazz.newInstance();
			dbAccess.setConnection(pConnection);
			dbAccess.bExternalConnection = true;
			
			return dbAccess;
		}
		catch (Exception e)
		{
			throw new DataSourceException("Database detection for '" + pConnection.getClass().getName() + "' failed!", e);
		}
	}
	
	/**
	 * Gets the suitable DBAccess class for the jdbc url.
	 * 
	 * @param pJdbcUrl the jdbc url.
	 * @return the suitable DBAccess class.
	 */
	private static Class<? extends DBAccess> getDBAccessClass(String pJdbcUrl)
	{
		for (String key : dbAccessClasses.keySet())
		{
			if (pJdbcUrl.startsWith(key))
			{
				return dbAccessClasses.get(key);
			}
		}
		return DBAccess.class;
	}
	
	/**
	 * Registers the Class for the jdbc url prefix.
	 * 
	 * @param pJdbcUrlPrefix the jdbc url prefix.
	 * @param pClass the Class.
	 */
	public static void registerDBAccessClass(String pJdbcUrlPrefix, Class<? extends DBAccess> pClass)
	{
		dbAccessClasses.put(pJdbcUrlPrefix, pClass);
	}
	
	/**
	 * Sets the TranslationMap for the automatic link column name custom Translation. its used in the 
	 * getAutomaticLinkColName(ForeignKey pForeignKey) to change the column, thats determined. Default is *_ID to *, and *_id to *.
	 * 
	 * @param pTranslationMap	the TranslationMap to use.
	 */
	public static void setAutomaticLinkColumnNameTranslation(TranslationMap pTranslationMap)
	{
		tmpAutoLinkColumnNames = pTranslationMap;
	}

	/**
	 * Returns the TranslationMap for the automatic link column name custom Translation. its used in the 
	 * getAutomaticLinkColName(ForeignKey pForeignKey) to change the column, thats determined. Default is *_ID to *, and *_id to *.
	 * 
	 * @return the TranslationMap for the automatic link column name custom Translation.
	 */
	public static TranslationMap getAutomaticLinkColumnNameTranslation()
	{
		return tmpAutoLinkColumnNames;
	}
	
	/**
	 *  It creates and returns a from clause with joins over all foreign tables.
	 *  
	 * @param pWritebackTable		the write back table to use
	 * @param auForeignKeys			the ForeignKey array to use
	 * @return a from clause with joins over all foreign tables.
	 * @throws DataSourceException	if an error occur in determining the from clause and query columns.
	 */
	protected String getFromClause(String pWritebackTable, List<ForeignKey> auForeignKeys) throws DataSourceException
	{			
		StringBuilder sbFromClause = new StringBuilder(pWritebackTable);
		sbFromClause.append(" m ");			

		if (auForeignKeys != null)
		{
			for (int i = 0, anz = auForeignKeys.size(); i < anz; i++)
			{
				ForeignKey fkForeignKey = auForeignKeys.get(i);
			
				// use correct join type 
				if (fkForeignKey.isNullable())
				{
					sbFromClause.append("LEFT OUTER JOIN ");
				}
				else
				{
					sbFromClause.append("INNER JOIN ");
				}
	
				if (fkForeignKey.getPKSchema() != null && fkForeignKey.getPKSchema().getQuotedName().length() > 0)
				{
					sbFromClause.append(fkForeignKey.getPKSchema().getQuotedName());
					sbFromClause.append('.');
				}
				sbFromClause.append(fkForeignKey.getPKTable().getQuotedName());
				sbFromClause.append(" l");
				sbFromClause.append((i + 1));
				sbFromClause.append(" ON ");
	
				Name[] auPKColumns = fkForeignKey.getPKColumns();		
				Name[] auFKColumns = fkForeignKey.getFKColumns();
				
				for (int j = 0; j < auFKColumns.length; j++)
				{
					if (j > 0)
					{
						sbFromClause.append(" AND ");					
					}
					sbFromClause.append("m.");
					sbFromClause.append(auFKColumns[j].getQuotedName());
					sbFromClause.append(" = ");
					sbFromClause.append("l");
					sbFromClause.append((i + 1));
					sbFromClause.append('.');
					sbFromClause.append(auPKColumns[j].getQuotedName());	
				}
				sbFromClause.append("\n");
			}
		}

		logger.debug("FROM CLAUSE::", sbFromClause);				

		return sbFromClause.toString();
	}
	
	/**
	 * Returns the automatic link column name for the specified ForeignKey and FK Column in the master table.
	 * It uses the ForeignKey column name, or if more then one column exists, all of it combined with an "_". 
	 * If the PrimaryKey only exists out of one column and the ForeignKey uses more then one (Unique Key and PrimaryKey columns), 
	 * then its only the ForeignKey column corresponding to this one PrimaryKey column used.
	 * After the AutomaticLinkColumnName is created, the defined TranslationMap is used to change the name. 
	 * 
	 * @param pForeignKey	the ForeignKEy to use.
	 * @return the automatc link column name for the specified ForeignKey and FK Column in the master table.
	 * @see #setAutomaticLinkColumnNameTranslation(TranslationMap)
	 */
	protected String getAutomaticLinkColumnName(ForeignKey pForeignKey)
	{
		String sColPrefix = null;
		
		// if only one FK column, then use this.
		if (pForeignKey.getFKColumns().length == 1)
		{
			sColPrefix = pForeignKey.getFKColumns()[0].getName();
		}
		else
		{
			
			Name[] sPKCols = null;
			try
			{
				Key pk = getPK(pForeignKey.getPKCatalog().getRealName(), 
						pForeignKey.getPKSchema().getRealName(), 
						pForeignKey.getPKTable().getRealName());
				if (pk != null)
				{
					sPKCols = pk.getColumns();
				}
			}
			catch (DataSourceException e)
			{
				// ignore PK not found.
			}
			
			// more then one FK column, check if the PK has only one column, then use this. 
			if (sPKCols != null && sPKCols.length == 1)
			{
				// bug #224 - wrong AutomaticLinkColumnName if PK with one column, and FK with more the one column used!
				int i = ArrayUtil.indexOf(pForeignKey.getPKColumns(), sPKCols[0]);
				if (i >= 0)
				{
					sColPrefix = pForeignKey.getFKColumns()[i].getName();
				}
			}
			else
			{
				Name[] sFkCols = pForeignKey.getFKColumns();
				
				if (sFkCols != null && sFkCols.length > 0)
				{
					// use all FK columns separated by an "_"
					StringBuilder sbPrefix = new StringBuilder(sFkCols[0].getName());
					
					for (int i = 1; i < sFkCols.length; i++)
					{
						sbPrefix.append("_");
						sbPrefix.append(sFkCols[i].getName());
					}
					
					sColPrefix = sbPrefix.toString();
				}
			}
		}

		if (sColPrefix == null)
		{
			String sResult = pForeignKey.getLinkReferenceColumn().getName();
			if (sResult.length() > iMaxColumnLength)
			{
				sResult = sResult.substring(0, iMaxColumnLength);
			}
			return sResult;
		}
		else
		{
			// if an Translation for the Column prefix is set, then use it. e.g. *_ID to * , YYY_ID -> YYY
			if (tmpAutoLinkColumnNames != null)
			{
				sColPrefix = tmpAutoLinkColumnNames.translate(sColPrefix);
			}	
			String sResult = sColPrefix + "_" + pForeignKey.getLinkReferenceColumn().getName();
			if (sResult.length() > iMaxColumnLength)
			{
				sResult = sResult.substring(0, iMaxColumnLength);
			}
			return sResult;
		}
	}
	
	/**
	 *  It creates and returns the query columns, with all write back columns and includes the columns for the 
	 *  LinkReferences (automatic link celleditors).
	 *  It also removes all intersecting and unused ForeignKeys from the list of ForeignKeys!
	 *  
	 * @param pWritebackTable		the write back table to use
	 * @param pForeignKeys			the ForeignKey array to use
	 * @param pWritebackColumns		the write back table columns to use.
	 * @return the query columns, with all write back columns and includs the columns for the LinkReferences (automatic link celleditors).
	 * @throws DataSourceException	if an error occur in determining the from clause and query columns.
	 */
	protected String[] getQueryColumns(String pWritebackTable, List<ForeignKey> pForeignKeys, String[] pWritebackColumns) throws DataSourceException
	{			
        ArrayUtil<String> auQueryColumns             = new ArrayUtil<String>();
        ArrayUtil<String> auAutomaticLinkColumnNames = new ArrayUtil<String>();
        String            sAutomaticLinkColumnName   = null;
        
        ForeignKey fk;
        
        // make a list of ForeignKeys to process (don't use intersecting ForeignKeys) 
        List<ForeignKey> auForeignKeys = new ArrayUtil<ForeignKey>();
        List<ForeignKey> auIntersectKeys = new ArrayUtil<ForeignKey>();
        
        for (int i = 0, anz = pForeignKeys.size(); i < anz; i++)
        {
        	fk = pForeignKeys.get(i);
        	
        	if (i == 0)
        	{
        		// empty, just add it.
        		auForeignKeys.add(fk);
        	}
        	else
        	{
	        	// search for ForeignKey with FK column intersection
        		int index = -1;
        		
        		Name[] sCurrFKColumns = fk.getFKColumns();
	        	for (int j = 0, anzj = auForeignKeys.size(); j < anzj && index == -1; j++)
	        	{
	        		Name[] sCheckFKColumns = auForeignKeys.get(j).getFKColumns();
	        		
	        		for (int k = 0; k < sCheckFKColumns.length && index == -1; k++)
	        		{
		        		if (ArrayUtil.indexOf(sCurrFKColumns, sCheckFKColumns[k]) >= 0)
		        		{
		        			index = j;
		        		}	        			
	        		}
	        	}
	        	
	        	if (index < 0)
	        	{
	        		// no intersection, so add it!
	        		auForeignKeys.add(fk);
	        	}
	        	else
	        	{
	        		if (auForeignKeys.get(index).getFKColumns().length < fk.getFKColumns().length)
	        		{
	        			// already copied ForeignKey with the intersection has less columns then the current one
	        			// -> replace the old fk with the current fk.
	        			auIntersectKeys.add(auForeignKeys.set(index, fk));	        			
		        	}
	        	}
        	}
        }
        
        // remove all intersecting ForeignKeys from the result list!!!!!!
        pForeignKeys.removeAll(auIntersectKeys);
        
		// add all WritebackTable columns as Query Columns
		for (int i = 0; i < pWritebackColumns.length; i++)
		{
			auQueryColumns.add("m." + pWritebackColumns[i]);
			
			if (auForeignKeys != null)
			{
				for (int j = auForeignKeys.size() - 1; j >= 0; j--)
				{
					fk = auForeignKeys.get(j);
					
					if (ArrayUtil.indexOf(BeanUtil.toArray(fk.getFKColumns(), new String[fk.getFKColumns().length], "quotedName"), 
							              pWritebackColumns[i]) >= 0)
					{
						sAutomaticLinkColumnName = getAutomaticLinkColumnName(fk);
						
						if (auAutomaticLinkColumnNames.indexOf(sAutomaticLinkColumnName) < 0)
						{
							auAutomaticLinkColumnNames.add(sAutomaticLinkColumnName);
							auQueryColumns.add("l" + (pForeignKeys.indexOf(fk) + 1) + "." + fk.getLinkReferenceColumn().getQuotedName() +
		                      		          " " + sAutomaticLinkColumnName);
							auForeignKeys.remove(j);
						}
					}
				}
			}
		}		

		if (auForeignKeys != null)
		{
			for (int i = 0, anz = auForeignKeys.size(); i < anz; i++)
			{
				fk = auForeignKeys.get(i);

				auQueryColumns.add("l" + (pForeignKeys.indexOf(fk) + 1) + "." + fk.getLinkReferenceColumn().getQuotedName() +
	                		       " " + getAutomaticLinkColumnName(fk));
			}
		}
		
		logger.debug("sQueryColumns::", auQueryColumns);				

		return auQueryColumns.toArray(new String[auQueryColumns.size()]);
	}	
	
	/**
	 * It gets all columns for each Unique Key and return it.
	 * 
	 * @param pCatalog				the catalog to use
	 * @param pSchema				the schema to use
	 * @param pTable				the table to use
	 * @return all columns for each Unique Key. 
	 * @throws DataSourceException	if an error occur during UK search process.
	 */
	public List<Key> getUKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		ResultSet rsResultSet = null;
		try
		{
			ArrayUtil<Key>  auResult           = new ArrayUtil<Key>();
			ArrayUtil<Name> auUniqueKeyColumns = new ArrayUtil<Name>();
			
			long lMillis = System.currentTimeMillis();
			DatabaseMetaData dbMetaData = cConnection.getMetaData();
			
			rsResultSet = dbMetaData.getIndexInfo(pCatalog, pSchema, pTable, true, false);
			
			if (!rsResultSet.next())
			{
				rsResultSet.close();
				return auResult;
			}
			
			String sUKName = null;
			do
			{
				if (rsResultSet.getString("COLUMN_NAME") != null)
				{
					if (sUKName != null && !rsResultSet.getString("INDEX_NAME").equals(sUKName))
					{
						Key uk = new Key(sUKName);
						uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
						auResult.add(uk);
						auUniqueKeyColumns.clear();
					}
					sUKName = rsResultSet.getString("INDEX_NAME");
				
					auUniqueKeyColumns.add(new Name(rsResultSet.getString("COLUMN_NAME"), quote(rsResultSet.getString("COLUMN_NAME"))));
				}
			}
			while (rsResultSet.next());
			
			//[JR] #188
			if (auUniqueKeyColumns.size() > 0)
			{
				Key uk = new Key(sUKName);
				uk.setColumns(auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
				auResult.add(uk);
			}
			
			if (auResult.size() > 0)
			{
				// remove PKs, because a PK is also a index, but we don't wanna return them too.
				Key pk = getPK(pCatalog, pSchema, pTable);
				if (pk != null)
				{
					for (int i = auResult.size() - 1; i >= 0; i--)
					{
						Name[] ukCols = auResult.get(i).getColumns();
						if (ArrayUtil.containsAll(ukCols, pk.getColumns()) && ukCols.length == pk.getColumns().length)
						{
							auResult.remove(i);
						}
					}
				}
			}
			
			logger.debug("getUKs(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");							

			return auResult;
		}
		catch (SQLException sqlException)
		{
  			logger.error("Unique Keys couldn't determined from database! - ", pTable, sqlException);	
  			return null;
		}		
		finally
		{
			try
			{
	    		if (rsResultSet != null)
	    		{
	    			rsResultSet.close();
	    		}
			}
			catch (SQLException e)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
			}						
		}			
	}
	
	/**
	 * It's gets all Primary Key columns and return it as String[].
	 * 
	 * @param pCatalog				the catalog to use
	 * @param pSchema				the schema to use
	 * @param pTable				the table to use
	 * @return all Primary Key columns and return it as String[].
	 * @throws DataSourceException	if an error occur during PK search process.
	 */
	public Key getPK(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		ResultSet rsResultSet = null;
		try
		{
			long lMillis = System.currentTimeMillis();

			ArrayUtil<Name> auPrimaryKeyColumns = new ArrayUtil<Name>();
			String          sPKName             = null;
			
	        DatabaseMetaData dbMetaData = cConnection.getMetaData();
			rsResultSet = dbMetaData.getPrimaryKeys(pCatalog, pSchema, pTable);
			
			if (!rsResultSet.next())
			{
				rsResultSet.close();
				return null;
			}
		
			do
			{
				if (sPKName == null)
				{
					sPKName = rsResultSet.getString("PK_NAME");
				}
				auPrimaryKeyColumns.add(new Name(rsResultSet.getString("COLUMN_NAME"), quote(rsResultSet.getString("COLUMN_NAME"))));
			}
			while (rsResultSet.next());
			
			logger.debug("getPK(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");							

			Key pk = new Key(sPKName);
			pk.setColumns(auPrimaryKeyColumns.toArray(new Name[auPrimaryKeyColumns.size()]));

			return pk;
		}
  		catch (SQLException sqlException)
  		{
  			logger.error("PrimaryKey couldn't determined from database! - ", pTable, sqlException);	
  			return null;
  		}		
  		finally
  		{
  			try
  			{
  	    		if (rsResultSet != null)
  	    		{
  	    			rsResultSet.close();
  	    		}
  			}
  			catch (SQLException e)
  			{
  				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
  			}						
  		}		
	}
		
	
	/**
	 * Returns all Foreign Keys for the specified table.
	 *  
	 * @param pCatalog				the catalog to use
	 * @param pSchema				the schema to use
	 * @param pTable the table to use as base table.
	 * @return all Foreign Keys for the specified table.
	 * @throws DataSourceException	if an error occur in determining the ForeignKeys.
	 */
	public List<ForeignKey> getFKs(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		ResultSet rsResultSetMetaData = null;

		try
		{				
			ArrayUtil<ForeignKey> auForeignKeys = new ArrayUtil<ForeignKey>();

			long lMillis = System.currentTimeMillis();
	        DatabaseMetaData dbMetaData = cConnection.getMetaData();
			
			rsResultSetMetaData = dbMetaData.getImportedKeys(pCatalog, pSchema, pTable);	
			if (!rsResultSetMetaData.next())
			{
				rsResultSetMetaData.close();
				// find no FKs -> From Clause can't determined
				return auForeignKeys;
			}
									
			String sTempKeyName = null;
			String sKeyName = null;
			ArrayUtil<Name> auPKColumns = new ArrayUtil<Name>();			
			ArrayUtil<Name> auFKColumns = new ArrayUtil<Name>();			

			ForeignKey fkForeignKey = new ForeignKey(
					new Name(rsResultSetMetaData.getString("PKTABLE_NAME"), quote(rsResultSetMetaData.getString("PKTABLE_NAME"))), 
					new Name(rsResultSetMetaData.getString("PKTABLE_CAT"), quote(rsResultSetMetaData.getString("PKTABLE_CAT"))), 
					new Name(rsResultSetMetaData.getString("PKTABLE_SCHEM"), quote(rsResultSetMetaData.getString("PKTABLE_SCHEM"))));
			
			do
			{
				sTempKeyName = rsResultSetMetaData.getString("FK_NAME");
				if (sTempKeyName == null || sTempKeyName.length() == 0)
				{
					sTempKeyName = rsResultSetMetaData.getString("PK_NAME");
					if (sTempKeyName == null || sTempKeyName.length() == 0)
					{
						throw new DataSourceException("Database/jdbc driver didn't support FK, PK names!");
					}
				}
				if (sKeyName != null && !sKeyName.equals(sTempKeyName))
				{
					fkForeignKey.setFKColumns(auFKColumns.toArray(new Name[auFKColumns.size()]));
					fkForeignKey.setPKColumns(auPKColumns.toArray(new Name[auPKColumns.size()]));
					auForeignKeys.add(fkForeignKey);
					
					auPKColumns.clear();			
					auFKColumns.clear();	
					
					fkForeignKey = new ForeignKey(
							new Name(rsResultSetMetaData.getString("PKTABLE_NAME"), quote(rsResultSetMetaData.getString("PKTABLE_NAME"))), 
							new Name(rsResultSetMetaData.getString("PKTABLE_CAT"), quote(rsResultSetMetaData.getString("PKTABLE_CAT"))), 
							new Name(rsResultSetMetaData.getString("PKTABLE_SCHEM"), quote(rsResultSetMetaData.getString("PKTABLE_SCHEM"))));
				}
				sKeyName = sTempKeyName;
				
				auPKColumns.add(new Name(rsResultSetMetaData.getString("PKCOLUMN_NAME"), quote(rsResultSetMetaData.getString("PKCOLUMN_NAME"))));
				auFKColumns.add(new Name(rsResultSetMetaData.getString("FKCOLUMN_NAME"), quote(rsResultSetMetaData.getString("FKCOLUMN_NAME"))));			
				
				fkForeignKey.setFKName(rsResultSetMetaData.getString("FK_NAME"));					
			}
			while (rsResultSetMetaData.next());

			fkForeignKey.setFKColumns(auFKColumns.toArray(new Name[auFKColumns.size()]));
			fkForeignKey.setPKColumns(auPKColumns.toArray(new Name[auPKColumns.size()]));
			auForeignKeys.add(fkForeignKey);
			
			logger.debug("getFKs(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");	
			
			return auForeignKeys;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Foreign Keys couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}		
		finally
		{
			try
			{
	    		if (rsResultSetMetaData != null)
	    		{
	    			rsResultSetMetaData.close();
	    		}
			}
			catch (SQLException e)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
			}						
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public List<Object[]> fetch(ServerMetaData pServerMetaData, String pBeforeQueryColumns, String[] pQueryColumns, String pFromClause,
								ICondition pFilter, String pLastWhereCondition, String pAfterWhereClause, 
								SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{ 
		if (!isOpen())
		{
			throw new DataSourceException("DBAccess is not open!");
		}

		// add select statement (without order by clause)
		StringBuilder sbSelectStatement = new StringBuilder(getSelectStatement(pFilter, pFromClause, pQueryColumns, 
								                          pBeforeQueryColumns, pLastWhereCondition, pAfterWhereClause, 
								                          pServerMetaData));
				
		ServerColumnMetaData[] scmd = pServerMetaData.getServerColumnMetaData();

		// add Sort
		if (pSort != null)
		{
			// use DB sorting algorithm
			sbSelectStatement.append(" ORDER BY ");
			
			String[]  saSortDefinitionNames = pSort.getColumns();
			boolean[] baSortDefinitionOrder = pSort.isAscending();
			
			String[] saColumns = BeanUtil.toArray(scmd, new String[scmd.length], "name");
			
			for (int i = 0; i < saSortDefinitionNames.length; i++)
			{
				int index = ArrayUtil.indexOf(saColumns, saSortDefinitionNames[i]);
				if (index < 0)
				{
					sbSelectStatement.append(saSortDefinitionNames[i]);
				}
				else
				{
					sbSelectStatement.append(scmd[index].getColumnName().getQuotedName());
				}
				if (baSortDefinitionOrder[i])
				{
					sbSelectStatement.append(" ASC ");
				}
				else
				{
					sbSelectStatement.append(" DESC ");
				}
				if (i + 1 < saSortDefinitionNames.length)
				{
					sbSelectStatement.append(", ");
				}
			}
		}
		
		PreparedStatement psSelect = null;		
		ResultSet rsResultSet = null;
		try 
		{
			long lMillis = System.currentTimeMillis();
			String sSelectStatement = sbSelectStatement.toString();

			Select currentSelect = new Select(sSelectStatement, pFilter, pFromRow);
			rsResultSet = htFetchResultSetCache.get(currentSelect);
			
			List<Object[]> auResult = new ArrayUtil<Object[]>();

			if (rsResultSet == null)
			{
				psSelect = getPreparedStatement(sSelectStatement, false);

				if (iQueryTimeOut > 0)
				{
					psSelect.setQueryTimeout(iQueryTimeOut);
				}
	
				// set Filter Parameter Values
				if (pFilter != null)
				{
					setFilterParameter(1, psSelect, pFilter);
				}		
				rsResultSet = psSelect.executeQuery();

				for (int j = 0; j < pFromRow; j++)
				{
					if (!rsResultSet.next())
					{
						auResult.add(null);							
						return auResult;
					}
				}
				
				if (logger.isEnabled(LogLevel.DEBUG))
				{
					logger.debug("select(", sSelectStatement, ", ", getParameter(pFilter), 
							", from Row ", Integer.valueOf(pFromRow), ", MinimumCount ", Integer.valueOf(pMinimumRowCount), ") in ",
						         Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
				}
			}
			else
			{
				psSelect = (PreparedStatement)rsResultSet.getStatement();
			}
			ResultSetMetaData rmSelectMetaData = rsResultSet.getMetaData();

			// try to fetch all rows in iMaxTime millis
			lMillis = System.currentTimeMillis();
			int  iTimeLeft = iMaxTime;

			int  iColumnCount = rmSelectMetaData.getColumnCount();
			
			int i = 0;
			for (; pMinimumRowCount < 0 || i < pMinimumRowCount || iTimeLeft >= 0; i++)				
			{
				Object[] oRow = null;
				
				if (rsResultSet.next())
				{
					oRow = new Object[iColumnCount];
					
					for (int j = 0; j < iColumnCount; j++)
					{
						Object oValue = rsResultSet.getObject(j + 1);
						
						// convert to Java types
						if (oValue instanceof Number
								 && !(oValue instanceof BigDecimal))
						{
							oValue = rsResultSet.getBigDecimal(j + 1);
						}
						else if (oValue instanceof java.util.Date 
							&& !(oValue instanceof Timestamp))
						{
							oValue = rsResultSet.getTimestamp(j + 1);
						} 
						else if (oValue instanceof java.sql.Clob)
						{
							Clob cValue = (Clob)oValue;
							oValue = cValue.getSubString(1, (int)cValue.length());
						}  
						else if (oValue instanceof java.sql.Blob)
						{
							Blob bValue = (Blob)oValue;
							oValue = bValue.getBytes(1, (int)bValue.length());
						}

						oRow[j] = convertDatabaseSpecificObjectToValue(scmd[j], oValue);
					}
					auResult.add(oRow);
					iTimeLeft = iMaxTime - (int)(System.currentTimeMillis() - lMillis);
				}
				else
				{
					auResult.add(null);	
					htFetchResultSetCache.remove(currentSelect);
					
					if (rsResultSet != null)
					{
						rsResultSet.close();
					}
					if (psSelect != null)
					{
						psSelect.close();
					}
					
					return auResult;
				}				
			}
			
			if (auResult.size() == 0)
			{
				auResult.add(null);
				
				if (rsResultSet != null)
				{
					rsResultSet.close();
				}
				if (psSelect != null)
				{
					psSelect.close();
				}
			}
			else
			{
				htFetchResultSetCache.remove(currentSelect);
	
				// save current ResultSetCursor and use the next time!!!
				htFetchResultSetCache.put(new Select(sSelectStatement, pFilter, pFromRow + i), rsResultSet);
			}
			
			return auResult;
		}
		catch (Exception ex)
		{
			if (rsResultSet != null)
			{
				try
				{
					rsResultSet.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}
			}
			
			if (psSelect != null)
			{
				try
				{
					psSelect.close();
				}
				catch (Exception e)
				{
					//nothing to be done
				}			
			}
			
			throw new DataSourceException("fetch statement failed! - " + sbSelectStatement, (ex instanceof SQLException ? formatSQLException((SQLException)ex) : ex));
		}		
	}

	/**
	 * {@inheritDoc}
	 */
	public void lockRow(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{	
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}
		
		if (pWritebackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}		

		try
		{
			if (!cConnection.getAutoCommit() && cConnection.getMetaData().supportsSelectForUpdate())
			{
				lockRowInternal(pWritebackTable, pServerMetaData, pPKFilter);
			}
		}
		catch (SQLException se)
		{
			throw new DataSourceException("Execute locking failed!", formatSQLException(se));			
		}
	}
		
	/**
	 * It locks the current row and return how many rows are affected.
	 * 
	 * @param pWritebackTable the storage unit to use
	 * @param pPKFilter 		the PrimaryKey in as an <code>ICondition</code> to identify the row to lock   
	 * @param pServerMetaData	the MetaDataColumn array to use.
	 * @return the counts of affected rows
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage 
	 */		
	protected int lockRowInternal(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{	
		return getRowCount(getDatabaseSpecificLockStatement(pWritebackTable, pServerMetaData, pPKFilter), pPKFilter, 1);
	}
	
    /**
	 * {@inheritDoc}
	 */
	public Object[] insert(String pWritebackTable, ServerMetaData pServerMetaData, Object[] pNewDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		if (pWritebackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}
		
		StringBuilder sInsertStatement = new StringBuilder("INSERT INTO ");
		sInsertStatement.append(pWritebackTable);
		sInsertStatement.append(" (");

		// add column names to insert
		int iColumnCount = 0;
		String sDummyColumn = null;
		
		ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
		
		for (int i = 0; i < iaWriteables.length; i++)
		{
			if (pNewDataRow[iaWriteables[i]] != null)
			{
				if (iColumnCount > 0)
				{
					sInsertStatement.append(", ");
				}
				
				sInsertStatement.append(cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getQuotedName());
				iColumnCount++;
			}
		}
		if (iColumnCount == 0)
		{
			// if no storable columns, put in a dummy one
			for (int i = 0; iColumnCount == 0 && i < iaWriteables.length; i++)
			{
				if (pServerMetaData.getPrimaryKeyColumnNames() == null
					|| pServerMetaData.getPrimaryKeyColumnNames().length == 0
					|| ArrayUtil.indexOf(pServerMetaData.getPrimaryKeyColumnNames(), 
							             cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getName()) == -1)
				{
					sDummyColumn = cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getQuotedName();
					sInsertStatement.append(sDummyColumn);
					iColumnCount++;
				}
			}
		}

		// Add values '?' to insert
		sInsertStatement.append(") VALUES (");
		
		if (iColumnCount > 0)
		{
			sInsertStatement.append("?");
			
			for (int i = 1; i < iColumnCount; i++)
			{
				sInsertStatement.append(",?");
			}
		}
		sInsertStatement.append(")");

		pNewDataRow = insertDatabaseSpecific(pWritebackTable, sInsertStatement.toString(), pServerMetaData, pNewDataRow, sDummyColumn);
		
		if (logger.isEnabled(LogLevel.DEBUG))
		{
			logger.debug(sInsertStatement.toString(), "[", pNewDataRow, "]");
		}
		
		// check Empty PK and throw Exception
		boolean bPKEmpty = true;
		int[] iPKColsIndices = pServerMetaData.getPrimaryKeyColumnIndices();
		
		if (iPKColsIndices != null)
		{
			for (int i = 0; i < iPKColsIndices.length && bPKEmpty; i++)
			{
				if (pNewDataRow[iPKColsIndices[i]] != null)
				{
					bPKEmpty = false;
				}
			}
		}
		
		if (bPKEmpty)
		{
			throw new DataSourceException("Primary key column empty after insert!" + pWritebackTable);
		}
		
		return pNewDataRow;
	}

	/**
	 * {@inheritDoc}
	 */
	public Object[] update(String pWritebackTable, ServerMetaData pServerMetaData, Object[] pOld, Object[] pNew) throws DataSourceException
	{	
		// TODO[RH] Sollte vielleicht geprft werden, ob die new Row unterschiedlich zu den Werten in der DB ist -> davor fetch!!!
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		if (pWritebackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}
		
		if (pServerMetaData.getPrimaryKeyColumnNames() == null || pServerMetaData.getPrimaryKeyColumnNames().length == 0)
		{
			throw new DataSourceException("PK Columns empty! - update not possible!");
		}
		
		StringBuilder sUpdateStatement = new StringBuilder("UPDATE ");
		sUpdateStatement.append(pWritebackTable);
		sUpdateStatement.append(" SET ");

		// add column names to update
		int iColumnCount = 0;
		ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
		
		for (int i = 0; i < iaWriteables.length; i++)
		{
			IDataType dtDataType = cmdServerColumnMetaData[iaWriteables[i]].getDataType();
	        if (dtDataType.compareTo(pNew[iaWriteables[i]], pOld[iaWriteables[i]]) != 0)
	        {
				if (iColumnCount > 0)
				{
					sUpdateStatement.append(", ");
				}
				sUpdateStatement.append(cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getQuotedName());
				sUpdateStatement.append(" = ? ");
				iColumnCount++;
	        }
		}
		
		// construct Filter over PK cols 
		ICondition pPKFilter = Filter.createEqualsFilter(
				pServerMetaData.getPrimaryKeyColumnNames(), pOld, pServerMetaData.getMetaData().getColumnMetaData());

		int iCount = 0;
		if (iColumnCount == 0)
		{
			// no storable columns; then....
			
			try
			{
				// if select for update supported from db and not in autocommit mode, then 
				// lock the record and determine the count of rows which exists to this PK!
				
				// the count of rows is important, to decide later if we make an update(iCount==1), insert(iCount==0) 
				// or throw an error because to many rows (iCount>1) are affected from the update.
				if (!cConnection.getAutoCommit() && cConnection.getMetaData().supportsSelectForUpdate())
				{
					iCount = lockRowInternal(pWritebackTable, pServerMetaData, pPKFilter);
				}
				else
				{
					// otherwise only determine the count of rows exists for this PK 
					StringBuilder sbfSelect = new StringBuilder("SELECT * FROM ");
					sbfSelect.append(pWritebackTable);
					sbfSelect.append(getWHEREClause(pPKFilter, pServerMetaData, false));						
					
					iCount = getRowCount(sbfSelect.toString(), pPKFilter, 1); 
				}
			}
			catch (SQLException se)
			{
				throw new DataSourceException("Connection access failed!", formatSQLException(se));			
			}
		}
		else
		{
			// add WHERE Clause		
			sUpdateStatement.append(getWHEREClause(pPKFilter, pServerMetaData, false));		

			iCount = updateDatabaseSpecific(pWritebackTable, sUpdateStatement.toString(), pServerMetaData, 
					                        pOld, pNew, pPKFilter);
			
			if (logger.isEnabled(LogLevel.DEBUG))
			{
				logger.debug(sUpdateStatement.toString(), "[", pNew, "]");
			}
		}

		// the count of rows is important to decide if we make an update(iCount==1), insert(iCount==0) 
		// or throw an error because to many rows (iCount>1) are affected from the update.
		if (iCount == 0)
		{
			return insert(pWritebackTable, pServerMetaData, pNew);
		}
		else if (iCount != 1)
		{
			throw new DataSourceException("Update failed ! - Result row count != 1 ! - " +  sUpdateStatement.toString());
		}
		
		if (iColumnCount == 0)
		{
			// no storable columns; then.... return oldRow
			return pOld;
		}
		return pNew;
	}

	/**
	 * Updates the specified row and return the count of affected rows. <br>
	 * Database specific implementation should override this method to implement the specific update code.
	 * 
	 * @param pWritebackTable	the table to use for the update
	 * @param sUpdateStatement	the SQL Statement to use for the update
	 * @param pServerMetaData	the meta data to use.
	 * @param pOld				the old row (values) to use.
	 * @param pNew				the new row (values) to use.
	 * @param pPKFilter			the PrimaryKey equals filter to use.
	 * @return the count of updates rows from a Ansi SQL Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during update to the storage
	 */
	public int updateDatabaseSpecific(String pWritebackTable, String sUpdateStatement, 
			ServerMetaData pServerMetaData, Object[] pOld, Object[] pNew, ICondition pPKFilter)  throws DataSourceException
	{
		return updateAnsiSQL(pWritebackTable, sUpdateStatement, pServerMetaData, pOld, pNew, pPKFilter);
	}

	/**
	 * Updates the specified row and return the count of affected rows. <br>
	 * 
	 * @param pWritebackTable	the table to use for the update
	 * @param sUpdateStatement	the SQL Statement to use for the update
	 * @param pServerMetaData	the meta data to use.
	 * @param pOld				the old row (values) to use.
	 * @param pNew				the new row (values) to use.
	 * @param pPKFilter			the PrimaryKey equals filter to use.
	 * @return the count of updates rows from a Ansi SQL Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during update to the storage
	 */
	public int updateAnsiSQL(String pWritebackTable, String sUpdateStatement, 
			ServerMetaData pServerMetaData, Object[] pOld, Object[] pNew, ICondition pPKFilter)  throws DataSourceException
	{
		ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
		
		// update DataRow into database
		PreparedStatement psUpdate = getPreparedStatement(sUpdateStatement, false);

		// set Parameter with storable Column Values
		int iLastParameterIndex = setColumnsToStore(psUpdate, cmdServerColumnMetaData, iaWriteables, pNew, pOld);
		
		// set WHERE Parameter with Values from PK of the old DataRow
		setFilterParameter(++iLastParameterIndex, psUpdate, pPKFilter);
		
		int iCount = executeUpdate(psUpdate);
		try
		{
			psUpdate.close();
			return iCount;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Jdbc statement close failed", formatSQLException(sqlException));
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void delete(String pWritebackTable, ServerMetaData pServerMetaData, Object[] pDelete) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		if (pWritebackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}

		if (pServerMetaData.getPrimaryKeyColumnNames() == null || pServerMetaData.getPrimaryKeyColumnNames().length == 0)
		{
			throw new DataSourceException("PK Columns empty! - delete not possible!");
		}
		
		ICondition pPKFilter = null;
		
		ServerColumnMetaData[] pServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		String[] pPKColumns = pServerMetaData.getPrimaryKeyColumnNames();
		
		if (pServerColumnMetaData != null)
		{
			pPKFilter = Filter.createEqualsFilter(pPKColumns, pDelete, pServerMetaData.getMetaData().getColumnMetaData());
		}
		else
		{
			// if pColumnMetaData is set, we use the pDelete columns as it is as values for the Equals
			// over the primary key columns
			pPKFilter = new Equals(pPKColumns[0], pDelete[0]);
			for (int i = 1; i < pPKColumns.length; i++)
			{
				pPKFilter = pPKFilter.and(new Equals(pPKColumns[i], pDelete[i]));
			}
		}
		
		StringBuilder sDeleteStatement = new StringBuilder("DELETE FROM ");
		sDeleteStatement.append(pWritebackTable);
		
		String sWHERE = getWHEREClause(pPKFilter, pServerMetaData, false);
		sDeleteStatement.append(sWHERE);
				
		logger.debug("delete(", sDeleteStatement, ")", pPKColumns, pDelete);

		// delete DataRow in database
		PreparedStatement psDelete = getPreparedStatement(sDeleteStatement.toString(), false);
		
		// set Filter Parameter Values
		setFilterParameter(1, psDelete, pPKFilter);
		
		int iCount = executeUpdate(psDelete);
		try
		{
			psDelete.close();
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Jdbc statement close failed", formatSQLException(sqlException));
		}
		
		if (iCount > 1)
		{
			throw new DataSourceException("Delete failed ! - Result row count > 1 ! - " + iCount + "," + sDeleteStatement.toString());
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		StringBuilder sbResult = new StringBuilder();
		
		String sShowPassword;
		if (sPassword != null)
		{
			sShowPassword = ",Password set";
		}
		else
		{
			sShowPassword = ",Password not set";
		}

		boolean bOpen = false;
		try 
		{
			bOpen = isOpen();
		}
		catch (DataSourceException dataSourceException)
		{
			return sbResult.toString() + " :: " + dataSourceException.getMessage();
		}
		
		Properties propCopy = new Properties(properties);
		propCopy.remove("user");
		propCopy.remove("password");
		
		sbResult.append("DBAccess :: Connected=");
		sbResult.append(bOpen);
		sbResult.append(", ConnectionString=");
		sbResult.append(sUrl);
		sbResult.append(", DriverName=");
		sbResult.append(sDriver);
		sbResult.append(", UserName=");
		sbResult.append(sUsername);
		sbResult.append(sShowPassword);
		sbResult.append(",Properties=");
		sbResult.append(propCopy);
		sbResult.append("\n");
		
		sbResult.append("ResultSet Cache:\n");
		
		for (Map.Entry<Select, ResultSet> entry : htFetchResultSetCache.entrySet())
		{
			sbResult.append(entry.getKey());
			sbResult.append("\n");
		}

		sbResult.append(super.toString());
		
		return sbResult.toString();
	}	

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Sets the database specific quote characters.
	 * 
	 * @param pOpen the open quote character
	 * @param pClose the close quote character
	 */
	protected void setQuoteCharacters(String pOpen, String pClose)
	{
		sOpenQuote  = pOpen;
		sCloseQuote = pClose;
	}
	
	/**
	 * It returns true if this name should be automated quoted. 
	 * e.g. in Oracle default all isUppercase(), so if the name has one loweCase character, then AutoQuote is true to quote this name.
	 * 
	 * @param pName the name to quote.
	 * @return true if this name should be automated quoted.
	 */
	public boolean isAutoQuote(String pName)
	{
		CaseSensitiveType type = StringUtil.getCaseSensitiveType(pName);
		
		return type == CaseSensitiveType.UpperCase ? false : true;
	}
	
	/**
	 * Quotes a named DB object with our internal quote character, if it should be quoted in this database.
	 * Thats the case if the name != to the default case sensitivness. e.g. in Oracle != upperCase()
	 * 
	 * @param pName	the name to use. 
	 * @return the quoted name if quoting in this database is necessary or otherwise just the name.
	 */
	public String quote(String pName)
	{
		if (pName == null)
		{
			return null;
		}
		if (isAutoQuote(pName))
		{
			return quoteAllways(pName);		
		}
		return pName;
	}
	
	/**
	 * Quotes a named DB object with the JVx DB QUOTE character.
	 * 
	 * @param pName		the name to use. 
	 * @return the name quoted.
	 */
	public String quoteAllways(String pName)
	{
		StringBuilder sbResult = new StringBuilder(QUOTE); 
		sbResult.append(StringUtil.replace(pName, QUOTE, "\\" + QUOTE));
		sbResult.append(QUOTE);
		return sbResult.toString();			
	}

	/**
	 * Removes the JVx DB Quotes of a named DB object.
	 * 
	 * @param pName		the name to use. 
	 * @return the unquoted name.
	 */
	public static String removeQuotes(String pName)
	{
		if (pName == null)
		{
			return null;
		}
		StringBuilder sbResult = new StringBuilder(); 
		char          q        = QUOTE.charAt(0);
		for (int i = 0, iSize = pName.length(); i < iSize; i++)
		{
			char curr = pName.charAt(i);
			if (q == curr)
			{
				// remove escape character
				if (i != 0 && pName.charAt(i - 1) == '\\')
				{
					sbResult.deleteCharAt(sbResult.length() - 1);
					sbResult.append(curr);
				}
			}
			else
			{
				sbResult.append(curr);
			}
		}
		return sbResult.toString();	
	}
	
	/**
	 * Removes the DB specific quotes of a named DB object.
	 * 
	 * @param pName		the name to use. 
	 * @return the unquoted name.
	 */
	public String removeDBSpecificQuotes(String pName)
	{
		return StringUtil.removeQuotes(pName, sOpenQuote, sCloseQuote);		
	}
	
	/**
	 * It replaces all JVx quotes with the database specific quote.
	 * 
	 * @param pStatement		the statement to use. 
	 * @return the database specific quoted statement.
	 */
	public String translateQuotes(String pStatement)
	{
		if (pStatement == null)
		{
			return null;
		}
		
		StringBuilder sbResult = new StringBuilder(); 
		char          q        = QUOTE.charAt(0);
		boolean       bUseOpen = true;
		for (int i = 0, iSize = pStatement.length(); i < iSize; i++)
		{
			char curr = pStatement.charAt(i);
			if (q == curr)
			{
				// remove escape character
				if (i != 0 && pStatement.charAt(i - 1) == '\\')
				{
					sbResult.deleteCharAt(sbResult.length() - 1);
					sbResult.append(curr);
				}
				else
				{
					if (bUseOpen)
					{
						sbResult.append(sOpenQuote);					
						bUseOpen = false;
					}
					else
					{
						sbResult.append(sCloseQuote);					
						bUseOpen = true;
					}
				}
			}
			else
			{
				sbResult.append(curr);
			}
		}
		return sbResult.toString();			
	}	
	
	/**
	 * It opens the database and stores the <code>Connection</code> object.
	 * 
	 * @throws DataSourceException
	 *             if the database couldn't opened
	 */
	public void open() throws DataSourceException
	{
		//#607
		if (!isOpen())
		{
			long lMillis = System.currentTimeMillis();
			
			if (!bExternalConnection)
			{
				if (sDriver == null)
				{
					throw new DataSourceException("Jdbc Driver is null!");
				}
				if (sUrl == null)
				{
					throw new DataSourceException("Connection String is null!");
				}
				try
				{
					Class.forName(sDriver);
				}
				catch (Exception exception)
				{
					throw new DataSourceException("Jdbc driver not found!", exception);
				}
				
				try
				{
					Properties propCopy = new Properties(properties);
					propCopy.setProperty("user", translateQuotes(sUsername));
					propCopy.setProperty("password", sPassword);
					
					Connection con = DriverManager.getConnection(sUrl, propCopy);
					
					//#515
					if (con.getTransactionIsolation() != Connection.TRANSACTION_READ_COMMITTED)
					{
						con.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
					}
					
					setConnection(con);
				}
				catch (SQLException sqlException)
				{
					String sShowPassword;
					if (sPassword != null)
					{
						sShowPassword = "Password set";
					}
					else
					{
						sShowPassword = "Password not set";
					}
					
					throw new DataSourceException("Connection failed! - " + sUrl + "; Username=" +
							                      sUsername + "; " + sShowPassword, formatSQLException(sqlException));
				}
			}
			
			try
			{
				iMaxColumnLength = cConnection.getMetaData().getMaxColumnNameLength();
				// #447 - if getMaxColumnNameLength() return 0, we should interprete it as unlimmited 
				if (iMaxColumnLength == 0)
				{
					iMaxColumnLength = Integer.MAX_VALUE;
				}
			}
			catch (SQLException sqlException)
			{
				iMaxColumnLength = 30;
				
				logger.debug(sqlException);
			}
			
			logger.debug("open(", sUrl, ",", translateQuotes(sUsername), ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
		}
		else
		{
			logger.debug("connection is already open!");
		}
	}

	/**
	 * Returns true, if the database is still open.
	 * 
	 * @return true, if the database is still open.
	 * @throws DataSourceException
	 *             if isClosed() on the DB <code>Connection</code> throws an Exception
	 */
	public boolean isOpen() throws DataSourceException
	{
		try
		{
			return cConnection != null && !cConnection.isClosed();
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Open failed!", formatSQLException(sqlException));
		}
	}

	/**
	 * Closes the database <code>Connection</code> and releases all memory.
	 * 
	 * @throws DataSourceException if database couldn't closed.
	 */
	public void close() throws DataSourceException
	{
		if (!bExternalConnection)
		{
			try
			{
				if (isOpen())
				{
					cConnection.close();
					
					setConnection(null);
				}
			}
			catch (SQLException sqlException)
			{
				setConnection(null);
	
				throw new DataSourceException("Close failed!", formatSQLException(sqlException));
			}
		}
	}

	/**
	 * Rollback the DB transaction.
	 * 
	 * @throws DataSourceException if the transaction couldn't rollback
	 */
	public void rollback() throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}
		try
		{
			cConnection.rollback();	
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Rollback failed!", formatSQLException(sqlException));
		}
	}

	/**
	 * Commits the DB transaction.
	 * 
	 * @throws DataSourceException
	 *             if the transaction couldn't commit
	 */
	public void commit() throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}
		try
		{
			cConnection.commit();
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Commit failed!", formatSQLException(sqlException));
		}
	}
	
	/**
	 * Sets the query time out.
	 * 
	 * @param pQueryTimeOut
	 * 				the timeout in miliseconds
	 */
	public void setQueryTimeOut(int pQueryTimeOut)
	{
		iQueryTimeOut = pQueryTimeOut;
	}
	
	/**
	 * Returns the <code>conncetion</code> to the database.
	 * 
	 * @return the <code>conncetion</code> to the database.
	 */
	public Connection getConnection()
	{
		return cConnection;
	}	
	
	/**
	 * Sets the internal <code>conncetion</code> to the database.
	 * 
	 * @param pConnection the <code>conncetion</code> to the database
	 */
	protected void setConnection(Connection pConnection)
	{
		cConnection = pConnection;
	}
	
	/**
	 * Gets the database driver name as <code>String</code>.
	 * 
	 * @return pDriverName the database driver name
	 */
	public String getDriver()
	{
		return sDriver;
	}

	/**
	 * Sets the database driver name as <code>String</code>.
	 * 
	 * @param pDriver the database driver name
	 */
	public void setDriver(String pDriver)
	{
		sDriver = pDriver;
	}

	/**
	 * Gets the jdbc url <code>String</code> for this database.
	 * 
	 * @return	the jdbc url <code>String</code>.
	 */
	public String getUrl()
	{
		return sUrl;
	}

	/**
	 * Sets the url <code>String</code> for this database.
	 * 
	 * @param pUrl	the jdbc url <code>String</code>.
	 */
	public void setUrl(String pUrl)
	{
		sUrl = pUrl;
	}

	/**
	 * Gets the user name to connect with.
	 * 
	 * @return pUser	the user name
	 */
	public String getUsername()
	{
		return sUsername;
	}

	/**
	 * Sets the user name to connect with.
	 * 
	 * @param pUsername  the user name
	 */
	public void setUsername(String pUsername)
	{
		sUsername = pUsername;
	}

	/**
	 * Sets the password to use for the connection to the database.
	 * 
	 * @return pPassword the password to use for the database
	 */
	public String getPassword()
	{
		return sPassword;
	}

	/**
	 * Sets the password to use for the connection to the database.
	 * 
	 * @param pPassword the password to use for the database
	 */
	public void setPassword(String pPassword)
	{
		sPassword = pPassword;
	}

	/**
	 * Sets a specific database property.
	 * 
	 * @param pName the property name
	 * @param pValue th value
	 */
	public void setDBProperty(String pName, String pValue)
	{
		if (pValue == null)
		{
			properties.remove(pName);
		}
		else
		{
			properties.setProperty(pName, pValue);
		}
	}
	
	/**
	 * Gets the value for a specific database property.
	 * 
	 * @param pName the property name
	 * @return the value or <code>null</code> if the property was not found
	 */
	public String getDBProperty(String pName)
	{
		return properties.getProperty(pName);
	}
	
	/**
	 * Sets DB specific initial parameters for the <code>Connection</code> creation.
	 * 
	 * @param pProperties
	 *            DB specific initial parameters
	 */
	public void setDBProperties(Properties pProperties)
	{
		properties = pProperties;
	}

	/**
	 * Returns the DB specific initial parameters for the <code>Connection</code> creation.
	 * 
	 * @return the DB specific initial parameters for the <code>Connection</code> creation.
	 */
	public Properties getDBProperties()
	{
		return properties;
	}

	/**
	 * Returns the maximum time in miliseconds to use, to try to fetch all rows. reduce open cursors, and increase performance.
	 *
	 * @return the iMaxTime.
	 */
	public int getMaxTime()
	{
		return iMaxTime;
	}

	/**
	 * Sets the maximum time in miliseconds to use, to try to fetch all rows. reduce open cursors, and increase performance.
	 *
	 * @param pMaxTime the iMaxTime to set
	 */
	public void setMaxTime(int pMaxTime)
	{
		iMaxTime = pMaxTime;
	}
	
	/**
	 * Executes a DB procedure with the specified parameters.
	 * 
	 * @param pProcedureName the procedure (optional with package) name. 
	 * @param pParameters the parameters to use with the correct and corresponding java type.
	 * @throws DataSourceException if the call failed.
	 */
	public void executeProcedure(String pProcedureName, Object...pParameters) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		StringBuilder sqlStatement = new StringBuilder("{ call ");
		sqlStatement.append(pProcedureName);
		
		if (pParameters.length > 0)
		{
			sqlStatement.append("(");
			for (int i = 0; i < pParameters.length; i++)
			{
				if (i > 0)
				{
					sqlStatement.append(", ");			
				}
				sqlStatement.append("?");			
			}
			sqlStatement.append(")");
		}
		sqlStatement.append(" }");
		
		CallableStatement call = null;
		try
		{
			logger.debug("executeProcedure -> ", sqlStatement);
			
			AbstractParam apParam;
			
			ParameterType type;

			call = cConnection.prepareCall(translateQuotes(sqlStatement.toString()));
			for (int i = 0; pParameters != null && i < pParameters.length; i++)
			{
				if (pParameters[i] == null)
				{
					call.setNull(i + 1, Types.VARCHAR);
				}
				else
				{
					if (pParameters[i] instanceof AbstractParam)
					{
						apParam = (AbstractParam)pParameters[i];

						type = apParam.getType();
						
						if (type == ParameterType.Out || type == ParameterType.InOut)
						{
							call.registerOutParameter(i + 1, apParam.getSqlType());
						}
						
						if (apParam.getValue() == null)
						{
							call.setNull(i + 1, apParam.getSqlType());							
						}
						else
						{
							call.setObject(i + 1, convertValueToDatabaseSpecificObject(apParam.getValue()), apParam.getSqlType());
						}
					}
					else
					{
						call.setObject(i + 1, convertValueToDatabaseSpecificObject(pParameters[i]));
					}
				}
			}
			
			call.execute();
			
			for (int i = 0; pParameters != null && i < pParameters.length; i++)
			{
				if (pParameters[i] instanceof AbstractParam)
				{
					apParam = (AbstractParam)pParameters[i];
					
					type = apParam.getType();
					
					if (type == ParameterType.Out || type == ParameterType.InOut)
					{
						apParam.setValue(call.getObject(i + 1));
					}
				}
			}
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("executeProcedure failed! - " + sqlStatement, formatSQLException(sqlException));
		}
		finally 
		{
			try
			{
				call.close();
			}
			catch (Throwable pThrowable)
			{
			    // Do nothing!	
			}
		}
	}	
	
	/**
	 * Executes a DB function with the specified parameters and return the result.
	 * 
	 * @param pFunctionName the function (optional with package) name. 
	 * @param pReturnType the return SQL Type (see {@link Types}
	 * @param pParameters the parameters to use with the correct and corresponding java type.
	 * @return the result of the DB function call.
	 * @throws DataSourceException	if the call failed.
	 */
	public Object executeFunction(String pFunctionName, int pReturnType, Object...pParameters) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		StringBuilder sqlStatement = new StringBuilder("{ ? = call ");
		sqlStatement.append(pFunctionName);
		
		if (pParameters.length > 0)
		{
			sqlStatement.append("(");
			for (int i = 0; i < pParameters.length; i++)
			{
				if (i > 0)
				{
					sqlStatement.append(", ");			
				}
				sqlStatement.append("?");			
			}
			sqlStatement.append(")");
		}
		sqlStatement.append(" }");
		
		CallableStatement call = null;
		try
		{
			logger.debug("executeFunction -> ", sqlStatement);
			
			call = cConnection.prepareCall(translateQuotes(sqlStatement.toString()));
			call.registerOutParameter(1, pReturnType);
			
			AbstractParam apParam;
			
			ParameterType type;
			
			for (int i = 0; pParameters != null && i < pParameters.length; i++)
			{
				if (pParameters[i] == null)
				{
					call.setNull(i + 2, Types.VARCHAR);
				}
				else
				{
					if (pParameters[i] instanceof AbstractParam)
					{
						apParam = (AbstractParam)pParameters[i];
						
						type = apParam.getType();
						
						if (type == ParameterType.Out || type == ParameterType.InOut)
						{
							call.registerOutParameter(i + 2, apParam.getSqlType());
						}
						
						if (apParam.getValue() == null)
						{
							call.setNull(i + 2, apParam.getSqlType());							
						}
						else
						{
							call.setObject(i + 2, convertValueToDatabaseSpecificObject(apParam.getValue()), apParam.getSqlType());
						}
					}
					else
					{
						call.setObject(i + 2, convertValueToDatabaseSpecificObject(pParameters[i]));
					}
				}
			}
			
			if (call.execute())
			{
				call.getResultSet().close();
			}
			
			Object oResult = call.getObject(1);
			
			for (int i = 0; pParameters != null && i < pParameters.length; i++)
			{
				if (pParameters[i] instanceof AbstractParam)
				{
					apParam = (AbstractParam)pParameters[i];
					
					type = apParam.getType();
					
					if (type == ParameterType.Out || type == ParameterType.InOut)
					{
						apParam.setValue(call.getObject(i + 2));
					}
				}
			}
			
			return oResult;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("executeFunction failed! - " + sqlStatement, formatSQLException(sqlException));
		}
		finally 
		{
			try
			{
				call.close();
			}
			catch (Throwable pThrowable)
			{
			    // Do nothing!	
			}
		}
	}

	/**
	 * Executes a DDL or DML Statement.
	 * 
	 * @param pStatement the statement.
	 * @throws SQLException	if an error occur during execution.
	 */
	public void executeStatement(String pStatement) throws SQLException
	{
		Statement stmt = null;
		
		try
		{
			stmt = getConnection().createStatement();
			
			String sStmt = translateQuotes(pStatement);
			
			logger.debug("executeStatement -> ",  sStmt);
			
			if (stmt.execute(sStmt))
			{
				stmt.getResultSet().close();
			}
		}
		finally
		{
    		if (stmt != null)
    		{
    			try
    			{
    				stmt.close();
    			}
    			catch (Exception e)
    			{
    				//nothing to be done
    			}
    		}
		}
	}
	
	/**
	 * Executes a SQL command as prepared statement.
	 * 
	 * @param pStatement the statement with or without parameters
	 * @param pParameters the parameters to use
	 * @return the first parameter from the result set, for every row, or <code>null</code> if the command returned no
	 *         result
	 * @throws SQLException if an error occurs during execution
	 */
	public List<Object> executeSql(String pStatement, Object... pParameters) throws SQLException
	{
		PreparedStatement psStmt = null;
		
		try
		{
			String sStmt = translateQuotes(pStatement);
			
			psStmt = getConnection().prepareStatement(sStmt);
			
			if (pParameters != null)
			{
				for (int i = 0; i < pParameters.length; i++)
				{
					if (pParameters[i] == null)
					{
						psStmt.setNull(i + 1, Types.VARCHAR);
					}
					else
					{
						psStmt.setObject(i + 1, convertValueToDatabaseSpecificObject(pParameters[i]));
					}
				}
			}
			
			logger.debug("executeSql -> ", pStatement);
			
			if (psStmt.execute())
			{
				ResultSet res = psStmt.getResultSet();
				
				if (res.next())
				{
					List<Object> liResult = new ArrayUtil<Object>();

					do
					{
						liResult.add(res.getObject(1));
					}
					while (res.next());
					
					return liResult;
				}
				
				if (res != null)
				{
					try
					{
						res.close();
					}
					catch (Exception e)
					{
						//nothing to be done
					}
				}
			}
			
			return null;
		}
		finally
		{
    		if (psStmt != null)
    		{
    			try
    			{
    				psStmt.close();
    			}
    			catch (Exception e)
    			{
    				//nothing to be done
    			}
    		}
		}
	}
	
	/**
	 * Return a <code>PreparedStatement</code> for the given SQL statement.<br>
	 * 
	 * @param pSqlStatement
	 *            the SQL statement to prepare
	 * @param pReturnKey
	 *            if the generated key in insert statements should returned
	 * @return a <code>PreparedStatement</code> for the given SQL statement.
	 * @throws DataSourceException
	 *             if the statement couldn't prepared.
	 */
	public PreparedStatement getPreparedStatement(String pSqlStatement, boolean pReturnKey) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAccess is not open!");
		}

		PreparedStatement psSqlStatement;
		
		pSqlStatement = translateQuotes(pSqlStatement.toString());
		try
		{
			if (pReturnKey)
			{
				try 
				{
					psSqlStatement = cConnection.prepareStatement(pSqlStatement, Statement.RETURN_GENERATED_KEYS);						
				}
				catch (SQLException sqlException)
				{
					psSqlStatement = cConnection.prepareStatement(pSqlStatement);
				}					
			}
			else
			{
				psSqlStatement = cConnection.prepareStatement(pSqlStatement);
			}
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("PrepareStatement failed! - " + pSqlStatement, formatSQLException(sqlException));
		}

		return psSqlStatement;
	}	

	/**
	 * Calls <code>executesUpdate()</code> for specified <code>PreparedStatement</code> 
	 * and returns the number of updated rows.
	 * 
	 * @param pSqlStatement
	 *            the <code>PreparedStatement</code> to use
	 * @return number of updated rows.
	 * @throws DataSourceException
	 *             if the statement couldn't updated.
	 */
	public int executeUpdate(PreparedStatement pSqlStatement) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAcces is not open!");
		}

		try
		{
			return pSqlStatement.executeUpdate();
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Execute update failed!", formatSQLException(sqlException));
		}
	}
	
	/**
	 * Adds the SQL Error Code into the message of the SQL Exception.
	 * 
	 * @param pSqlException		the SQL Exception to use.
	 * @return the SQLException with the modified error message with SQL Error Code.
	 */
	protected SQLException formatSQLException(SQLException pSqlException)
	{
		return formatSQLException(pSqlException, pSqlException.getMessage(), "" + pSqlException.getErrorCode());
	}

	/**
	 * Adds the SQL Error Code into the message of the SQL Exception.
	 * 
	 * @param pSqlException the SQL Exception to use.
	 * @param pMessage the message to use
	 * @param pCode the detected error code
	 * @return the SQLException with the modified error message with SQL Error Code.
	 */
	protected SQLException formatSQLException(SQLException pSqlException, String pMessage, String pCode)
	{
		if (pMessage.indexOf(pCode) < 0)
		{
			String sVendor = getClass().getSimpleName();
			int    index   = sVendor.indexOf("DBAccess");
			
			if (index >= 0)
			{
				sVendor = sVendor.substring(0, index);
			}
			
			SQLException sqleNew = new SQLException(sVendor + "-" + pCode + ": " + pMessage);
			sqleNew.setStackTrace(pSqlException.getStackTrace());
			sqleNew.initCause(pSqlException.getCause());
			
			return sqleNew;
		}
		
		return pSqlException;
	}
	
	/**
	 * It initialize the select for a specified storage unit and return the SELECT statement.<br>
	 * It doesn't add the ORDER BY clause.<br>
	 * 
	 * @param pFilter
	 *            the <code>Filter</code> to use
	 * @param pFromClause
	 * 			  the list of query tables to use in the SELECT statement.
	 * @param pQueryColumns
	 * 			  the list of query columns to use in the SELECT statement.
	 * @param pBeforeQueryColumns
	 * 			  the string to place in the SELECT statement between the SELECT and the first query column.
	 * @param pWhereClause
	 * 			  the string to place in the SELECT statement after the last WHERE condition from the
	 * 			  Filter or MasterReference (Master-Detail Condition).
	 * @param pAfterWhereClause
	 * 			  the string to place in the SELECT statement after the WHERE clause and before the ORDER BY clause.
	 * @param pServerMetaData			
	 * 			  the MetaDataColumn array to use.
     * @return the SELECT statement as String.
     * @throws DataSourceException if it's not possible to build the select statement in fact of missing elements    
	 */	
	public String getSelectStatement(ICondition pFilter, 
									 String pFromClause, String[] pQueryColumns,
									 String pBeforeQueryColumns, String pWhereClause, String pAfterWhereClause,
									 ServerMetaData pServerMetaData) throws DataSourceException
	{
		if (pFromClause == null)
		{
			throw new DataSourceException("Missing FROM clause!");
		}
		
		// add select column names to select
		StringBuilder sSelectStatement = new StringBuilder("SELECT ");

		if (pBeforeQueryColumns != null)
		{
			sSelectStatement.append(pBeforeQueryColumns);
			sSelectStatement.append(" ");
		}

		if (pQueryColumns != null && pQueryColumns.length > 0 && pQueryColumns[0].length() > 0)
		{
			for (int i = 0; i < pQueryColumns.length; i++)
			{
				if (i > 0)
				{
					sSelectStatement.append(", ");
				}
				sSelectStatement.append(pQueryColumns[i]);
			}
		}
		else
		{
			sSelectStatement.append("*");
		}

		sSelectStatement.append(" FROM ");

		sSelectStatement.append(pFromClause);

		// add Filter
		if (pFilter != null)
		{
			sSelectStatement.append(getWHEREClause(pFilter, pServerMetaData, true));
		}

		if (pWhereClause != null)
		{
			if (pFilter == null)
			{
				sSelectStatement.append(" WHERE ");
			}
			else
			{
				sSelectStatement.append(" AND ");
			}
			sSelectStatement.append(" ");
			sSelectStatement.append(pWhereClause);
			sSelectStatement.append(" ");
		}

		if (pAfterWhereClause != null)
		{
			if (pWhereClause != null || pFilter != null)
			{
				sSelectStatement.append(" AND ");
			}
			sSelectStatement.append(" ");
			sSelectStatement.append(pAfterWhereClause);
			sSelectStatement.append(" ");
		}

		return sSelectStatement.toString();
	} 

	/**
	 * Returns the database specific statement to lock the specified row in the database.
	 *  
	 * @param pWritebackTable	the table to use.
	 * @param pPKFilter			the PK filter with the values to use.
	 * @param pServerMetaData	the MetaDataColumn array to use.
	 * @return the database specific statement to lock the specified row in the database.
	 * @throws DataSourceException if some parts are missing for the statement
	 */
	public String getDatabaseSpecificLockStatement(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{
		if (pWritebackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}
		
		StringBuilder sbfSelectForUpdate = new StringBuilder("SELECT * FROM ");
		sbfSelectForUpdate.append(pWritebackTable);
		sbfSelectForUpdate.append(getWHEREClause(pPKFilter, pServerMetaData, false));						
		sbfSelectForUpdate.append(" FOR UPDATE");						
		return sbfSelectForUpdate.toString();
	}
		
	/**
	 * Returns the newly inserted row from an Database specific insert statement. <br>
	 * Database specific derivations of DBAcces need to implement it database specific. 
	 * 
	 * @param pWritebackTable	the table to use for the insert
	 * @param pInsertStatement	the SQL Statement to use for the insert
	 * @param pServerMetaData	the meta data to use.
	 * @param pNewDataRow		the new row (Object[]) with the values to insert
	 * @param pDummyColumn		<code>null</code>, if all writeable columns are null, but for a correct INSERT it have
	 *                          to be minimum one column to use in the syntax.
	 * @return the newly inserted row from an Oracle Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert to the storage
	 */		
	public Object[] insertDatabaseSpecific(String pWritebackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
            Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
    {
		return insertAnsiSQL(pWritebackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
    }
	
	/**
	 * Returns if this Database specific supports generated keys. Because of Derby bug!
	 * @return if this Database specific supports generated keys.
	 */
	public boolean supportsGetGeneratedKeys()
	{
		try
		{
			return cConnection.getMetaData().supportsGetGeneratedKeys();
		}
		catch (SQLException sqlException)
		{
			return false;
		}
	}

	/**
	 * Returns the newly inserted row from a Ansi SQL Database. <br>
	 * It uses getGeneratedKeys to get the primary key values back from the database. Its recommend that
	 * the jdbc driver of the database support that clean.
	 * 
	 * @param pWritebackTable	the table to use for the insert
	 * @param pInsertStatement	the SQL Statement to use for the insert
	 * @param pServerMetaData			the meta data to use.
	 * @param pNewDataRow		the new row (Object[]) with the values to insert
	 * @param pDummyColumn		true, if all writeable columns are null, but for a correct INSERT it have
	 *                          to be minimum one column to use in the syntax.
	 * @return the newly inserted row from a Ansi SQL Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert to the storage
	 */
	public Object[] insertAnsiSQL(String pWritebackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
									Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
	{
		// insert DataRow to database
		boolean bSupportGeneratedKeys = supportsGetGeneratedKeys();

		PreparedStatement psInsert = getPreparedStatement(pInsertStatement, bSupportGeneratedKeys);

		ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
		
		if (pDummyColumn == null)
		{
			setColumnsToStore(psInsert, cmdServerColumnMetaData, iaWriteables, pNewDataRow, null);
		}
		else
		{
			// set null value for it!
			try
			{
				for (int i = 0; i < cmdServerColumnMetaData.length; i++)
				{
					if (cmdServerColumnMetaData[i].getColumnName().getQuotedName().equals(pDummyColumn))
					{
						psInsert.setObject(1, null, cmdServerColumnMetaData[i].getSQLType());
						break;
					}
				}
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("Insert failed! - " + pInsertStatement, formatSQLException(sqlException));
			}				
		}
			
	    ResultSet rsPK = null;
		try
		{		
			if (executeUpdate(psInsert) == 1)
			{
				if (bSupportGeneratedKeys)
				{
					// Return GeneratedKeys (typical used in PK)
					try 
					{
					    rsPK = psInsert.getGeneratedKeys();
					    if (rsPK.next()) 
					    {
						    String[] pPKColumns = pServerMetaData.getPrimaryKeyColumnNames();
					    	for (int i = 0, anz = rsPK.getMetaData().getColumnCount(); i < anz && pPKColumns != null && i < pPKColumns.length; i++)
					    	{
								for (int j = 0; j < cmdServerColumnMetaData.length; j++)
								{
									if (cmdServerColumnMetaData[j].getName().equals(pPKColumns[i]))
									{
										// #440 - DBAccess.insert return null if no auto increment column in table 
										Object value = rsPK.getObject(i + 1);
										if (value != null)
										{
											pNewDataRow[j] = value;
										}
										break;
									}
								}
					    	}
					    }
					}
					catch (SQLException sqlException)
					{
						throw new DataSourceException("The generated keys couldn't read! - " + pInsertStatement, formatSQLException(sqlException));
					}
				}
				return pNewDataRow;
			}
			throw new DataSourceException("Insert failed! - Result row count != 1" + pInsertStatement);
		}
		finally
		{
			try
			{
				if (rsPK != null)
				{
					rsPK.close();
				}
				if (psInsert != null)
				{
					psInsert.close();
				}
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(sqlException));
			}			
		}
	}
	
	/**
	 * Returns the meta data information for the specified query, and configures all columns with defaults.
	 * 
	 * @param pBeforeQueryColumns	the before query columns
	 * @param pQueryColumns			the query columns	
	 * @param pFromClause			the from clause with query tables and join definitions
	 * @param pWhereClause			the last where condition in query
	 * @param pAfterWhereClause		the after where clause in query
	 * @param pWritebackTable		the write back table to use for the isWriteable() state (Optional)
	 * @param pWritebackColumns		the write back columns to use for the isWriteable() state (Optional)
 	 * @return the meta data for the specified query, and initials all columns.
	 * @throws DataSourceException 
	 *            if an <code>Exception</code> occur during getting the meta data or 
	 *            if the storage is not opened or 
	 *            if one columns SQL type is not supported
	 */
	public ColumnMetaDataInfo getColumnMetaData(String pFromClause, 
										     	String[] pQueryColumns,
										     	String pBeforeQueryColumns, 
										     	String pWhereClause, 
										     	String pAfterWhereClause,
										     	String pWritebackTable,
										     	String[] pWritebackColumns) throws DataSourceException
	{		
		PreparedStatement psQueryMetaData = null;
		ResultSet rsQueryMetaData = null;
		String sMetaDataQuery = getSelectStatement(null, pFromClause, pQueryColumns, 
                                                   pBeforeQueryColumns, pWhereClause, pAfterWhereClause, 
                                                   null);
			
		try
		{
			long lMillis = System.currentTimeMillis();
			
			// get Columns
			psQueryMetaData = getPreparedStatement(sMetaDataQuery, false);
			rsQueryMetaData = psQueryMetaData.executeQuery();
			ResultSetMetaData rsMetaData = rsQueryMetaData.getMetaData();
			
			ColumnMetaDataInfo mdi = new ColumnMetaDataInfo();
			
			String sCatalog = rsMetaData.getCatalogName(1);
			String sSchema  = rsMetaData.getSchemaName(1);
			
			// #136 - Mysql PK refetch not working -> extract the given table without schema. 
//			String sTable = rsMetaData.getTableName(1); // Table name is alias, if alias is set

			if (sCatalog != null && sCatalog.trim().length() == 0)
			{
				sCatalog = null;
			}
			
			if (sSchema != null && sSchema.trim().length() == 0)
			{
				sSchema = null;
			}
			
			String[] sSchemaTable = splitSchemaTable(pFromClause);
			
			if (sSchema == null)
			{
				sSchema = sSchemaTable[0];
			}
			
			String sTable = sSchemaTable[1];
			
			if (sTable != null && sTable.endsWith(" m "))
			{
				sTable = sTable.substring(0, sTable.length() - 3);
			}
			
			mdi.setCatalog(sCatalog != null ? sCatalog.trim() : null);
			mdi.setSchema(sSchema != null ? sSchema.trim() : sDefaultSchema);
			mdi.setTable(sTable != null ? sTable.trim() : null);
			
			ArrayUtil<ServerColumnMetaData> auCmd = new ArrayUtil<ServerColumnMetaData>();
			ArrayUtil<String> auColumnNames = new ArrayUtil<String>();
			
			// set every ColumnMetaData
			for (int i = 1; i <= rsMetaData.getColumnCount(); i++)
			{
				Name nColumnName = new Name(rsMetaData.getColumnName(i), quote(rsMetaData.getColumnName(i)));
				
				if (auColumnNames.indexOf(nColumnName.getName()) >= 0)
				{
					throw new DataSourceException("Duplicate definition of '" + nColumnName.getName() + 
							"' in DBStorage " + sMetaDataQuery + "!");
				}
				ServerColumnMetaData cd = new ServerColumnMetaData(nColumnName);
								
				//set real DB column name, which is specified from the developer to use for query, filter and sort (only Query side)
				if (pQueryColumns != null)
				{
					String sQueryColumn = pQueryColumns[i - 1].toLowerCase();
					String sName = cd.getName().toLowerCase();
					int iLen = sQueryColumn.length() - sName.length() - 1;
					
					if (sQueryColumn.endsWith(sName) 
					    && iLen > 0
						&& Character.isWhitespace(sQueryColumn.charAt(iLen)))
					{
						sQueryColumn = pQueryColumns[i - 1].substring(0, iLen).trim(); 
						iLen = sQueryColumn.length() - 3;
						
						if (sQueryColumn.endsWith("as") 
						    && iLen > 0
							&& Character.isWhitespace(sQueryColumn.charAt(iLen)))
						{
							sQueryColumn = pQueryColumns[i - 1].substring(0, iLen).trim(); 
						}
					}
					else
					{
						sQueryColumn = pQueryColumns[i - 1];
					}
					cd.setRealQueryColumnName(sQueryColumn);
					logger.debug("Name=", cd.getName(), "==", sQueryColumn);
				}
				else
				{
					cd.setRealQueryColumnName(cd.getColumnName().getRealName());
				}
				
				if (rsMetaData.isNullable(i) == ResultSetMetaData.columnNoNulls)
				{
					cd.setNullable(false);
				}
				else
				{
					cd.setNullable(true);
				}
				// #544 - In databases with default lower case column names, the default labels in the RowDefinition are wrong 
				// setLabel removed!
				cd.setAutoIncrement(rsMetaData.isAutoIncrement(i));
				
				int iColumnType = rsMetaData.getColumnType(i);
				
				cd.setSQLType(iColumnType);
				cd.setSQLTypeName(rsMetaData.getColumnTypeName(i));

				if (!setDatabaseSpecificType(rsMetaData, i, cd))
				{
					switch(iColumnType)
					{
						case NCHAR:
						case NVARCHAR:
						case LONGNVARCHAR:
						case Types.CHAR:
		                case Types.VARCHAR:
		                case Types.LONGVARCHAR:
		                case Types.CLOB:
		                	cd.setDataType(StringDataType.TYPE_IDENTIFIER);
		                	if (rsMetaData.getPrecision(i) > 0)
		                	{
		                		cd.setPrecision(rsMetaData.getPrecision(i));
		                	}
		                	else
		                	{
		                		cd.setPrecision(Integer.MAX_VALUE);
		                	}
		                	break;
	
		                case Types.BIT:
						case Types.BOOLEAN:
		                	cd.setDataType(BooleanDataType.TYPE_IDENTIFIER);
		                	break;
		                	
		                case Types.FLOAT:
		                case Types.REAL:
		                case Types.DOUBLE:
		                	cd.setDataType(BigDecimalDataType.TYPE_IDENTIFIER);
		                	cd.setPrecision(0);
		                	cd.setScale(-1);
		                	cd.setSigned(rsMetaData.isSigned(i)); 
		                	break;
		                	
		                case Types.TINYINT:
		                case Types.SMALLINT:
		                case Types.INTEGER:
		                case Types.BIGINT:
		                case Types.NUMERIC:
		                case Types.DECIMAL:
		                	cd.setDataType(BigDecimalDataType.TYPE_IDENTIFIER);
		                	if (rsMetaData.getPrecision(i) <= 0 // none, use decimal without any precision and scale -> unlimited!
		                			|| rsMetaData.getPrecision(i) > 30) // over 30 , we use decimal without any precision and scale -> unlimited!
		                	{
			                	cd.setPrecision(0);
			                	cd.setScale(-1);
		                	}
		                	else
		                	{
			                	cd.setPrecision(rsMetaData.getPrecision(i));
			                	cd.setScale(rsMetaData.getScale(i));
		                	}
		                	cd.setSigned(rsMetaData.isSigned(i)); 
		                	break;
		                	
		                case Types.DATE:
		                case Types.TIME:
		                case Types.TIMESTAMP:
		                	cd.setDataType(TimestampDataType.TYPE_IDENTIFIER);
		                	break;
		                	
		                case Types.BLOB:
		                case Types.BINARY:
		                case Types.VARBINARY:
		                case Types.LONGVARBINARY:
		                	cd.setDataType(BinaryDataType.TYPE_IDENTIFIER);
		                	if (rsMetaData.getPrecision(i) > 0)
		                	{
		                		cd.setPrecision(rsMetaData.getPrecision(i));
		                	}
		                	else
		                	{
		                		cd.setPrecision(Integer.MAX_VALUE);
		                	}
		                	
		                	break;
		                	
		                default:
			    			throw new DataSourceException(cd.getName() + " :: SQL Type '" + iColumnType + "' is not support!");	                
					}
				}
				
				cd.setWritable(false);
				
				if (pWritebackTable != null)
				{
					if (pWritebackColumns != null)
					{
						String sRealName = cd.getColumnName().getRealName();
						
						// set the writeable columns with the pWritebackColumns
						for (int j = 0; j < pWritebackColumns.length; j++)
						{
							if (pWritebackColumns[j].toLowerCase().equals(sRealName.toLowerCase()))
							{
								cd.setWritable(true);
								break;
							}
						}
					}
				}
							
				auCmd.add(cd);		
				auColumnNames.add(cd.getName());
			}
			
			logger.debug("getMetaData(", sMetaDataQuery, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
			
			mdi.setColumnMetaData(auCmd.toArray(new ServerColumnMetaData[auCmd.size()]));
			
			return mdi;
		}
		catch (SQLException sqlException)
		{    		
			throw new DataSourceException("Meta data couldn't load from database! - " + 
										  sMetaDataQuery, formatSQLException(sqlException));
		}
		finally
		{
			try
			{
				
	    		if (rsQueryMetaData != null)
	    		{
	    			rsQueryMetaData.close();
	    		}
	    		if (psQueryMetaData != null)
	    		{
	    			psQueryMetaData.close();
	    		}
			}
			catch (SQLException e)
			{
				throw new DataSourceException("Jdbc statement close failed", formatSQLException(e));
			}			
		}
	}

	/**
	 * Returns the WHERE clause for a UPDATE, DELETE or SELECT statement specified with
	 * a <code>Filter</code>.
	 * 
	 * @param pFilter       	the <code>Filter</code> to use
	 * @param pServerMetaData	the <code>MetaData</code> to use
	 * @param pUsePrefix		<code>true</code> to use the prefixed column (real column name) and <code>false</code> to
	 *                      	ignore prefixes (the simple name of the column) 
	 * @throws DataSourceException if columns not existing
	 * @return the WHERE clause for a UPDATE, DELETE or SELECT statement specified with
	 *         a <code>Filter</code>.
	 */
	public String getWHEREClause(ICondition pFilter, ServerMetaData pServerMetaData, boolean pUsePrefix) throws DataSourceException
	{
		String sCondion = getSQL(pFilter, pServerMetaData, pUsePrefix);
		if (sCondion !=  null && sCondion.length() > 0)
		{
			return " WHERE " + sCondion;
		}
		return "";
	}
	
	/**
	 * Sets all <code>Filter</code> parameter values into the <code>PreparedStatement</code>.
	 * 
	 * @param iStartParameterIndex 
	 *            the start index for the parameters to set.
	 * @param pStatement 
	 * 			  the <code>PreparedStatement</code> to initialize
	 * @param pFilter 
	 *            the <code>Filter</code> to get the values
	 * @throws DataSourceException     
	 *            if the values can't set into the <code>PreparedStatement</code>     
	 */
	protected void setFilterParameter(int iStartParameterIndex, PreparedStatement pStatement, ICondition pFilter) throws DataSourceException
	{				
		Object[] oaParameters = getParameter(pFilter);
		
		int j = 0;
		for (int i = 0; i < oaParameters.length; i++)
		{
			try
			{
				if (oaParameters[i] != null)
				{
					pStatement.setObject(iStartParameterIndex + j, convertValueToDatabaseSpecificObject(oaParameters[i]));
					
					j++;
				}
			}
			catch (SQLException sqlException)
			{
				throw new DataSourceException("Set value into PreparedStatement failed!",
											  formatSQLException(sqlException));
			}			
		}
	}
	
	/**
	 * Sets the values of all changed columns to store from the value Object[]s into the 
	 * <code>PreparedStatement</code> and returns the last used parameter index.
	 * 
	 * @param pInsert 		  		the <code>PreparedStatement</code> to initialize
	 * @param pServerColumnMetaData	the column meta data to use.
	 * @param iaWriteables			the writable columns as int index array
	 * @param pNew             		the new values Object[]
	 * @param pOld  	        	the old values Object[]
	 * @throws DataSourceException	if the values can't set into the <code>PreparedStatement</code>     
	 * @return the last used parameter index of the <code>PreparedStatement</code>.           
	 */
	protected int setColumnsToStore(PreparedStatement pInsert, ServerColumnMetaData[] pServerColumnMetaData, 
			                        int[] iaWriteables, Object[] pNew, Object[] pOld) throws DataSourceException
	{
		int i = 1;
		for (int j = 0; j < iaWriteables.length; j++)
		{
			int k = iaWriteables[j];
	        if (pOld == null && pNew[k] != null
	        	|| pOld != null 
	        	   && pServerColumnMetaData[k].getDataType().compareTo(pNew[k], pOld[k]) != 0)
	        {
				try
				{
					if (pNew[k] == null)
					{
						pInsert.setNull(i, pServerColumnMetaData[k].getSQLType());
					}
					else
					{
						pInsert.setObject(i, convertValueToDatabaseSpecificObject(pNew[k]));
					}
					i++;
				}
				catch (SQLException sqlException)
				{
					throw new DataSourceException("Set value into PreparedStatement failed!",
												  formatSQLException(sqlException));
				}
			}
		}	
		return --i;
	}

	/**
	 * Gets all default column values of a specific table.
	 *  
	 * @param pCatalog the catalog name
	 * @param pSchema the schema name
	 * @param pTable the table name
	 * @return a {@link Hashtable} with the column name as key and the default value as value. It only contains columns
	 *         with a default value
	 * @throws DataSourceException if the database access throws an exception
	 */
	public Hashtable<String, Object> getDefaultValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		Hashtable<String, Object> htDefaults = new Hashtable<String, Object>();

		ResultSet res = null;
		
		try
		{
			res = cConnection.getMetaData().getColumns(pCatalog, pSchema, pTable, null);
			
			Object objValue;
			
			String sValue;
			String sColumnName;
			
			while (res.next())
			{
				sValue = res.getString("COLUMN_DEF");

				if (sValue != null)
				{
					sColumnName = res.getString("COLUMN_NAME");
					
					try
					{
						objValue = translateDefaultValue(sColumnName, res.getInt("DATA_TYPE"), sValue.trim());

						if (objValue != null)
						{
							htDefaults.put(sColumnName, objValue);
						}
					}
					catch (Exception e)
					{
						//no default value
						logger.debug(sValue, e);
					}
				}
			}
			
			return htDefaults;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Get default values failed!", formatSQLException(sqlException));
		}
		finally
		{
			try
			{
				res.close();
			}
			catch (Exception e)
			{
				//nothing to be done
			}
		}
	}
	
	/**
	 * Translates a default value from a column to the datatype object.
	 * 
	 * @param pColumnName the column name to translate
	 * @param pDataType the datatype of the column
	 * @param pDefaultValue the original default value from the database
	 * @return the default value with the datatype of the column or <code>null</code> if the default value is not valid
	 * @throws Exception if the type translation causes an error or the datatype is not supported
	 */
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		return translateValue(pDataType, pDefaultValue);
	}
	
	/**
	 * Translates an object value to the datatype object.
	 * 
	 * @param pDataType the datatype of the column
	 * @param pValue the value from the database
	 * @return the value with the specified datatype or <code>null</code> if the value is not valid
	 * @throws Exception if the type translation causes an error or the datatype is not supported
	 */
	protected Object translateValue(int pDataType, String pValue) throws Exception
	{
		if (pValue == null || pValue.equalsIgnoreCase("null"))
		{
			return null;
		}
		
		switch(pDataType)
		{
			case NCHAR:
			case NVARCHAR:
			case LONGNVARCHAR:
			case Types.CHAR:
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
            case Types.CLOB:
            	return pValue;

            case Types.BIT:
			case Types.BOOLEAN:
				return Boolean.valueOf(pValue);
            	
            case Types.FLOAT:
            case Types.REAL:
            case Types.DOUBLE:
            case Types.TINYINT:
            case Types.SMALLINT:
            case Types.INTEGER:
            case Types.BIGINT:
            case Types.NUMERIC:
            case Types.DECIMAL:

            	return new BigDecimal(pValue);
            	
            case Types.DATE:
            case Types.TIME:
            case Types.TIMESTAMP:
            	
            	if (pValue.startsWith("0000-00-00"))
            	{
            		return null;
            	}
            	
            	return new Timestamp(((Date)dateUtil.parse(pValue)).getTime());
            	
            case Types.BLOB:
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY:
            	return null;
            	
            default:
    			throw new DataSourceException("SQL Type '" + pDataType + "' is not support!");	                
		}
	}
	
	/**
	 * Gets the allowed values from a specific table. The allowed values are defined through check constraints
	 * or other table related restrictions.
	 * 
	 * @param pCatalog the catalog name
	 * @param pSchema the schema name
	 * @param pTable the table to check
	 * @return a {@link Hashtable} with a column name as key and the allowed values as array of {@link Object}s or 
	 *         <code>null</code> if there are no allowed values
	 * @throws DataSourceException if the database access throws an exception          
	 */
	public Hashtable<String, Object[]> getAllowedValues(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		return null;
	}
	
	/**
	 * Gets the default allowed values for a given column from a given table. This method will be called if no other 
	 * default values for the given column are available.
	 * 
	 * @param pCatalog the catalog name
	 * @param pSchema the schema name
	 * @param pTable the table to check
	 * @param pMetaData the column meta data
	 * @return the default allowed values or <code>null</code> if there are no default values
	 */
	public Object[] getDefaultAllowedValues(String pCatalog, String pSchema, String pTable, ServerColumnMetaData pMetaData)
	{
		if (pMetaData.getDataType().getTypeIdentifier() == BooleanDataType.TYPE_IDENTIFIER)
		{
			return new Boolean[] {Boolean.TRUE, Boolean.FALSE};
		}
		
		return null;
	}
	
	/**
	 * Returns the database specific statement to lock the specified row in the database.
	 *  
	 * @param pSelectStatement	the select statement to execute.
	 * @param pPKFilter			the PK filter with the values to use.
	 * @param pMaxRows          the maximum number of allowed rows. A negative number means no limit. 
	 * @return the number of affected rows
	 * @throws DataSourceException if some parts are missing for the statement
	 */
	private int getRowCount(String pSelectStatement, ICondition pPKFilter, int pMaxRows) throws DataSourceException
	{
		long lMillis = System.currentTimeMillis();
		
		// update DataRow into database
		PreparedStatement psSelect = getPreparedStatement(pSelectStatement.toString(), false);

		// set WHERE Parameter with Values from PK of the new DataRow
		setFilterParameter(1, psSelect, pPKFilter);

		ResultSet res = null;
		
		try
		{
			res = psSelect.executeQuery();
			
			logger.debug("select (", pSelectStatement, ",...) in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");

			int iCount = 0;
			
			while (res.next())
			{
				iCount++;
				
				if (pMaxRows >= 0 && iCount > pMaxRows)
				{
					throw new DataSourceException("Too many rows found! " + pSelectStatement);
				}
			}
			
			// if iCount == 0, do nothing!!!
			
			return iCount;
		}
		catch (SQLException se)
		{
			throw new DataSourceException("Execute statement failed! " + pSelectStatement, formatSQLException(se));
		}
		finally
		{
			if (res != null)
			{
				try
				{
					res.close();
				}
				catch (SQLException se)
				{
					throw new DataSourceException("Jdbc statement close failed", formatSQLException(se));
				}
			}
		}
	}
	
	/**
	 * Returns the ANSI SQL String for this <code>ICondition</code>.
	 * 
	 * @param cCondition		the Condition to use.
	 * @param pServerMetaData	the MetaDataColumn array to use.
	 * @param pUsePrefix		<code>true</code> to use the prefixed column (real column name) and <code>false</code> to
	 *                      	ignore prefixes (the simple name of the column)
	 * @throws DataSourceException if columns not existing
	 * @return the ANSI SQL String for this <code>ICondition</code>.
	 */
	String getSQL(ICondition cCondition, ServerMetaData pServerMetaData, boolean pUsePrefix) throws DataSourceException
	{
		StringBuilder result = new StringBuilder();
		
		if (cCondition instanceof CompareCondition)
		{
			CompareCondition cCompare = (CompareCondition)cCondition;
			Object           oValue   = cCompare.getValue();
			
			if (!cCompare.isIgnoreNull() || oValue != null)
			{
				String sColumnName = null;
				if (pServerMetaData != null)
				{
					try
					{
						if (pUsePrefix)
						{
							sColumnName = pServerMetaData.getServerColumnMetaData(cCompare.getColumnName()).getRealQueryColumnName();
						}
						else
						{
							sColumnName = pServerMetaData.getServerColumnMetaData(cCompare.getColumnName()).getColumnName().getQuotedName();
						}
					}
					catch (ModelException modelException)
					{
						// Column doesn't exit in MetaData, then use the column name of the Condition itself.
					}
				}
				if (sColumnName == null)
				{
					sColumnName = cCompare.getColumnName();
				}
				
				if ((cCompare instanceof LikeReverse || cCompare instanceof LikeReverseIgnoreCase)
					&& cCompare.getValue() != null)
				{
					if (cCompare instanceof LikeReverse)
					{
						result.append(createWhereParam(pServerMetaData, cCompare));
					}
					else if (cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append("UPPER(");
						result.append(createWhereParam(pServerMetaData, cCompare));
						result.append(")");
					}						
				}
				else if (cCompare instanceof LikeIgnoreCase)
				{
					result.append("UPPER(");
					result.append(createWhereColumn(pServerMetaData, cCompare, sColumnName));
					result.append(")");
				}						
				else 
				{
					result.append(createWhereColumn(pServerMetaData, cCompare, sColumnName));
				}						
				
				result.append(' ');
				
				if (cCompare.getValue() == null)
				{
					result.append("IS NULL");
				}
				else
				{
					if (cCompare instanceof Equals)
					{
						result.append("=");
					}
					else if (cCompare instanceof LikeIgnoreCase || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append("LIKE UPPER(");
					}						
					else if (cCompare instanceof Like
							|| cCompare instanceof LikeReverse)
					{
						result.append("LIKE");
					}						
					else if (cCompare instanceof Greater)
					{
						result.append(">");
					}						
					else if (cCompare instanceof GreaterEquals)
					{
						result.append(">=");
					}						
					else if (cCompare instanceof Less)
					{
						result.append("<");
					}						
					else if (cCompare instanceof LessEquals)
					{
						result.append("<=");
					}		
					
					result.append(' ');						

					if (cCompare instanceof LikeReverse || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append(createReplace(createReplace(createWhereColumn(pServerMetaData, cCompare, sColumnName), "*", "%"), "?", "_"));						
					}
					else
					{
						result.append(createWhereParam(pServerMetaData, cCompare));
					}
					
					if (cCompare instanceof LikeIgnoreCase || cCompare instanceof LikeReverseIgnoreCase)
					{
						result.append(")");
					}						
				}
			}
		}
		else if (cCondition instanceof OperatorCondition)
		{			
			OperatorCondition cOperator = (OperatorCondition)cCondition;
						
			ICondition[] caConditions = cOperator.getConditions();
			for (int i = 0; i < caConditions.length; i++)
			{
				String sTempSQL = getSQL(caConditions[i], pServerMetaData, pUsePrefix);
							
				if (sTempSQL != null && sTempSQL.length() > 0)
				{
					if (i > 0 && result.length() > 0)
					{
						if (cOperator instanceof And)
						{
							result.append(" AND ");
						}
						else if (cOperator instanceof Or)
						{
							result.append(" OR ");
						}						
					}
					if (caConditions[i] instanceof OperatorCondition)
					{
						result.append("(");
						result.append(sTempSQL);
						result.append(")");
					}
					else
					{
						result.append(sTempSQL);
					}
				}
			}
		}
		else if (cCondition instanceof Not)
		{
			ICondition cCond = ((Not)cCondition).getCondition();
			String sTempSQL  = getSQL(cCond, pServerMetaData, pUsePrefix);
					
			result.append("NOT ");			
			if (cCond instanceof OperatorCondition)
			{
				result.append("(");
				result.append(sTempSQL);
				result.append(")");
			}
			else
			{
				result.append(sTempSQL);
			}
		}
		
		return result.toString();    
	}      

	/**
	 * Create an DB specific replace command, which replacs in the pSource all pOld to pNew.
	 * This implementation use the ANSI SQL REPLACE command.
	 *  
	 * @param pSource	the source to replace.
	 * @param pOld		the old value.
	 * @param pNew		the new value.
	 * 
	 * @return	the SQL command to to this.
	 */
	protected String createReplace(String pSource, String pOld, String pNew) 
	{
		StringBuilder sb = new StringBuilder("REPLACE(");
		sb.append(pSource);
		sb.append(",\'");
		sb.append(pOld);
		sb.append("\',\'");
		sb.append(pNew);
		sb.append("\')");
		return sb.toString();
	}
	
	/**
	 * Creates the where parameter. That is normally a single question mark, but it depends on
	 * the database if conversions are needed.
	 * 
	 * @param pServerMetaData 	the server metadata
	 * @param pCompare 			the compare condition
	 * @return the parameter representation for where clause
	 */
	protected String createWhereParam(ServerMetaData pServerMetaData, CompareCondition pCompare)
	{
		return "?";
	}

	/**
	 * Creates the where column. That is normally just the column name, but it depends on
	 * the database if conversions are needed.
	 * 
	 * @param pServerMetaData 	the server metadata
	 * @param pCompare 			the compare condition
	 * @param pColumnName		the column name to use
	 * @return the column name representation for where clause
	 */
	protected String createWhereColumn(ServerMetaData pServerMetaData, CompareCondition pCompare, String pColumnName)
	{
		return pColumnName;
	}
	
	
	/**
	 * Check if the Type of the column and the value to compare is equal.
	 *  
	 * @param pServerColumnMetaData		the server column meta data for the column to compare. 
	 * @param pCompare					the compare condition to use.
	 * @return false if the type isn't equal.
	 */
	public boolean isTypeEqual(ServerColumnMetaData pServerColumnMetaData, CompareCondition pCompare)
	{
		if (pCompare instanceof Equals)					
		{
			// check if type is !=
			Object    oValue   = pCompare.getValue();
			IDataType dataType = pServerColumnMetaData.getDataType();
			if (oValue instanceof String && !(dataType instanceof StringDataType)
					|| oValue instanceof Timestamp && !(dataType instanceof TimestampDataType)
					|| oValue instanceof BigDecimal && !(dataType instanceof BigDecimalDataType)
					|| oValue instanceof Boolean && !(dataType instanceof BooleanDataType))
			{
				return false;
			}
		}
		return true;
	}
	
	/**
	 * Returns the parameters used for this <code>ICondition</code>.
	 * 
	 * @param cCondition		the Condition to use.
	 * @return the parameters used for this <code>ICondition</code>.
	 */
	static Object[] getParameter(ICondition cCondition)
	{
		Object[] oaParameter = null;
		
		if (cCondition instanceof CompareCondition)
		{
			CompareCondition cCompare = (CompareCondition)cCondition;
			Object           oValue   = cCompare.getValue();
	
			if (oValue != null)
			{
				if ((cCompare instanceof Like || cCompare instanceof LikeIgnoreCase)
					&& oValue instanceof String)
				{
					oaParameter = new Object[] { ((String)oValue).replace('*', '%').replace('?', '_') };
				}						
				else 
				{
					oaParameter = new Object[] { oValue };
				}
			}
			else
			{
				oaParameter = new Object[0];
			}			
		}
		else if (cCondition instanceof OperatorCondition)
		{			
			OperatorCondition cOperator = (OperatorCondition)cCondition;
						
			ICondition[] caConditions = cOperator.getConditions();
			for (int i = 0; i < caConditions.length; i++)
			{
				Object[] params = getParameter(caConditions[i]);
				if (oaParameter == null)
				{
					oaParameter = new Object[0];
				}
				oaParameter = ArrayUtil.addAll(oaParameter, params);				
			}
		}
		else if (cCondition instanceof Not)
		{
			oaParameter = getParameter(((Not)cCondition).getCondition());
		}
		return oaParameter; 
	}
	
	/**
	 * Separates the schema and table from the given from clause. The separation is only 
	 * possible if the from clause is simple, e.g. it does not contain any quote characters
	 * and does not contain separators (',').
	 * 
	 * @param pFromClause the from clause
	 * @return [0]...schema, [1]...table
	 */
	protected String[] splitSchemaTable(String pFromClause)
	{
		boolean bComplex = false;
		
		//check if the from clause was special (means more than a view or table)
		if (pFromClause.indexOf(',') >= 0)
		{
			bComplex = true;
		}
		
		if (pFromClause.indexOf(sOpenQuote) >= 0)
		{
			bComplex = true;
		}
		
		int iFirstSpacePos = pFromClause.indexOf(' ');
		
		if (iFirstSpacePos > 0)
		{
			int iNextSpacePos = pFromClause.indexOf(' ', iFirstSpacePos + 1);
			
			if (iNextSpacePos > iFirstSpacePos)
			{
				bComplex = true;
			}
		}
		
		String[] sResult = new String[2];
		
		if (!bComplex)
		{
			//[schema.]table[ alias]

			if (iFirstSpacePos < 0)
			{
				iFirstSpacePos = pFromClause.length();
			}

			int iDotPos = pFromClause.indexOf('.');
			
			if (iDotPos > 0)
			{
				sResult[0] = pFromClause.substring(0, iDotPos);
				sResult[1] = pFromClause.substring(iDotPos + 1, iFirstSpacePos);
			}
			else
			{
				sResult[1] = pFromClause.substring(0, iFirstSpacePos);
			}
		}
		else
		{
			sResult[1] = pFromClause;
		}
		
		return sResult;
	}
	
	/**
	 * Sets the user-defined default schema.
	 * 
	 * @param pSchema the schema name
	 * @see #getDefaultSchema()
	 */
	public void setDefaultSchema(String pSchema)
	{
		sDefaultSchema = pSchema;
	}
	
	/**
	 * Gets the default schema name, if it was set manually.
	 * 
	 * @return the user-defined default schema name
	 * @see #setDefaultSchema(String)
	 */
	public String getDefaultSchema()
	{
		return sDefaultSchema;
	}
	
	/**
	 * Enables the database specific implementation to handle/convert special objects. Some databases have
	 * datatypes that are not defined in the standard. This datatypes have specific classes in the JDBC drivers.
	 * With this method it is possible to convert the values of specific JDBC driver classes into usable objects.
	 * We can not send any JDBC object to the client!
	 * 
	 * @param pColumnMetaData the column metadata
	 * @param pObject the read object
	 * @return the value to use
	 */
	protected Object convertDatabaseSpecificObjectToValue(ServerColumnMetaData pColumnMetaData, Object pObject)
	{
		return pObject;
	}
	
	/**
	 * Converts an object to a standard value for the specific database. Not all values are supported from
	 * the underlying database, e.g. java.sql.Timestamp is preferred instead of java.util.Date.
	 *  
	 * @param pValue any value
	 * @return the database specific value
	 */
	protected Object convertValueToDatabaseSpecificObject(Object pValue)
	{
		if (pValue instanceof Date && !(pValue instanceof Timestamp))
		{
			return new Timestamp(((Date)pValue).getTime());
		}
		else
		{
			return pValue;
		}
	}
	
	/**
	 * Enables the database specific implementation to change the metadata before the default settings are made.
	 * 
	 * @param pMetaData the metadata from the resultset
	 * @param pColumnIndex the column index in the resultset metadata
	 * @param pColumnMetaData the newly created and pre-configured column metadata
	 * @return <code>true</code> if column metadata are changed and the default settings should not be applied, 
	 *         <code>false</code> to apply the standard settings
	 * @throws SQLException if metadata access fails
	 */
	protected boolean setDatabaseSpecificType(ResultSetMetaData pMetaData, int pColumnIndex, ServerColumnMetaData pColumnMetaData) throws SQLException
	{
		return false;
	}

	//****************************************************************
	// Subclass definition
	//****************************************************************
	
	/**
	 * It stores all relevant information of the cached ResultSet for the fetch(...).
	 *  
	 * @author Roland Hrmann
	 */
	private static final class Select
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** The select statement. */
		private String 		sSelectStatement;
		
		/** The used filter. */
		private ICondition  cFilter;
		
		/** The last fetched row. */
		private int 		iLastRow;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Construct a Select cache object.
		 * 
		 * @param pSelectStatement	The select statement.
		 * @param pFilter			The used filter.
		 * @param pLastRow			The last fetched row.
		 */
		private Select(String pSelectStatement, ICondition pFilter, int pLastRow)
		{
			sSelectStatement = pSelectStatement;
			cFilter = pFilter;
			iLastRow = pLastRow;
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Overwritten methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * {@inheritDoc}
		 */
		@Override
		public int hashCode()
		{
			return sSelectStatement.hashCode() + iLastRow;
		}
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public boolean equals(Object o)
		{
			if (o instanceof Select)
			{
				Select obj = (Select)o;
				
				if (sSelectStatement.equals(obj.sSelectStatement)
					&& iLastRow == obj.iLastRow)
				{
					if (cFilter == null && obj.cFilter == null)
					{
						return true;
					}
					else if (cFilter != null && obj.cFilter != null)
					{
						Object [] oSrcParameter = getParameter(cFilter);
						Object [] oTrgParameter = getParameter(obj.cFilter);
						
						if (oSrcParameter.length == oTrgParameter.length)
						{
							for (int i = 0; i < oSrcParameter.length; i++)
							{
								if (!oSrcParameter[i].equals(oTrgParameter[i]))
								{
									return false;
								}
							}
							return true;
						}
					}
				}
				return false;
			}
			return super.equals(o);
		}
		
		/**
		 * {@inheritDoc}
		 */
		@Override
		public String toString()
		{
			StringBuilder sbResult = new StringBuilder();
			
			sbResult.append("Select :: ");
			sbResult.append(sSelectStatement);
			sbResult.append(", Filter=");
			sbResult.append(cFilter);
			sbResult.append(", LastRow=");
			sbResult.append(iLastRow);
			sbResult.append("\n");
			
			return sbResult.toString();
		}
		
	}	// Select

	/**
	 * The <code>ColumnMetaDataInfo</code> is caches information about column meta data requests.
	 * 
	 * @author Ren Jahn
	 */
	public static final class ColumnMetaDataInfo
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** the server column meta data. */
		private ServerColumnMetaData[] scmMetaData;
		
		/** the catalog name. */
		private String sCatalog = null;
		
		/** the schema name. */
		private String sSchema = null;
		
		/** the table name. */
		private String sTable = null;
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * Sets the column meta data.
		 * 
		 * @param pMetaData the meta data
		 */
		public void setColumnMetaData(ServerColumnMetaData[] pMetaData)
		{
			scmMetaData = pMetaData;
		}
		
		/**
		 * Gets the column meta data.
		 * 
		 * @return the meta data
		 */
		public ServerColumnMetaData[] getColumnMetaData()
		{
			return scmMetaData;
		}
		
		/**
		 * Sets the catalog name.
		 * 
		 * @param pCatalog the catalog name
		 */
		public void setCatalog(String pCatalog)
		{
			sCatalog = pCatalog;
		}
		
		/**
		 * Gets the catalog name.
		 * 
		 * @return the catalog name
		 */
		public String getCatalog()
		{
			return sCatalog;
		}
	
		/**
		 * Sets the schema name.
		 * 
		 * @param pSchema the schema name
		 */
		public void setSchema(String pSchema)
		{
			sSchema = pSchema;
		}
		
		/**
		 * Gets the schema name.
		 * 
		 * @return the schema
		 */
		public String getSchema()
		{
			return sSchema;
		}
		
		/**
		 * Sets the table name. 
		 * 
		 * @param pTable the table name
		 */
		public void setTable(String pTable)
		{
			sTable = pTable;
		}
		
		/**
		 * Gets the table name.
		 * 
		 * @return the table name
		 */
		public String getTable()
		{
			return sTable;
		}
		
	}	// ColumnMetaDataInfo	
	
} 	// DBAccess
