/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 04.01.2011 - [RH] - creation
 * 04.01.2011 - [RH] - #233 - Server Side Trigger implementation
 *                          - logger, createCSV function moved to AbstractStorage.
 * 05.01.2011 - [RH] - #233 - XXXbean() methods moved and changed to support IBean/POJOs as parameters and return type.
 * 07.01.2011 - [JR] - checked if event was modified (object saving)
 * 10.03.2011 - [JR] - changed formatInitCap call (always remove space characters and format)
 * 11.04.2011 - [JR] - #332: createBeanType implemented
 * 08.06.2011 - [JR] - used convertToMemberName instead of formatMemberName and formatInitCap
 * 16.12.2011 - [JR] - #498: set/get specific name 
 * 08.12.2012 - [JR] - #611: implemented xxxAsBean methods and made a default implementation
 * 19.12.2012 - [JR] - made getAndInitBeanType protected
 */
package com.sibvisions.rad.persist;

import java.io.OutputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import javax.rad.io.IFileHandle;
import javax.rad.io.RemoteFileHandle;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.ICondition;
import javax.rad.persist.DataSourceException;
import javax.rad.persist.IStorage;
import javax.rad.persist.MetaData;
import javax.rad.remote.IConnectionConstants;
import javax.rad.server.ISession;
import javax.rad.server.SessionContext;
import javax.rad.type.bean.Bean;
import javax.rad.type.bean.BeanType;
import javax.rad.type.bean.IBean;

import com.sibvisions.rad.model.DataBookUtil;
import com.sibvisions.rad.persist.event.StorageEvent;
import com.sibvisions.rad.persist.event.StorageEvent.ChangedType;
import com.sibvisions.rad.persist.event.StorageHandler;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.LocaleUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>AbstractStorage</code> implements the server side triggers mechnism for different types of Storages.
 * It is the abstract base class for storages which needs a trigger mechanism.
 * 
 * @author Roland Hrmann
 */
public abstract class AbstractStorage implements IStorage
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the logger. */
	private static ILogger logger = null;
	
	/** the mapping between column name and POJO property name. */
	private HashMap<String, String> hmpPropertyNames = null;
	
	/** The <code>EventHandler</code> for calculate row event. */
	private StorageHandler	eventCalculateRow;
	/** The <code>EventHandler</code> for before insert event. */
	private StorageHandler	eventBeforeInsert;
	/** The <code>EventHandler</code> for after insert event. */
	private StorageHandler	eventAfterInsert;
	/** The <code>EventHandler</code> for before update event. */
	private StorageHandler	eventBeforeUpdate;
	/** The <code>EventHandler</code> for after update event. */
	private StorageHandler	eventAfterUpdate;
	/** The <code>EventHandler</code> for before delete event. */
	private StorageHandler	eventBeforeDelete;
	/** The <code>EventHandler</code> for after delete event. */
	private StorageHandler	eventAfterDelete;
	
	/** The BeanType with all columns names from the metadata. */
	private BeanType btColumns;
	
	/** the property names for POJOs in the same order as the meta data. */
	private String[] sPropertyNames;
	
	/** the storage name. */
	private String sName;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public final MetaData getMetaData() throws DataSourceException
	{
		return executeGetMetaData();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final List<Object[]> fetch(ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		return fetch(false, pFilter, pSort, pFromRow, pMinimumRowCount);
	}

	/**
	 * {@inheritDoc}
	 */
	public final Object[] refetchRow(Object[] pDataRow) throws DataSourceException
	{
		return refetchRow(false, pDataRow);
	}

	/**
	 * {@inheritDoc}
	 */
	public final Object[] insert(Object[] pDataRow) throws DataSourceException
	{
		return insert(false, pDataRow);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public final Object[] update(Object[] pOldDataRow, Object[] pNewDataRow) throws DataSourceException
	{
		return update(false, pOldDataRow, pNewDataRow);
	}

	/**
	 * {@inheritDoc}
	 */
	public final void delete(Object[] pDeleteDataRow) throws DataSourceException
	{
		delete(false, pDeleteDataRow);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Write the current storage with the specified filter and sort into the given output stream as CSV Format.
	 * 
	 * @param pStream the output stream to use for the CSV stream
	 * @param pColumnNames the column names to include in the export
	 * @param pLabels the labels to show as header in the export
	 * @param pFilter the condition
	 * @param pSort the sort definition
	 * @param pSeparator the column separator
	 * @throws Exception if the CSV output couldn't written to stream
	 */
	public abstract void writeCSV(OutputStream pStream, 
			                      String[] pColumnNames, 
			                      String[] pLabels, 
			                      ICondition pFilter, 
			                      SortDefinition pSort, 
			                      String pSeparator) throws Exception;
		
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the getMetaData method.
	 * 
	 * @return the {@link MetaData}
	 * @throws DataSourceException if an <code>Exception</code> occur during getting the meta data from the storage 
	 * @see javax.rad.persist.IStorage#getMetaData()
	 */
	public abstract MetaData executeGetMetaData() throws DataSourceException;
	
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the refetchRow method.
	 * 
	 * @param pDataRow the specified row as <code>Object[]</code>.
	 * @return the refetched row as <code>Object[]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 * @see javax.rad.persist.IStorage#refetchRow(Object[])
	 */
	public abstract Object[] executeRefetchRow(Object[] pDataRow) throws DataSourceException;
	
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the fetch method.
	 * 
	 * @param pFilter the <code>ICondition</code> to use.
	 * @param pSort	the <code>SortDefinition</code> to use.
	 * @param pFromRow the from row index to request from storage.
	 * @param pMinimumRowCount the minimum row count to request, beginning from the pFromRow.
	 * @return the requested rows as <code>List[Object[]]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 * @see javax.rad.persist.IStorage#fetch(ICondition, SortDefinition, int, int)
	 */
	public abstract List<Object[]> executeFetch(ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException;
	
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the insert method.
	 * 
	 * @param pDataRow the new row as <code>Object[]</code> to insert.
	 * @return the newly inserted row from this IStorage.
	 * @throws DataSourceException if an <code>Exception</code> occur during insert the row to the storage
	 * @see javax.rad.persist.IStorage#insert(Object[])
	 */
	public abstract Object[] executeInsert(Object[] pDataRow) throws DataSourceException;
	
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the update method.
	 * 
	 * @param pOldDataRow the old row as <code>Object[]</code>
	 * @param pNewDataRow the new row as <code>Object[]</code> to update
	 * @return the updated row as <code>Object[]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during updating the row.
	 * @see javax.rad.persist.IStorage#update(Object[], Object[])
	 */
	public abstract Object[] executeUpdate(Object[] pOldDataRow, Object[] pNewDataRow) throws DataSourceException;
	
	/**
	 * This method has to be implemented in the Storage implementation and should have the functionality of the delete method.
	 * 
	 * @param pDeleteDataRow the row as <code>Object[]</code> to delete.
	 * @throws DataSourceException if an <code>Exception</code> occur during deleting the row or
	 *             				   if the storage isn't opened or the PrimaryKey is wrong and more/less 
	 *                             then one row is deleted.
	 * @see javax.rad.persist.IStorage#delete(Object[])
	 */
	public abstract void executeDelete(Object[] pDeleteDataRow) throws DataSourceException;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Fetches data. This methods makes a difference between object[] fetch and bean fetch.
	 * 
	 * @param pAsBean <code>true</code> means called via bean access and <code>false</code> means called via client API
	 * @param pFilter the filter
	 * @param pSort the sort condition
	 * @param pFromRow fetch from row
	 * @param pMinimumRowCount fetch minimum rows
	 * @return the fetched records
	 * @throws DataSourceException if fetch fails
	 */
	private List<Object[]> fetch(boolean pAsBean, ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		List<Object[]> liRows;
		
		if (pAsBean)
		{
			liRows = executeFetchAsBean(pFilter, pSort, pFromRow, pMinimumRowCount);
		}
		else
		{
			liRows = executeFetch(pFilter, pSort, pFromRow, pMinimumRowCount);			
		}
		
		if (eventCalculateRow != null)
		{
			StorageEvent event;

			for (int i = 0; i < liRows.size() && liRows.get(i) != null; i++)
			{
				event = new StorageEvent(this, ChangedType.CALCULATE_ROW, null, createBean(liRows.get(i)));
				
				eventCalculateRow.dispatchEvent(event);
				
				if (event.isNewModified())
				{
					liRows.set(i, createArray(event.getNew()));
				}
			}
		}

		return liRows;
	}
	
	/**
	 * Refetches data. This methods makes a difference between object[] refetch and bean refetch.
	 * 
	 * @param pAsBean <code>true</code> means called via bean access and <code>false</code> means called via client API 
	 * @param pDataRow the old data
	 * @return the new data
	 * @throws DataSourceException if refetch fails
	 */
	private Object[] refetchRow(boolean pAsBean, Object[] pDataRow) throws DataSourceException
	{
		Object[] oValues;
		
		if (pAsBean)
		{
			oValues = executeRefetchRowAsBean(pDataRow);
		}
		else
		{
			oValues = executeRefetchRow(pDataRow);
		}
		
		if (eventCalculateRow != null)
		{
			StorageEvent event = new StorageEvent(this, ChangedType.CALCULATE_ROW, null, createBean(oValues));
			
			eventCalculateRow.dispatchEvent(event);
			
			if (event.isNewModified())
			{
				oValues = createArray(event.getNew());
			}
		}
		
		return oValues;
	}
	
	/**
	 * Inserts data. This methods makes a difference between object[] insert and bean insert.
	 * 
	 * @param pAsBean <code>true</code> means called via bean access and <code>false</code> means called via client API
	 * @param pDataRow the data to insert
	 * @return the inserted data (includes changed values)
	 * @throws DataSourceException if insert fails
	 */
	private Object[] insert(boolean pAsBean, Object[] pDataRow) throws DataSourceException
	{
		Object[] oRow = pDataRow;
		
		if (eventBeforeInsert != null)
		{
			StorageEvent event = new StorageEvent(this, ChangedType.BEFORE_INSERT, null, createBean(oRow));
			
			eventBeforeInsert.dispatchEvent(event);			
			
			if (event.isNewModified())
			{
				oRow = createArray(event.getNew());
			}
		}

		if (pAsBean)
		{
			oRow = executeInsertAsBean(oRow);
		}
		else
		{
			oRow = executeInsert(oRow);
		}

		boolean bModified = false;

		IBean bnNew = null;

		if (eventAfterInsert != null)
		{
			bnNew = createBean(oRow);
			
			StorageEvent event = new StorageEvent(this, ChangedType.AFTER_INSERT, null, bnNew);
			
			eventAfterInsert.dispatchEvent(event);

			if (event.isNewModified())
			{
				bModified = true;
				
				bnNew = event.getNew();
			}
		}

		if (eventCalculateRow != null)
		{
			if (bnNew == null)
			{
				bnNew = createBean(oRow);
			}

			StorageEvent event = new StorageEvent(this, ChangedType.CALCULATE_ROW, null, bnNew);
			
			eventCalculateRow.dispatchEvent(event);
			
			if (event.isNewModified())
			{
				bModified = true;
				
				bnNew = event.getNew();
			}
		}
		
		if (bModified)
		{
			return createArray(bnNew);
		}
		else
		{
			return oRow;
		}
	}
	
	/**
	 * Updates data. This methods makes a difference between object[] update and bean update.
	 * 
	 * @param pAsBean <code>true</code> means called via bean access and <code>false</code> means called via client API
	 * @param pOldDataRow the old values
	 * @param pNewDataRow the new values
	 * @return the updated - new - values
	 * @throws DataSourceException if update fails
	 */
	private Object[] update(boolean pAsBean, Object[] pOldDataRow, Object[] pNewDataRow) throws DataSourceException
	{
		Object[] oOldRow = pOldDataRow;
		Object[] oNewRow = pNewDataRow;
		
		IBean bnOld = null;
		
		if (eventBeforeUpdate != null)
		{
			StorageEvent event = new StorageEvent(this, ChangedType.BEFORE_UPDATE, createBean(oOldRow), createBean(oNewRow));
			
			eventBeforeUpdate.dispatchEvent(event);

			bnOld = event.getOld();
			
			if (event.isOldModified())
			{
				oOldRow = createArray(bnOld);
			}
			
			if (event.isNewModified())
			{
				oNewRow = createArray(event.getNew());
			}
		}

		if (pAsBean)
		{
			oNewRow = executeUpdateAsBean(oOldRow, oNewRow);
		}
		else
		{
			oNewRow = executeUpdate(oOldRow, oNewRow);
		}
		
		boolean bModified = false;

		IBean bnNew = null;
		
		if (eventAfterUpdate != null)
		{
			if (bnOld == null)
			{
				bnOld = createBean(oOldRow);
			}
			
			bnNew = createBean(oNewRow);
			
			StorageEvent event = new StorageEvent(this, ChangedType.AFTER_UPDATE, bnOld, bnNew);
			
			eventAfterUpdate.dispatchEvent(event);
			
			//ignore old!
			
			if (event.isNewModified())
			{
				bModified = true;

				bnNew = event.getNew();
			}
		}
		
		if (eventCalculateRow != null)
		{
			if (bnNew == null)
			{
				bnNew = createBean(oNewRow);
			}
			
			StorageEvent event = new StorageEvent(this, ChangedType.CALCULATE_ROW, null, bnNew);
			
			eventCalculateRow.dispatchEvent(event);
			
			if (event.isNewModified())
			{
				bModified = true;
				
				bnNew = event.getNew();
			}
		}		
		
		if (bModified)
		{
			return createArray(bnNew);
		}
		else
		{
			return oNewRow;
		}
	}
	
	/**
	 * Deletes data. This methods makes a difference between object[] delete and bean delete.
	 * 
	 * @param pAsBean <code>true</code> means called via bean access and <code>false</code> means called via client API
	 * @param pDeleteDataRow the data to delete
	 * @throws DataSourceException if delete fails
	 */
	private void delete(boolean pAsBean, Object[] pDeleteDataRow) throws DataSourceException
	{
		Object[] oRow = pDeleteDataRow; 
		
		IBean bnOld = null;
		
		if (eventBeforeDelete != null)
		{
			bnOld = createBean(oRow);
			
			StorageEvent event = new StorageEvent(this, ChangedType.BEFORE_DELETE, bnOld, null);
			
			eventBeforeDelete.dispatchEvent(event);			

			if (event.isOldModified())
			{
				bnOld = event.getOld();

				oRow = createArray(bnOld);
			}
		}

		if (pAsBean)
		{
			executeDeleteAsBean(oRow);
		}
		else
		{
			executeDelete(oRow);
		}
		
		if (eventAfterDelete != null)
		{
			if (bnOld == null)
			{
				bnOld = createBean(oRow);
			}
			
			StorageEvent event = new StorageEvent(this, ChangedType.AFTER_DELETE, bnOld, null);
			
			eventAfterDelete.dispatchEvent(event);			
		}		
	}
	
	/**
	 * Sets the storage name.
	 * 
	 * @param pName the name
	 */
	public void setName(String pName)
	{
		sName = pName;
	}
	
	/**
	 * Gets the storage name.
	 * 
	 * @return the name
	 */
	public String getName()
	{
		return sName;
	}
	
	/**
	 * Logs debug information.
	 * 
	 * @param pInfo the debug information
	 */
	protected void debug(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}
		
		logger.debug(pInfo);
	}

	/**
	 * Logs information.
	 * 
	 * @param pInfo the information
	 */
	protected void info(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}
		
		logger.info(pInfo);
	}
	
	/**
	 * Logs error information.
	 * 
	 * @param pInfo the error information
	 */
	protected void error(Object... pInfo)
	{
		if (logger == null)
		{
			logger = LoggerFactory.getInstance(getClass());
		}

		logger.error(pInfo);
	}	
	
	/**
	 * Write the current DBStorage with the specified filter and sort to the 
	 * export.csv file in CSV format and returns the file handle. The filename
	 * will be built depending of the object name from the {@link SessionContext}.
	 * 
	 * @param pColumnNames the column names to include in the export
	 * @param pLabels the labels to show as header in the export
	 * @param pFilter the filter to use on the DBStorage
	 * @param pSort the sort to use
	 * @return the file handle for the new generated CSV file. 
	 * @throws Exception if the CSV output couldn't written to stream. 
	 */
	public IFileHandle createCSV(String[] pColumnNames, String[] pLabels, ICondition pFilter, SortDefinition pSort) throws Exception
	{
		SessionContext context = SessionContext.getCurrentInstance();
		
		String sFileName = null;
		
		if (context != null)
		{
			sFileName = context.getObjectName();
		}
		
		return createCSV(sFileName, pColumnNames, pLabels, pFilter, pSort);
	}

	/**
	 * Write the current DBStorage with the specified filter and sort to the 
	 * export.csv file in CSV format and returns the file handle. The filename
	 * will be built depending of the object name from the {@link SessionContext}.
	 * 
	 * @param pColumnNames the column names to include in the export
	 * @param pLabels the labels to show as header in the export
	 * @param pFilter the filter to use on the DBStorage
	 * @param pSort the sort to use
	 * @param pLanguage the language to use
	 * @return the file handle for the new generated CSV file. 
	 * @throws Exception if the CSV output couldn't written to stream. 
	 */
	public IFileHandle createCSV(String[] pColumnNames, String[] pLabels, ICondition pFilter, SortDefinition pSort, String pLanguage) throws Exception
	{
		try
		{
			LocaleUtil.setThreadDefault(LocaleUtil.forLanguageTag(pLanguage));
			
			return createCSV(pColumnNames, pLabels, pFilter, pSort);
		}
		finally
		{
			LocaleUtil.setThreadDefault(null);
		}
	}

	/**
	 * Write the current DBStorage with the specified filter and sort to the 
	 * export.csv file in CSV format and returns the file handle.
	 * 
	 * @param pFileName the filename to use.
	 * @param pColumnNames the column names to include in the export
	 * @param pLabels the labels to show as header in the export
	 * @param pFilter the filter to use on the DBStorage
	 * @param pSort the sort to use
	 * @return the file handle for the new generated CSV file. 
	 * @throws Exception if the CSV output couldn't written to stream. 
	 */
	public IFileHandle createCSV(String pFileName, String[] pColumnNames, String[] pLabels, ICondition pFilter, SortDefinition pSort) throws Exception
	{
		RemoteFileHandle fileHandle = new RemoteFileHandle(DataBookUtil.formatCSVFileName(pFileName));
		
		String sSeperator;

		ISession sess = SessionContext.getCurrentSession();

		if (sess != null)
		{
			String sLang = (String)sess.getProperty(IConnectionConstants.PREFIX_CLIENT + "locale.language");
			String sCountry = (String)sess.getProperty(IConnectionConstants.PREFIX_CLIENT + "locale.country");
			String sVariant = (String)sess.getProperty(IConnectionConstants.PREFIX_CLIENT + "locale.variant");

			if (sLang != null)
			{
				sSeperator = LocaleUtil.getListSeparator(new Locale(sLang, sCountry, sVariant));				
			}
			else
			{
				sSeperator = LocaleUtil.getListSeparator();
			}
		}
		else
		{
			sSeperator = LocaleUtil.getListSeparator();
		}
		
		OutputStream os = fileHandle.getOutputStream();
		writeCSV(os, pColumnNames, pLabels, pFilter, pSort, sSeperator);
		os.close();
		
		return fileHandle;
	}

    /**
     * Gets the EventHandler for calculate row event.
     * This is called after fetch, refetchRow, insert and update method calls.
     * 
     * @return the EventHandler for after deleted event.
     */
	public StorageHandler eventCalculateRow()
	{
		if (eventCalculateRow == null)
		{
			eventCalculateRow = new StorageHandler();
		}
		return eventCalculateRow;
	}
		
    /**
     * Gets the EventHandler for before insert event.
     * 
     * @return the EventHandler for before insert event.
     */
	public StorageHandler eventBeforeInsert()
	{
		if (eventBeforeInsert == null)
		{
			eventBeforeInsert = new StorageHandler();
		}
		return eventBeforeInsert;
	}
	
    /**
     * Gets the EventHandler for after insert event.
     * 
     * @return the EventHandler for after insert event.
     */
	public StorageHandler eventAfterInsert()
	{
		if (eventAfterInsert == null)
		{
			eventAfterInsert = new StorageHandler();
		}
		return eventAfterInsert;
	}
		
    /**
     * Gets the EventHandler for before update event.
     * 
     * @return the EventHandler for before update event.
     */
	public StorageHandler eventBeforeUpdate()
	{
		if (eventBeforeUpdate == null)
		{
			eventBeforeUpdate = new StorageHandler();
		}
		return eventBeforeUpdate;
	}
	
    /**
     * Gets the EventHandler for after update event.
     * 
     * @return the EventHandler for after update event.
     */
	public StorageHandler eventAfterUpdate()
	{
		if (eventAfterUpdate == null)
		{
			eventAfterUpdate = new StorageHandler();
		}
		return eventAfterUpdate;
	}
		
    /**
     * Gets the EventHandler for before delete event.
     * 
     * @return the EventHandler for before delete event.
     */
	public StorageHandler eventBeforeDelete()
	{
		if (eventBeforeDelete == null)
		{
			eventBeforeDelete = new StorageHandler();
		}
		return eventBeforeDelete;
	}
	
    /**
     * Gets the EventHandler for after delete event.
     * 
     * @return the EventHandler for after delete event.
     */
	public StorageHandler eventAfterDelete()
	{
		if (eventAfterDelete == null)
		{
			eventAfterDelete = new StorageHandler();
		}
		return eventAfterDelete;
	}
	
	/**
	 * Gets the bean type based on the meta data.
	 * 
	 * @return the bean type
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	protected BeanType getAndInitBeanType() throws DataSourceException
	{
		if (btColumns == null)
		{
			btColumns = createBeanType(executeGetMetaData().getColumnNames());
		}
		
		return btColumns;
	}
	
	/**
	 * Creates a {@link BeanType} for the given column names.
	 * 
	 * @param pColumnNames the column names which are allowed for beans
	 * @return the bean type
	 */
	protected BeanType createBeanType(String[] pColumnNames)
	{
		BeanType btCols = new BeanType(pColumnNames); 
		
		//format the property names for POJOs
		sPropertyNames = new String[pColumnNames.length];
		
		String sPropName;
		
		for (int i = 0; i < sPropertyNames.length; i++)
		{
			if (hmpPropertyNames != null)
			{
				sPropName = hmpPropertyNames.get(pColumnNames[i]);
			}
			else
			{
				sPropName = null;
			}
			
			if (sPropName == null)
			{
				sPropName = StringUtil.convertToMemberName(pColumnNames[i]);
			}
			
			sPropertyNames[i] = sPropName;
		}
		
		return btCols;
	}
	
	/**
	 * Creates a new bean with all column names from the meta data. Only this column names are allowed.
	 * 
	 * @return a new instance of an {@link IBean} implementation
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	public IBean createEmptyBean() throws DataSourceException
	{
		return new Bean(getAndInitBeanType());
	}	
	
	/**
	 * Creates a bean with given values. The bean contains the property names from the column meta data 
	 * and not more.
	 * 
	 * @param pValues the values in same order as the meta data
	 * @return a new bean
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	protected IBean createBean(Object[] pValues) throws DataSourceException
	{
		BeanType btype = getAndInitBeanType();

		Bean dbRow = new Bean(btype);
		
		String[] sProps = btype.getPropertyNames();
		
		for (int i = 0; i < sProps.length; i++)
		{
			if (pValues != null && i < pValues.length)
			{
				dbRow.put(sProps[i], pValues[i]);
			}
			else if (pValues == null)
			{
				dbRow.put(sProps[i], null);
			}
		}
		
		return dbRow;
	}
	
	/**
	 * Creates an array of values from a given object. The array has the same element count
	 * as the meta data column count.
	 * 
	 * @param pObject a POJO or bean
	 * @return an array containing the property values from the given object
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	protected Object[] createArray(Object pObject) throws DataSourceException
	{
		BeanType btype = getAndInitBeanType();
	
		if (pObject == null)
		{
			return new Object[btype.getPropertyCount()];
		}
		
		IBean bnData;

		String[] sProps;
		
		if (pObject instanceof IBean)
		{
			bnData = (IBean)pObject;
			sProps = btype.getPropertyNames();
		}
		else
		{
			bnData = new Bean(pObject);
			
			sProps = sPropertyNames;
		}

		//always ALL columns, not the columns contained in pObject
		Object[] oResult = new Object[sProps.length];
		
		for (int i = 0; i < sProps.length; i++)
		{
			try
			{
				oResult[i] = bnData.get(sProps[i]);
			}
			catch (Exception e)
			{
				debug(e);
			}
		}
		
		return oResult;
	}

	/**
	 * Creates a POJO from the given type and with given values.
	 * 
	 * @param <T> the type of the POJO
	 * @param pClass the class of the POJO
	 * @param pValues the values for the properties in the same order as the meta data
	 * @return the POJO
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	protected <T> T createPOJO(Class<T> pClass, Object[] pValues) throws DataSourceException
	{
		Bean bean = new Bean(pClass);
		
		getAndInitBeanType();

		for (int i = 0; i < sPropertyNames.length; i++)
		{
			try
			{
				bean.put(sPropertyNames[i], pValues[i]);
			}
			catch (Exception e)
			{
				debug(e);
			}
		}
		
		return (T)bean.getObject();
	}
	
	/**
	 * Creates a POJO from the tiven type and with the values from a bean.
	 * 
	 * @param <T> the type of the POJO
	 * @param pClass the class of the POJO
	 * @param pBean the bean with values for the POJO
	 * @return the POJO
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	public <T> T createPOJO(Class<T> pClass, IBean pBean) throws DataSourceException
	{
		Bean bean = new Bean(pClass);
		
		BeanType btype = getAndInitBeanType();
		
		String[] sCols = btype.getPropertyNames();
		
		for (int i = 0; i < sCols.length; i++)
		{
			try
			{
				bean.put(sPropertyNames[i], pBean.get(sCols[i]));
			}
			catch (Exception e)
			{
				debug(e);
			}
		}
		
		return (T)bean.getObject();
	}
	
	/**
	 * Updates a bean with values from a POJO. Only values from known properties will be updated. The
	 * property names from the meta data will be used.
	 * 
	 * @param pBean the bean
	 * @param pPOJO the POJO
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage
	 */
	public void updateBean(IBean pBean, Object pPOJO) throws DataSourceException
	{
		Bean bean = new Bean(pPOJO);
		
		BeanType btype = getAndInitBeanType();
		
		String[] sCols = btype.getPropertyNames(); 
		
		for (int i = 0; i < sCols.length; i++)
		{
			try
			{
				pBean.put(sCols[i], bean.get(sPropertyNames[i]));
			}
			catch (Exception e)
			{
				debug(e);
			}
		}
	}
	
	/**
	 * Sets the property name (Java standard) for a given column name. The name will be used
	 * for synchronizing POJOs with beans.
	 * 
	 * @param pColumnName the column name
	 * @param pPropertyName the java property name e.g. firstName instead of FIRST_NAME
	 * @throws DataSourceException if meta data access failed
	 */
	public void setPropertyNameForColumn(String pColumnName, String pPropertyName) throws DataSourceException
	{
		if (pPropertyName == null && hmpPropertyNames == null)
		{
			return;
		}

		MetaData metaData = executeGetMetaData();
		
		if (hmpPropertyNames == null)
		{
			hmpPropertyNames = new HashMap<String, String>();
		}
		
		if (pPropertyName == null)
		{
			hmpPropertyNames.remove(pColumnName);
			
			if (metaData != null)
			{
				int iPos = metaData.getColumnMetaDataIndex(pColumnName);
				
				if (iPos >= 0)
				{
					//reset property name
					sPropertyNames[iPos] = StringUtil.convertToMemberName(pColumnName);
				}
			}
		}
		else
		{
			hmpPropertyNames.put(pColumnName, pPropertyName);
			
			if (metaData != null)
			{
				int iPos = metaData.getColumnMetaDataIndex(pColumnName);
				
				if (iPos >= 0)
				{
					//set new name
					sPropertyNames[iPos] = pPropertyName;
				}
			}
		}
	}
	
	/**
	 * Returns the requested bean from the storage.
	 * 
	 * @param pFilter the filter as <code>ICondition</code> to get exactly one Bean.
	 * @return the requested rows as {@link IBean}s, or <code>null</code> if no row was found
	 * @throws DataSourceException if an <code>Exception</code> occurs during interacting with the storage or
	 *                             more than one Bean were found
	 */
	public IBean fetchBean(ICondition pFilter) throws DataSourceException
	{
		return (IBean)fetch(IBean.class, pFilter);
	}
	
	/**
	 * Returns the requested bean/POJO from the storage.
	 * 
	 * @param <T> the type of the POJO or bean.
	 * @param pClass the class of the POJO or bean.
	 * @param pFilter the filter as <code>ICondition</code> to use get exactly one Bean.
	 * @return the requested Bean/POJO, or <code>null</code> if no Bean/POJO was found
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage or
	 *                             more than one Bean/POJO were found
	 */
	public <T> T fetch(Class<T> pClass, ICondition pFilter) throws DataSourceException
	{
		List<Object[]> liObject = fetch(true, pFilter, null, 0, 2);

		int iSize = liObject.size();
		
		if (iSize < 2)
		{
			return null;
		}

		if (iSize == 2 && liObject.get(1) == null)
		{
			if (IBean.class.isAssignableFrom(pClass))
			{
				return (T)createBean(liObject.get(0));
			}
			else
			{
				return (T)createPOJO(pClass, liObject.get(0));
			}			
		}

		throw new DataSourceException("More than one bean available!");
	}
	
	/**
	 * Returns the requested list of beans from the storage.
	 * It uses the filter to reduce the result, and the sort for the order.
	 * 
	 * @param pFilter the filter as <code>ICondition</code> to use
	 * @param pSort the <code>SortDefinition</code> to use
	 * @param pFromRow the from row index to request from storage
	 * @param pMinimumRowCount the minimum row count to request, beginning from the pFromRow.
	 * @return the requested list of Beans from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 */
	public List<IBean> fetchBean(ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		return (List<IBean>)fetch(IBean.class, pFilter, pSort, pFromRow, pMinimumRowCount);
	}

	/**
	 * Returns the requested list of beans/POJOs from the storage.
	 * It uses the filter to reduce the result, and the sort for the order.
	 * 
	 * @param <T> the type of the POJO or bean.
	 * @param pClass the class of the POJO or bean.
	 * @param pFilter the filter as <code>ICondition</code> to use
	 * @param pSort the <code>SortDefinition</code> to use
	 * @param pFromRow the from row index to request from storage
	 * @param pMinimumRowCount the minimum row count to request, beginning from the pFromRow.
	 * @return the requested list of Beans/POJOs from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 */
	public <T> List<T> fetch(Class<T> pClass, ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		List<Object[]> liObject = fetch(true, pFilter, pSort, pFromRow, pMinimumRowCount);
		
		// create beans from the objects
		List<T> liBean = new ArrayUtil<T>(liObject.size());
		
		Object[] oValues;
		
		for (int i = 0, anz = liObject.size(); i < anz; i++)
		{
			oValues = liObject.get(i);
			
			if (oValues != null)
			{
				if (IBean.class.isAssignableFrom(pClass))
				{
					liBean.add((T)createBean(oValues));
				}
				else
				{
					liBean.add((T)createPOJO(pClass, oValues));
				}					
			}
		}
		
		return liBean;
	}
	
	/**
	 * Refetchs the specified bean/POJO and returns a bean/POJO with current values from the storage.
	 * 
	 * @param <T> the type of the POJO or bean.	
	 * @param pObject the specified bean/POJO to refetch from the storage. 
	 * @return the refetched bean/POJO with all values from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the Storage.
	 */
	public <T> T refetch(T pObject) throws DataSourceException
	{
		if (pObject instanceof IBean) 
		{
			return (T)createBean(refetchRow(true, createArray(pObject)));
		}
		else
		{
			if (pObject instanceof Class<?>)
			{
				return (T)createPOJO((Class<?>)pObject, refetchRow(true, createArray(pObject)));
			}
			else
			{
				return (T)createPOJO(pObject.getClass(), refetchRow(true, createArray(pObject)));
			}
		}
	}	
	
	/**
	 * Inserts the new bean/POJO and returns a bean/POJO with current values from the storage.
	 * 
	 * @param <T> the type of the POJO or bean.	
	 * @param pObject the new IBean/POJO to insert to the storage.
	 * @return the newly inserted IBean/POJO with all automatically filled in values from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during insert into the storage
	 */
	public <T> T insert(T pObject) throws DataSourceException
	{
		if (pObject instanceof IBean) 
		{ 
			return (T)createBean(insert(true, createArray(pObject)));
		}
		else
		{
			if (pObject instanceof Class<?>)
			{
				return (T)createPOJO((Class<?>)pObject, insert(true, createArray(pObject)));
			}
			else
			{
				return (T)createPOJO(pObject.getClass(), insert(true, createArray(pObject)));
			}
		}
	}	
	
	/**
	 * Updates a bean/POJO with the PrimaryKey columns and provides values. It returns a bean/POJO with the 
	 * current values from the storage.
	 * 
	 * @param <T> the type of the POJO or bean.	
	 * @param pObject the new IBean/POJO to use for the update
	 * @return the updated IBean/POJO from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during updating process in the storage.
	 */
	public <T> T update(T pObject) throws DataSourceException
	{
		BeanType btype = getAndInitBeanType();

		MetaData metaData = executeGetMetaData();
		
		Object[] oNewValues = createArray(pObject);
		Object[] oOldValues = new Object[btype.getPropertyCount()];
		
		String[] sPKCols = metaData.getPrimaryKeyColumnNames(); 
		
		//copy PK columns -> important for update!
		for (int i = 0, idx; i < sPKCols.length; i++)
		{
			idx = metaData.getColumnMetaDataIndex(sPKCols[i]);
			
			oOldValues[idx] = oNewValues[idx];
		}
		
		if (pObject instanceof IBean) 
		{ 
			return (T)createBean(update(true, oOldValues, oNewValues));
		}
		else
		{
			if (pObject instanceof Class<?>)
			{
				return (T)createPOJO((Class<?>)pObject, update(true, oOldValues, oNewValues));
			}
			else
			{
				return (T)createPOJO(pObject.getClass(), update(true, oOldValues, oNewValues));
			}
		}
	}

	/**
	 * Deletes the specified bean/POJO from the storage.
	 * 
	 * @param <T> the type of the POJO or bean.
	 * @param pObject the bean/POJO to delete from the storage.
	 * @throws DataSourceException if an <code>Exception</code> occur during the delete operation in the storage.
	 */
	public <T> void delete(T pObject) throws DataSourceException
	{
		delete(true, createArray(pObject));
	}
	
	/**
	 * This method will be used if {@link #refetch(Object)} was called.
	 * 
	 * @param pDataRow the specified row as <code>Object[]</code>.
	 * @return the refetched row as <code>Object[]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 * @see javax.rad.persist.IStorage#refetchRow(Object[])
	 */
	public Object[] executeRefetchRowAsBean(Object[] pDataRow) throws DataSourceException
	{
		return executeRefetchRow(pDataRow);
	}
	
	/**
	 * This method will be used if {@link #fetchBean(ICondition, SortDefinition, int, int)} was called.
	 * 
	 * @param pFilter the <code>ICondition</code> to use.
	 * @param pSort	the <code>SortDefinition</code> to use.
	 * @param pFromRow the from row index to request from storage.
	 * @param pMinimumRowCount the minimum row count to request, beginning from the pFromRow.
	 * @return the requested rows as <code>List[Object[]]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during interacting with the storage.
	 * @see javax.rad.persist.IStorage#fetch(ICondition, SortDefinition, int, int)
	 */
	public List<Object[]> executeFetchAsBean(ICondition pFilter, SortDefinition pSort, int pFromRow, int pMinimumRowCount) throws DataSourceException
	{
		return executeFetch(pFilter, pSort, pFromRow, pMinimumRowCount);
	}
	
	/**
	 * This method will be used if {@link #insert(Object)} was called.
	 * 
	 * @param pDataRow the new row as <code>Object[]</code> to insert.
	 * @return the newly inserted row from this IStorage.
	 * @throws DataSourceException if an <code>Exception</code> occur during insert the row to the storage
	 * @see javax.rad.persist.IStorage#insert(Object[])
	 */
	public Object[] executeInsertAsBean(Object[] pDataRow) throws DataSourceException
	{
		return executeInsert(pDataRow);
	}
	
	/**
	 * This method will be used if {@link #update(Object)} was called.
	 * 
	 * @param pOldDataRow the old row as <code>Object[]</code>
	 * @param pNewDataRow the new row as <code>Object[]</code> to update
	 * @return the updated row as <code>Object[]</code>.
	 * @throws DataSourceException if an <code>Exception</code> occur during updating the row.
	 * @see javax.rad.persist.IStorage#update(Object[], Object[])
	 */
	public Object[] executeUpdateAsBean(Object[] pOldDataRow, Object[] pNewDataRow) throws DataSourceException
	{
		return executeUpdate(pOldDataRow, pNewDataRow);
	}
	
	/**
	 * This method will be used if {@link #delete(Object)} was called.
	 * 
	 * @param pDeleteDataRow the row as <code>Object[]</code> to delete.
	 * @throws DataSourceException if an <code>Exception</code> occur during deleting the row or
	 *             				   if the storage isn't opened or the PrimaryKey is wrong and more/less 
	 *                             then one row is deleted.
	 * @see javax.rad.persist.IStorage#delete(Object[])
	 */
	public void executeDeleteAsBean(Object[] pDeleteDataRow) throws DataSourceException
	{
		executeDelete(pDeleteDataRow);
	}
	
}	// AbstractStorage
