/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.10.2008 - [RH] - bug fixes, toString() optimized
 * 12.10.2008 - [RH] - bug fixes, store(), insert(), restore()
 * 02.11.2008 - [RH] - bug fixes in SetValue 
 *                     - copy original values corrected
 *                     - convertAndCheckToTypeClass add in setValue for correct DataType conversion 
 *                     SelectedDataRow support added
 * 11.11.2008 - [RH] - MasterRow set in MemDataPage
 * 17.11.2008 - [RH] - optimized; ArrayUtil to int[] changed; unnecessary intern removed    
 * 19.11.2008 - [RH] - fetchToRowInternal added for PK check, that no row get requested from AbstractStorage
 * 10.04.2009 - [RH] - interface review - size to getRowCount renamed
 *                                        getDataRow uses an ChangeableDataRow
 *                                        getMasterRow(), getChangedDataRows() added
 *                                        change state management is moved to ChangeableDataRow and MemDataBook
 * 18.04.2009 - [RH] - javadoc reviewed, code optimized                                                     
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 16.10.2009 - [RH] - fetchAll in one request implemented -> fecthToRow(-1)=fetchAll
 *                     filter StackOverflow Error fixed [BUGFIXED]
 * 18.10.2009 - [RH] - setFilter(xx), with isMemFilter == true, on an RemoteDataBook, didn't fetched all rows, before it filters in mem. [BUGFIXED]
 * 29.09.2010 - [RH] - countRows renamed to getEstimatedRowCount(),  getEstimatedRowCount() returns in MemDataPage getRowCount() 
 * 08.04.2011 - [RH] - #330 - restoreAllRows fails in DataSource Level.
 */
package com.sibvisions.rad.model.mem;

import javax.rad.model.IChangeableDataRow;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.IDataType;

import com.sibvisions.util.ArrayUtil;

/**
 * A <code>MemDataPage</code> is the memory implementation for a data page of an 
 * <code>IDataBook</code>.<br>
 * A master <code>IDataBook</code> has one <code>IDataPage</code> for itself. If the 
 * <code>IDataBook</code> is (also) a detail <code>IDataBook</code> 
 * it stores all <code>IDataPage</code>'s for each loaded master row (parent master). <br>
 * 
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IChangeableDataRow
 * @author Roland Hrmann
 */
public class MemDataPage implements IDataPage
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** The initial storage size for the DataRow/Object array. */
	private static final int	INIT_STORAGE_SIZE = 16;

	/** The IDataBook that uses this MemDataPage. */
	private MemDataBook			dbDataBook;
	/** The master row from the corresponding master DataBook. */
	private IDataRow			drMasterDataRow; 		
	/** The storage array with all DataRows of the DataPage. */
	private ArrayUtil<Object[]>	alStorage;
	
	/** The array with all changes rows. */
	private int[]			 	iaChangedRows = new int[] {};
	
	/** 
	 * The boolean indicates if all rows are fetched from the under laying storage.
	 * In this MemDataPage its this always true.
	 */
	private boolean 			bAllFetched = true;
		
	/** Array with all rows that shows the MemDataBook outside. -> mem sort, filter. */
	private int[]				iaUsedRows;
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Construct a new MemDataPage for the specified IDataBook and the corresponding master row. 
	 * 
	 * @param pDataBook 		the IDataBook which uses this MemDataPage 
	 * @param pMasterDataRow	the corresponding master row of the master IDataBook of the 
	 *                          above specified IDataBook 
	 */
	public MemDataPage(MemDataBook pDataBook, IDataRow pMasterDataRow)
	{
		dbDataBook = pDataBook;
		drMasterDataRow = pMasterDataRow; 
		alStorage  = new ArrayUtil<Object[]>(INIT_STORAGE_SIZE);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook getDataBook()
	{
		return dbDataBook;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDataRow getMasterDataRow()
	{
		return drMasterDataRow;
	}
	
	/**
	 * {@inheritDoc}  
	 */
	public IChangeableDataRow getDataRow(int pDataRowIndex) throws ModelException
	{
		initFilterSort();
		
		if (!bAllFetched && pDataRowIndex >= alStorage.size())
		{
			fetchToRow(pDataRowIndex);
		}

		if (pDataRowIndex >= 0 && pDataRowIndex < getRowCountInternal())
		{
			return getDataRowWithoutSort(getInternalRowIndex(pDataRowIndex));
		}
		else
		{
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void fetchAll() throws ModelException
	{
		if (!bAllFetched)				
		{
			fetchToRow(-1);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isAllFetched() throws ModelException
	{
		return bAllFetched;
	}

	/**
	 * {@inheritDoc}
	 */
	public int getRowCount() throws ModelException
	{
		initFilterSort();

		return getRowCountInternal();
	}		
	
	/**
	 * {@inheritDoc}
	 */
	public int[] getChangedDataRows()
	{
		return iaChangedRows;
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition) throws ModelException
	{
		return searchNext(pCondition, 0);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition, int pRowNum) throws ModelException
	{
		if (pCondition != null && pRowNum >= 0)
		{
			initFilterSort();
			
			if (!bAllFetched && pRowNum >= alStorage.size())
			{
				fetchToRow(pRowNum);
			}
			
			while (pRowNum < getRowCountInternal())
			{
				if (pCondition.isFulfilled(getDataRowWithoutSort(getInternalRowIndex(pRowNum))))
				{
					return pRowNum;
				}
				pRowNum++;
				if (!bAllFetched && pRowNum >= alStorage.size())
				{
					fetchToRow(pRowNum);
				}
			}
		}
		return -1;
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition) throws ModelException
	{
		return searchPrevious(pCondition, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition, int pRowNum) throws ModelException
	{
		if (pCondition != null && pRowNum != 0)
		{
			fetchAll();
			initFilterSort();
			
			int rowCount = getRowCountInternal();

			if (pRowNum < 0 || pRowNum > rowCount)
			{
				pRowNum = rowCount - 1;
			}
			else
			{
				pRowNum--;
			}
			
			while (pRowNum >= 0)
			{
				if (pCondition.isFulfilled(getDataRowWithoutSort(getInternalRowIndex(pRowNum))))
				{
					return pRowNum;
				}
				pRowNum--;
			}
		}
		return -1;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		StringBuilder sbResult = new StringBuilder();
		
		sbResult.append("DataPage ::");
		
		if (dbDataBook != null)
		{
			sbResult.append("[");
			sbResult.append(dbDataBook.getName());
			sbResult.append("] -");
		}
		
		try
		{
			int iLen = getRowCount();
	        for (int i = 0; i < iLen; i++)
			{
				sbResult.append("[");
				sbResult.append(i);
				sbResult.append("]=");
				
				String[] sColumns = dbDataBook.getRowDefinition().getColumnNames();
	
				for (int j = 0; j < sColumns.length; j++)
				{
					sbResult.append("[");
					sbResult.append(alStorage.get(getInternalRowIndex(i))[j]);
					sbResult.append("]");
				}
				
				sbResult.append(", ");
			}
		}
		catch (ModelException modelException)
		{
			return sbResult.toString() + " :: " + modelException.getMessage();
		}
		
        sbResult.append(", CHANGES: ");
        for (int i = 0; i < iaChangedRows.length; i++)
        {
        	sbResult.append("[");
        	sbResult.append(iaChangedRows[i]);
        	sbResult.append("]");
        }
        
        sbResult.append("\n");
        
        return sbResult.toString();
	}	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * Sets that the MemDataPage has all fetched. Should only used from derived classes.
	 * 
	 * @param pAllFetched	the boolean with the fetch state.
	 */
	protected void setAllFetched(boolean pAllFetched)
	{
		bAllFetched = pAllFetched;
	}

	/**
	 * It adds an new IDataRow to the DataPage in the object[] storage.
	 * 
	 * @param pValues			the values.
	 * @throws ModelException	if the IDataRow couldn't add to the storage.
	 */
	protected void addFetchedRow(Object[] pValues) throws ModelException
	{
		alStorage.add(pValues);
	}
	
	/**
	 * It inserts a new <code>IChangeableDataRow</code> in the MemDataPage at the specified index. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @param pStorage			the storage of the new <code>IChangeableDataRow</code> to insert
	 * @throws ModelException	if the Filter/Sort couldn't initialized.
	 */
	protected void insert(int pDataRowIndex, Object[] pStorage) throws ModelException
	{
		initFilterSort();

	 	alStorage.add(pDataRowIndex, pStorage);

		if (iaUsedRows != null)
		{
			shiftIndicies(iaUsedRows, pDataRowIndex, 1);
			iaUsedRows = ArrayUtil.add(iaUsedRows, pDataRowIndex, pDataRowIndex);
		}

		shiftIndicies(iaChangedRows, pDataRowIndex, 1);
		addChange(pDataRowIndex);
	}
	
	/**
	 * Deletes the specified row with the given index in the MemDataPage. 
	 * 
	 * @param pDataRowIndex	the row index to use.
	 */
	protected void delete(int pDataRowIndex) 
	{
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		
		alStorage.remove(iInternalIndex);		
		
		if (iaUsedRows != null)
		{
			iaUsedRows = ArrayUtil.remove(iaUsedRows, pDataRowIndex);
			shiftIndicies(iaUsedRows, iInternalIndex, -1);
		}

		removeChange(pDataRowIndex);			
		shiftIndicies(iaChangedRows, pDataRowIndex, -1);			
	}

	/**
	 * Adds a new change in the internal list of changes.
	 * 
	 * @param iRowIndex	the row index where the change was made.
	 */
	protected void addChange(int iRowIndex)
	{
		if (ArrayUtil.indexOf(iaChangedRows, iRowIndex) < 0)
		{
			iaChangedRows = ArrayUtil.add(iaChangedRows, iRowIndex);
		}
	}
	
	/**
	 * Remove an existing change by row index.
	 * 
	 * @param iRowIndex	the row index to use.
	 */
	protected void removeChange(int iRowIndex)
	{
		if (ArrayUtil.indexOf(iaChangedRows, iRowIndex) >= 0)
		{
			iaChangedRows = ArrayUtil.remove(iaChangedRows, ArrayUtil.indexOf(iaChangedRows, iRowIndex));
		}
	}

	/**
	 * Returns the internal storage for the specified DataRow Object[].
	 *  
	 * @param pDataRowIndex	the row index.
	 * @return the internal storage for the specified DataRow Object[].
	 * @throws ModelException	if the mem sort and/or filter  fails.
	 */
	protected Object[] getDataRowInternal(int pDataRowIndex) throws ModelException
	{
		initFilterSort();
		return alStorage.get(getInternalRowIndex(pDataRowIndex));
	}
	
	/**
	 * Sets the internal storage for the specified DataRow Object[].
	 * 
	 * @param pDataRowIndex	the row index.
	 * @param pStorage		the new Object[].
	 * @throws ModelException	if the mem sort and/or filter  fails.
	 */
	protected void setDataRowInternal(int pDataRowIndex, Object[] pStorage) throws ModelException
	{
		initFilterSort();
		// #330 - if page empty, don't set the storage. Can happen in DataSource level if all details get restored in an MasterReference hierarchy.
		int i = getInternalRowIndex(pDataRowIndex);
		if (i < alStorage.size())
		{
			alStorage.set(i, pStorage);
		}
	}
	
	/**
	 * It shifts all items (row indices) by pShift (-1/+1).
	 *  
	 * @param pIntArray the int[] to shift.
	 * @param pRowIndex	all items with >= pRowIndex will be shifted. 
	 * @param pShift	the value to add/shift
	 */
	protected void shiftIndicies(int[] pIntArray, int pRowIndex, int pShift)
	{
		for (int i = 0; i < pIntArray.length; i++)
		{
			int iCurrent = pIntArray[i];
			if (iCurrent >= pRowIndex)
			{
				iCurrent += pShift;
				pIntArray[i] = iCurrent;
			}
		}
	}
	
	/**
	 * Sets a new Master DataRow, if it changes. Internal function!
	 * 
	 * @param pMasterDataRow	the new master DataRow
	 * @throws ModelException	if the column value couldn't converted 
	 */
	protected void setMasterDataRow(IDataRow pMasterDataRow) throws ModelException
	{
		drMasterDataRow = pMasterDataRow;

		// master row changed, copy master PK to Details FK Columns
		String[] saColumns = dbDataBook.getMasterReference().getColumnNames();
        int[] iaColumnIndicies = new int[saColumns.length];
        IRowDefinition rdRowDefinition = dbDataBook.getRowDefinition();
        
        // cache the column indicies
		for (int j = 0; j < iaColumnIndicies.length; j++)
		{
			iaColumnIndicies[j] = rdRowDefinition.getColumnDefinitionIndex(saColumns[j]);
		}
        
		// copy
		initFilterSort();
    	for (int i = 0; i < alStorage.size(); i++)
    	{
    		for (int j = 0; j < iaColumnIndicies.length; j++)
    		{
        		IDataType ctDataType = rdRowDefinition.getColumnDefinition(iaColumnIndicies[j]).getDataType();
        		alStorage.get(getInternalRowIndex(i))[iaColumnIndicies[j]] = ctDataType.convertAndCheckToTypeClass(drMasterDataRow.getValue(j));		    			
    		}
    	}		
	}
	
	/**
	 * Returns -1. Will/should be overridden in the derived Classes. 
	 * 
	 * @return -1.
	 * @throws ModelException see derived classes. 
	 */
	public int getEstimatedRowCount() throws ModelException
	{		
		return getRowCount();
	}	
	
	/**
	 * Will/should be overridden in the derived Classes to fetch data from the storage. 
	 * 
	 * @param pRowIndex	the row index to use.
	 * @throws ModelException see derived classes. 
	 */
	public void fetchToRow(int pRowIndex) throws ModelException
	{
	}
	
	/**
	 * Returns the row count of this MemDataPage. Internal use only.
	 * 
	 * @return the row count of this MemDataPage. Internal use only. 
	 */
	protected int getRowCountInternal()
	{
		if (iaUsedRows != null)
		{
			return iaUsedRows.length;
		}
		else
		{
			return alStorage.size();
		}
	}		
	
	/**
	 * Returns the specified DataRow without init the mem Filter/Sort and get data from
	 * storage.
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @return the specified DataRow without init the mem Filter/Sort and get data from
	 * 		   storage.
	 * @throws ModelException	if the IChangeableDataRow couldn't constructed.
	 */
	private IChangeableDataRow getDataRowWithoutSort(int pDataRowIndex) throws ModelException
	{		
		return new ChangeableDataRow(dbDataBook.getRowDefinition(),
									 alStorage.get(pDataRowIndex),
							         this,
							         pDataRowIndex);
	}
		
	/**
	 * Returns the internal / real row index of the row in the MemDataPage.
	 * Its necessary because of memory sort and filter functionality. 
	 * 
	 * @param pRowIndex	the external used row index.
	 * @return the internal / real row index of the row in the MemDataPage.
	 */
	private int getInternalRowIndex(int pRowIndex)
	{
		if (iaUsedRows == null)
		{
			return pRowIndex;
		}
		return iaUsedRows[pRowIndex];
	}
	
	/**
	 * Initialize the mem filter and sort.
	 * 
	 * @throws ModelException	if the mem sort and/or filter fails.
	 */
	private void initFilterSort() throws ModelException
	{
		if (iaUsedRows == null && dbDataBook != null)
		{
			if (dbDataBook.isMemFilter())
			{
				iaUsedRows = filter();
			}
			if (dbDataBook.isMemSort())
			{
				iaUsedRows = sort();
			}
		}		
	}
	
	/**
	 * It clears the mem filter and sort in the MemDataPage.  
	 */
	protected void clear()
	{
		iaUsedRows = null;
	}
	
	/**
	 * It filters the MemDataPage and return an int[] with DataRow indices to use for 
	 * the filter result.
	 * 
	 * @return an int[] with DataRow indices to use for the filter result.
	 * @throws ModelException	if the isFulfilled fails.
	 */
	private int[] filter() throws ModelException
	{
		ICondition cFilter = dbDataBook.getFilter();
		
		if (cFilter != null)
		{
			fetchAll();

			int[] iaResult = new int[getRowCountInternal()];
			int iCount = 0;
			for (int i = 0; i < iaResult.length; i++)
			{
				if (cFilter.isFulfilled(getDataRowWithoutSort(i)))
				{
					iaResult[iCount] = i;
					iCount++;
				}
			}
			return ArrayUtil.truncate(iaResult, iCount);
		}
		return null;
	}
	
	/**
	 * It sorts the MemDataPage and return an int[] with DataRow indices to use for 
	 * the sorted result.
	 * 
	 * @return an int[] with DataRow indices to use for the sorted result.
	 * @throws ModelException	if the compareTo fails.
	 */
	private int[] sort() throws ModelException
	{
		SortDefinition sort = dbDataBook.getSort();
		
		if (sort != null)
		{
			fetchAll();
	
			int[] iaResult = new int[getRowCountInternal()];
			for (int i = 0; i < iaResult.length; i++)
			{
				iaResult[i] = getInternalRowIndex(i);
			}
			quickSort(sort, iaResult, 0, iaResult.length - 1);
			insertionSort(sort, iaResult, 0, iaResult.length - 1);
			return iaResult;
		}
		return iaUsedRows;
	}

	/**
	 * Swap the source and target index in the result in[].
	 * 
	 * @param pResult		the result int[] with the row indices.
	 * @param pSourceIndex	the source index.
	 * @param pTargetIndex	the target index.
	 */
	private void swap(int[] pResult, int pSourceIndex, int pTargetIndex)
	{
		int iTemp = pResult[pSourceIndex];
		pResult[pSourceIndex] = pResult[pTargetIndex];
		pResult[pTargetIndex] = iTemp;
	}

	/**
	 * It compares two DataRow with the specified row indices and SortDefinition.
	 * 
	 * @param pSort				the SortDefinition to use.
	 * @param pSourceRowIndex	the source row index of the DataRow to compare
	 * @param pTargetRowIndex	the target row index of the DataRow to compare
	 * @return it compares two DataRow with the specified row indices and SortDefinition and return like the Compareable interface.
	 * @throws ModelException	 if the compareTo fails.
	 */
	private int compareTo(SortDefinition pSort, int pSourceRowIndex, int pTargetRowIndex) throws ModelException
	{
		return getDataRowWithoutSort(pSourceRowIndex).compareTo(getDataRowWithoutSort(pTargetRowIndex), pSort);
	}

	/**
	 * QuickSort implementation for the MemDataPage.
	 * 
	 * @param pSort			the SortDefinition to use.
	 * @param pResult		the result int[] with the row indices.
	 * @param pFromIndex	the from index to sort.
	 * @param pToIndex		the to index to sort.
	 * @throws ModelException	 if the compareTo fails.
	 */
	private void quickSort(SortDefinition pSort, int[] pResult, int pFromIndex, int pToIndex) throws ModelException
	{
		if (pToIndex - pFromIndex > 4)
		{
			int i = (pToIndex + pFromIndex) / 2;
			
			if (compareTo(pSort, pResult[pFromIndex], pResult[i]) > 0)
			{
				swap(pResult, pFromIndex, i);
			}
			if (compareTo(pSort, pResult[pFromIndex], pResult[pToIndex]) > 0)
			{
				swap(pResult, pFromIndex, pToIndex);
			}
			if (compareTo(pSort, pResult[i], pResult[pToIndex]) > 0)
			{
				swap(pResult, i, pToIndex);
			}

			int j = pToIndex - 1;
			
			swap(pResult, i, j);
			i = pFromIndex;
			
			int v = pResult[j];
			
			while (true)
			{
				while (compareTo(pSort, pResult[++i], v) < 0) 
				{ /* . */ }
				
				while (compareTo(pSort, pResult[--j], v) > 0)
				{ /* . */ }
				
				if (j < i)
				{
					break;
				}
				swap(pResult, i, j);
			}
			
			swap(pResult, i, pToIndex - 1);
			quickSort(pSort, pResult, pFromIndex, j);
			quickSort(pSort, pResult, i + 1, pToIndex);
		}
	}

	/**
	 * InsertionSort implementation for the MemDataPage.
	 * 
	 * @param pSort			the SortDefinition to use.
	 * @param pResult		the result int[] with the row indices.
	 * @param pFromIndex	the from index to sort
	 * @param pToIndex		the to index to sort
	 * @throws ModelException	 if the compareTo fails.
	 */
	private void insertionSort(SortDefinition pSort, int[] pResult, int pFromIndex, int pToIndex) throws ModelException
	{
		for (int i = pFromIndex + 1; i <= pToIndex; i++)
		{
			int v = pResult[i];
			int j = i;
			while (j > pFromIndex && compareTo(pSort, pResult[j - 1], v) > 0)
			{
				pResult[j] = pResult[j - 1];
				j--;
			}
			pResult[j] = v;
		}
	}	
	
} 	// MemDataPage
