/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.10.2008 - [RH] - bug fixes, toString() optimized, NO_SELECTED_ROW optimized
 * 12.10.2008 - [RH] - bug fixes, sync()
 *                     writebackIsolationsLevel support added
 * 02.11.2008 - [RH] - InsertEnabled, UpdateEnabled, DeleteEnabled, ReadOnly added
 *                     getRowCount, SelectedDataRow support added
 *                     comments added
 *                     bugs fixed
 * 11.11.2008 - [RH] - use instead SetSelectedRow -> SetSelectRowInternal with 
 *                     BEFORE & AFTER SELECTED Event-> bugs fixed
 *                     DeleteCascade added, notifyMasterReferenceColumnsChanged(IDataRow) add
 *                     MasterRow set in MemDataPage   
 * 17.11.2008 - [RH] - get/setSelectionMode added; DefaultDeselected removed   
 *                     delete bug with IDataBook.DATAROW WritebackIsolationLevel fixed  
 * 16.04.2009 - [RH] - remove/add/getDetailDataBooks moved to IDataBook.          
 *                     checks in open for PK columns and MasterReferenceDefinition added   
 * 27.04.2009 - [RH] - interface review changes tested and fixed
 * 28.04.2009 - [RH] - javadoc added, NLS removed         
 * 13.05.2009 - [RH] - sort/filter in memory added                           
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 02.07.2009 - [JR] - invokeDataBookListeners: finally block [BUGFIX]
 * 13.10.2009 - [RH] - WriteBackIsolationLevel.DATASOURCE is considered at saveAllDataRows() [BUGFIX]
 * 14.10.2009 - [RH] - In AfterDeleting event is the selected row corrected in WriteBackIsolationLevel.DATASOURCE [BUGFIX]
 * 15.10.2009 - [RH] - Insert over Master - Detail reference in WriteBackIsolationLevel.DATASOURCE fixed (UID missed) [BUGFIX]
 *            - [JR] - clearFilterSortInMemDataPages: isOutOfSync called
 *                   - clearFilterSortinMemDataPages renamed to clearFilterSortInMemDataPages
 * 19.03.2010 - [RH] - lock before delete only happens, if isLockAndRefetchEnabled() == true.
 * 25.03.2010 - [JR] - #92: createNewRow implemented
 * 21.07.2010 - [RH] - setTreePath = null if emptyPage is set.
 * 29.09.2010 - [RH] - countRows renamed to getEstimatedRowCount()
 * 15.10.2010 - [RH] - detailChanged renamed to notifyDetailChanged        
 *                     #186: notifyDetailedChanged changed, to set the DetailedChanged state correct. 
 *                           store(), restore() call now setDetailsChangedInMasterBook()       
 * 28.10.2010 - [RH] - #197: insert bug with DATASOURCE level with master/detail. reference columns doesn't copy to details.
 *                           rehash is moved for getDataPageIntern to notifyMasterChanged. UID is correct reset to null.
 *                           updates on null PK (reference columns) not supported anymore. Won't work correctly. 
 * 28.10.2010 - [HM] - open() sets the open state now correctly -> at the end of the open, and not before
 *                     setValues (null, xxx) uses now all columns as default.
 * 15.02.2011 - [RH] - #286: no DataPage to master row exists, then return null -> no changes can exits - performance tuning in getChangedDataRows()  
 * 			           #288: master DataRow rehash not working in DATASOUCE Level over more then 1 hierarchy ! - bug fixed.                  
 * 24.02.2011 - [RH] - #290: if the master DataBook is in DATASOURCE level && Inserting() no fetch is necessary, because no rows can exists!
 * 06.03.2011 - [JR] - #115: 
 *                     * open: check allowed values
 *                     * createNewRow: use default value    
 * 21.03.2011 - [RH] - #315: saveSelectedRow() bug with more then one leave with isInsertung state in DATASOURCE level
 * 31.03.2011 - [JR] - #318: setRowDefinition() clear/sets all controls to the row definition 
 * 31.03.2011 - [RH] - #163 - IChangeableDataRow should support isWritableColumnChanged 
 * 13.04.2011 - [RH] - #336 - First check if a DataPage with the UID exists. -> Reuse new DataPages in DATASOURCE level.
 * 12.05.2011 - [RH] - #347 - DataBook in DATASOURCE level should in notifyMasterChanged always check if rehash is required !
 *                     #348 - sync() fails if a seljoined MemDataBook has the TreePath wrong - null check added.
 * 13.05.2011 - [RH] - #350 - MemDataBook should remove all details in saveSelectedRow(), if a row is deleted 
 * 23.05.2011 - [RH] - #357 - setSort on MemDataBook with setMemSort(true), doesn't setSelectedRow correct after sort
 * 30.05.2011 - [HM] - #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
 * 30.05.2011 - [HM] - #375 - delete, cancelEditing missing 
 * 30.05.2011 - [HM] - #369 - delete all rows from a databook, Master changed has to be set.
 * 30.05.2011 - [HM] - #376 - restore row fails in DATASOURCE level with more then one detail level with isInserting rows 
 * 31.05.2011 - [RH] - #377 - MemDataBook has invalid javadoc 
 * 01.06.2011 - [RH] - #378 - Exception in AfterDeleted Event, if an insert() is called in the user event
 * 30.06.2011 - [RH] - #405 - NullPointer Exception in notifyMasterChanged(), if parent IDataBook out of sync
 * 14.07.2011 - [RH] - #424 - MemDataBook.setRowDefintion(null) should addControls
 *                     #425 - MemDataBook close should not close the detail DataBooks.
 * 31.10.2011 - [RH] - #492 - MemDataBook.restoreAllRows set wrong SelectedRow :: save the current row after the restore e.g is Inserting....
 * 02.11.2011 - [RH] - #495 - MemDataBook removeDataPage should setTreePath(null) 
 * 13.11.2011 - [RH] - #505 - delete should should check the WritebackIsolation level
 * 15.11.2011 - [RH] - #507 - restore in MemDataBook delete() has be done after the BEFORE_DELETING event
 * 17.11.2011 - [JR] - #509 - setReadOnlyWithoutSave implemented 
 * 10.12.2012 - [JR] - #615 - setFilter should not save in mem filter mode and datasource writeback isolation level
 */
package com.sibvisions.rad.model.mem;

import java.lang.ref.WeakReference;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.Hashtable;

import javax.rad.genui.UIFactoryManager;
import javax.rad.genui.celleditor.UIChoiceCellEditor;
import javax.rad.genui.celleditor.UILinkedCellEditor;
import javax.rad.model.ColumnDefinition;
import javax.rad.model.IChangeableDataRow;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IDataSource;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.RowDefinition;
import javax.rad.model.SortDefinition;
import javax.rad.model.TreePath;
import javax.rad.model.condition.ICondition;
import javax.rad.model.event.DataBookEvent;
import javax.rad.model.event.DataBookEvent.ChangedType;
import javax.rad.model.event.DataBookHandler;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.model.ui.IControl;
import javax.rad.ui.celleditor.IChoiceCellEditor;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>MemDataBook</code> is a storage independent table, and handles all operations
 * to load, save and manipulate table oriented data. <br>
 * An <code>MemDataBook</code> has at least one <code>MemDataPage</code> to store all 
 * <code>DataRow</code>'s of this <code>MemDataBook</code>. If an <code>MemDataBook</code> 
 * has detail <code>IDataBook</code>'s, it handles for each detail <code>IDataBook</code> 
 * the <code>IDataPage</code>'s. Thats necessary because every change of the selected
 * <code>IDataRow</code> needs to initiate the change of the corresponding details. 
 * So the <code>IDataBook</code> adds the appropriate <code>IDataPage</code>'s into the 
 * detail <code>IDataBook</code>'s.<br>
 * The <code>MemDataBook</code> is also a <code>IChangeableDataRow</code>, the selected row 
 * of the <code>MemDataBook</code>.
 * 
 * @see com.sibvisions.rad.model.mem.MemDataPage
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IRowDefinition
 * @see javax.rad.model.IChangeableDataRow
 * @see javax.rad.model.IDataSource
 * 
 * @author Roland Hrmann
 */
public class MemDataBook extends ChangeableDataRow 
                         implements IDataBook 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** constant value for 1. */
	private static final Integer ONE = Integer.valueOf(1);

	/** 
	 * The internal unique for the MemDataBook. Its used for new DataRow's as identifier 
	 * till the get a unique Primary Key value. 
	 */
	private static int iUID = 0;
	
	
	/** The TreePath inside self joined IDataBook's. */
	private TreePath		treePath = new TreePath();
	
	/** It holds the current SelectionMode of the MemDataBook. Default is SelectionMode.CURRENT_ROW. */
	private SelectionMode   iSelectionMode = SelectionMode.CURRENT_ROW;
	
	/** The Filter of the MemDataBook. */
	private ICondition		cFilter;
	/** The Sort of the MemDataBook. */
	private SortDefinition	sSort;
	
	/** The ReferenceDefinition to the master IDataBook. */
	private ReferenceDefinition rdMasterReference;
	
	/** The ReferenceDefinition to the master IDataBook. */
	private ReferenceDefinition rdTreeRootReference;

	/** The ReferenceDefinition to the master IDataBook. */
	private String[] saTreeRootMasterColumnNames;

	/** The write back isolation level. */
	private WriteBackIsolationLevel	iWritebackIsolationLevel;
	
	/** The <code>ArrayUtil</code> of all detail <code>IDataBooks</code>. */
	private ArrayUtil<WeakReference<IDataBook>>	auDetailDataBooks;
	
	/** The DataSource of the MemDataBook. */
	private IDataSource	dsDataSource;

	/** It stores for the lazy reload, the last selected Row PK column values. */
	private IDataRow auStoredSelection;
	
	/** 
	 * The current MemDataPage of the MemDataBook. 
	 * It gets synchronized at the first use. (lazy load) 
	 */
	private MemDataPage	dpCurrentDataPage;
	
	/** The Empty MemDataPage, which is used as long no DataRows are available. */
	private MemDataPage	dpEmptyDataPage;	

	/** 
	 * The Hashtable of all MemDataPages of the MemDataBook. It holds for each master DataRow
	 * a MemDataPage. If the master DataRow is Inserting (new), the MemDataPage is stored
	 * under the UID. 
	 */
	private Hashtable<Object, MemDataPage>   htDataPagesCache   = new Hashtable<Object, MemDataPage>();
	
	/** The <code>EventHandler</code> for before selected event. */
	private DataBookHandler	eventBeforeRowSelected;
	/** The <code>EventHandler</code> for after selected event. */
	private DataBookHandler	eventAfterRowSelected;
	/** The <code>EventHandler</code> for before inserting event. */
	private DataBookHandler	eventBeforeInserting;
	/** The <code>EventHandler</code> for after inserting event. */
	private DataBookHandler	eventAfterInserting;
	/** The <code>EventHandler</code> for before inserted event. */
	private DataBookHandler	eventBeforeInserted;
	/** The <code>EventHandler</code> for after inserted event. */
	private DataBookHandler	eventAfterInserted;
	/** The <code>EventHandler</code> for before updating event. */
	private DataBookHandler	eventBeforeUpdating;
	/** The <code>EventHandler</code> for after updating event. */
	private DataBookHandler	eventAfterUpdating;
	/** The <code>EventHandler</code> for before updated event. */
	private DataBookHandler	eventBeforeUpdated;
	/** The <code>EventHandler</code> for after updated event. */
	private DataBookHandler	eventAfterUpdated;
	/** The <code>EventHandler</code> for before deleting event. */
	private DataBookHandler	eventBeforeDeleting;
	/** The <code>EventHandler</code> for after deleting event. */
	private DataBookHandler	eventAfterDeleting;
	/** The <code>EventHandler</code> for before deleted event. */
	private DataBookHandler	eventBeforeDeleted;
	/** The <code>EventHandler</code> for after deleted event. */
	private DataBookHandler	eventAfterDeleted;
	/** The <code>EventHandler</code> for before restore event. */
	private DataBookHandler	eventBeforeRestore;
	/** The <code>EventHandler</code> for after restore event. */
	private DataBookHandler	eventAfterRestore;
	/** The <code>EventHandler</code> for before reload event. */
	private DataBookHandler	eventBeforeReload;
	/** The <code>EventHandler</code> for after reload event. */
	private DataBookHandler	eventAfterReload;
	/** The <code>EventHandler</code> for before selected event. */
	private DataBookHandler	eventBeforeColumnSelected;
	/** The <code>EventHandler</code> for after selected event. */
	private DataBookHandler	eventAfterColumnSelected;
	
	/** Name of the MemDataBook. */
	private String			sName;
	/** The current selected row index. */
	private String			sSelectedColumn	= null;
	
	/**
	 * The read a head row count of the <code>RemoteDataBook</code>. This means how many rows are
	 * read ahead in the <code>MemDataPage</code>.
	 */
	private int	iReadAheadRowCount	= 10;
	/** The current selected row index. */
	private int	iSelectedRowIndex	= -1;

	/** 
	 * It determines that the master has changed and this IDataBoolk needs to synchronize
	 * to the current MemDataPage and selected row depending on the Selection Mode.
	 */
	private boolean 		bMasterChanged 	= false;
	/** It determines that insert()'s are allowed. */
	private boolean			bInsertEnabled 	= true;
	/** It determines that update()'s (setValueXXX) are allowed. */
	private boolean			bUpdateEnabled 	= true;
	/** It determines that delete()'s are allowed. */
	private boolean			bDeleteEnabled 	= true;
	/** It determines that lock on refetch is enabled. */
	private boolean			bLockAndRefetchEnabled = false;
	/** It determines if the MemDataBook is read only. ->all change operations are allowed. */
	private boolean			bReadOnly 		= false;
	/** It defines if all details will be before delete cascade. */
	private boolean         bDeleteCascade 	= true;
	/** Determines if the sort should be made in memory. */
	private boolean			bMemSort 		= true;
	/** Determines if the filter should be made in memory. */
	private boolean			bMemFilter 		= true;
	/** Determines if the RemoteDataBook should write back the changes to the AbstractStorage. */
	private boolean bWritebackEnabled = true;
	/** It determines if the MemDataBook is open. */
	private boolean			bIsOpen			= false;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Contructs a new <code>MemDataBook</code> with an empty {@link RowDefinition}.
	 */
	public MemDataBook()
	{
	}

	/**
	 * Constructs a new <code>MemDataBook</code> with the given {@link RowDefinition}.
	 * 
	 * @param pRowDefinition the row definition for the data book
	 */
	public MemDataBook(RowDefinition pRowDefinition)
	{
		super(pRowDefinition);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook getDataBook()
	{
		return this;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getName()
	{
		return sName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setName(String pName) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks! - " + getName());			
		}				
		sName = pName;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setWritebackIsolationLevel(WriteBackIsolationLevel pIsolationLevel)
	{
		iWritebackIsolationLevel = pIsolationLevel;
	}

	/**
	 * {@inheritDoc}
	 */
	public WriteBackIsolationLevel getWritebackIsolationLevel()
	{	
		if (iWritebackIsolationLevel == null)
		{
			if (getDataSource() != null)
			{
				return getDataSource().getWritebackIsolationLevel();
			}
			return WriteBackIsolationLevel.DATA_ROW;
		}
		return iWritebackIsolationLevel;
	}	
		
	/**
	 * {@inheritDoc}
	 */
	public void setRowDefinition(IRowDefinition pRowDefinition) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		
		IControl[] controls = null;
		
		//cleanup old row definition
		if (rdRowDefinition != null)
		{
			controls = getControls();
			
			for (int i = 0; i < controls.length; i++)
			{
				rdRowDefinition.removeControl(controls[i]);
			}
		}
		
		rdRowDefinition = pRowDefinition;
		
		// #424 - MemDataBook.setRowDefintion(null) should addControls
		if (rdRowDefinition == null)
		{
			rdRowDefinition = new RowDefinition();
		}
		
		//configure new row definition
		if (rdRowDefinition != null)
		{
			if (controls == null)
			{
				controls = getControls();
			}
			
			for (int i = 0; i < controls.length; i++)
			{
				rdRowDefinition.addControl(controls[i]);
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public IDataSource getDataSource()
	{
		return dsDataSource;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDataSource(IDataSource pDataSource) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");
		}
		dsDataSource = pDataSource;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getMasterReference()
	{
		return rdMasterReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setMasterReference(ReferenceDefinition pMasterReference) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		rdMasterReference = pMasterReference;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isSelfJoined()
	{
		return rdMasterReference != null && rdMasterReference.getReferencedDataBook() == this;
	}

	/**
	 * {@inheritDoc}
	 */
	public IDataPage getDataPage(TreePath pTreePath) throws ModelException
	{
		if (rdTreeRootReference == null)
		{
			return getDataPage(null, pTreePath);
		}
		else
		{
			return getDataPage(rdTreeRootReference.getReferencedDataBook(), pTreePath);
		}
	}

	/**
	 * Returns the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns() created form a root row.
	 * 
	 * @param pRootDataRow	the full/complete master IDataRow to use.
	 * @return internal the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * @throws ModelException	if the master row couldn't created.
	 */
	private IDataRow getMasterDataRowFromRootDataRow(IDataRow pRootDataRow) throws ModelException
	{
		IDataRow masterDataRow = rdMasterReference.getReferencedDataBook().createEmptyRow(rdMasterReference.getReferencedColumnNames());
		
		if (rdTreeRootReference != null && pRootDataRow != null)
		{
			masterDataRow.setValues(saTreeRootMasterColumnNames, pRootDataRow.getValues(rdTreeRootReference.getReferencedColumnNames()));
		}
		
		return masterDataRow;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDataPage getDataPage(IDataRow pRootRow, TreePath pTreePath) throws ModelException
	{
		IDataPage dpDataPage = getDataPageWithRootRow(pRootRow);
		
		if (pTreePath != null)
		{
			for (int i = 0; i < pTreePath.length(); i++)
			{
				// #348 - sync() fails if a seljoined MemDataBook has the TreePath wrong 
				IDataRow masterRow = dpDataPage.getDataRow(pTreePath.get(i));
				if (masterRow == null)
				{
					return null;
				}
				else
				{
					dpDataPage = getDataPageIntern(masterRow);
				}
			}
		}

		return dpDataPage;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataPage getDataPageWithRootRow(IDataRow pRootDataRow) throws ModelException
	{
		// if no master reference exits, then, don't call getDataPageIntern!!!!
		if (getMasterReference() == null)
		{
			sync();
			return dpCurrentDataPage;
		}		
		return getDataPageIntern(getMasterDataRowFromRootDataRow(pRootDataRow));
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getRootReference()
	{
		return rdTreeRootReference;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setRootReference(ReferenceDefinition pTreeRootReference) throws ModelException
	{
		if (isOpen())
		{
			throw new ModelException("It's not allowed on open DataBooks!");			
		}
		rdTreeRootReference = pTreeRootReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public TreePath getTreePath()
	{
		try
		{
			sync();
		}
		catch (ModelException e)
		{
			// Ignore it!
		}
		return treePath;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setTreePath(TreePath pTreePath) throws ModelException
	{
		if (!treePath.equals(pTreePath) || bMasterChanged)
		{
			sync();
			
			invokeSaveEditingControls();
			
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			
			saveDataRowLevel(null);

			if (eventBeforeRowSelected != null)
			{
				eventBeforeRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_ROW_SELECTED, drOld));
			}

			if (pTreePath == null)
			{
				treePath = new TreePath();
			}
			else
			{
				treePath = pTreePath;
			}
			
			dpCurrentDataPage = (MemDataPage)getDataPage(treePath);
			
			if (getSelectionMode() == SelectionMode.DESELECTED)
			{
				setSelectedRowInternal(-1);
			}
			else
			{
				setSelectedRowInternal(0);
			}
			
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void notifyMasterChanged()
	{
		try
		{				
			// #347 - DataBook in DATASOURCE level should in notifyMasterChanged always check if rehash is required !
			// if master row is inserted and saved, then rehash!
			if (rdMasterReference != null 
					&& rdMasterReference.getReferencedDataBook().getUID() != null
					&& !rdMasterReference.getReferencedDataBook().isInserting())
			{
				// rehash current DataPage, because PK changed after insert in Master
				// #405 - NullPointer Exception in notifyMasterChanged(), if parent IDataBook out of sync
				if (dpCurrentDataPage != null && rdMasterReference.getReferencedDataBook().getUID() != null)
				{
					htDataPagesCache.remove(rdMasterReference.getReferencedDataBook().getUID());
	
					// [HM] Force refetch of the Detail DataPage, if it is not changed.
					if (dpCurrentDataPage.getChangedDataRows().length == 0)
					{
						// [HM] enable selecting current datarow!
						handleStoreSelection(iSelectionMode);
						
						dpCurrentDataPage = null;
					}
					else
					{
						IDataRow drMaster = getMasterDataRow(rdMasterReference.getReferencedDataBook());					
						dpCurrentDataPage.setMasterDataRow(drMaster);
						htDataPagesCache.put(drMaster, dpCurrentDataPage);
					}
				}					
			}
		}
		catch (ModelException e)
		{
			throw new RuntimeException(e);
		}
		
		if (!bMasterChanged)
		{
			try
			{				
				// master changed is only allowed to be set after rehashing, because we have to
				// prevent changing the currentPage in sync.			
				bMasterChanged = true;					

				if (eventAfterReload != null)
				{
					eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
				}				
			}
			catch (ModelException e)
			{
				throw new RuntimeException(e);
			}
			
			invokeMasterChangedDetailsListeners();
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void removeDataPage(IDataRow pMasterDataRow, TreePath pTreePath) throws ModelException
	{
		IDataRow  drMaster = null;
		IDataPage dpCurrent = null;
		
		if (isSelfJoined())
		{
//TODO RH different fix !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!			
			// #495 - MemDataBook removeDataPage should setTreePath(null) 
			//if (treePath == null || treePath.length() != 0)
			//{
			//	setTreePath(new TreePath());
			//}
			
			if (pTreePath == null)
			{
				pTreePath = new TreePath();
			}
			IDataRow drRoot = null;
			if (rdTreeRootReference != null)
			{
				drRoot = rdTreeRootReference.getReferencedDataBook();
			}
			
			if (hasDataPage(drRoot, pTreePath))
			{				
				dpCurrent = getDataPage(pTreePath);
				drMaster  = dpCurrent.getMasterDataRow();
				
				for (int i = 0; i < dpCurrent.getRowCount(); i++)
				{
					removeDataPage(null, pTreePath.getChildPath(i));				
				}
			}
			else
			{
				return;
			}
		}
		else
		{
			if (hasDataPage(pMasterDataRow))
			{			
				dpCurrent = getDataPage(pMasterDataRow);
				drMaster = dpCurrent.getMasterDataRow();
			}
			else
			{
				return;
			}
		}	
		
		for (int i = 0; i < dpCurrent.getRowCount(); i++)
		{
			IDataRow drCurrent = dpCurrent.getDataRow(i);
			// delete details first
			if (auDetailDataBooks != null)
			{
				for (int j = 0, size = auDetailDataBooks.size(); j < size; j++)
				{
					IDataBook dataBook = auDetailDataBooks.get(j).get();
					if (dataBook != null)
					{
		        		dataBook.removeDataPage(drCurrent, pTreePath);
					}
		        }					
			}
		}
				
		if (drMaster instanceof IChangeableDataRow && ((IChangeableDataRow)drMaster).getUID() != null)
		{
			htDataPagesCache.remove(((IChangeableDataRow)drMaster).getUID());
		}
		else 
		{
			htDataPagesCache.remove(drMaster);
		}
		if (dpCurrent == dpCurrentDataPage)
		{
			dpCurrentDataPage = null;
			
			// #369 - delete all rows from a databook
			// Master change has to be set, to sync the page!!!
			bMasterChanged = true;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public SelectionMode getSelectionMode()
	{
		return iSelectionMode;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSelectionMode(SelectionMode pSelectionMode)
	{
		iSelectionMode = pSelectionMode;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isInsertEnabled()
	{
		return bInsertEnabled && !isReadonly() && dpCurrentDataPage != dpEmptyDataPage;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setInsertEnabled(boolean pInsertEnabled)
	{
		bInsertEnabled = pInsertEnabled;
		
		invokeRepaintListeners();
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isUpdateEnabled() throws ModelException
	{
		return bUpdateEnabled && !isReadonly() && getSelectedRow() >= 0 && !isDeleting();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setUpdateEnabled(boolean pUpdateEnabled)
	{
		bUpdateEnabled = pUpdateEnabled;

		invokeRepaintListeners();
	}

	/**
	 * Returns true if lock and refetch is enabled.
	 * 
	 * @return true if lock and refetch is enabled.
	 */
	public boolean isLockAndRefetchEnabled()
	{
		return bLockAndRefetchEnabled;
	}

	/**
	 * Sets whether if lock and refetch is enabled.
	 * 
	 * @param pLockAndRefetchEnabled true if lock and refetch is enabled.
	 */
	public void setLockAndRefetchEnabled(boolean pLockAndRefetchEnabled)
	{
		bLockAndRefetchEnabled = pLockAndRefetchEnabled;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleteEnabled() throws ModelException
	{
		return bDeleteEnabled && !isReadonly() && getSelectedRow() >= 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDeleteEnabled(boolean pDeleteEnabled)
	{
		bDeleteEnabled = pDeleteEnabled;

		invokeRepaintListeners();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isReadonly()
	{
		return bReadOnly;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setReadOnly(boolean pReadOnly) throws ModelException
	{
		if (isOpen() && !bReadOnly && pReadOnly)
		{
			saveAllRows();
		}
		
		setReadOnlyWithoutSave(pReadOnly);
	}
	
	/**
	 * Returns the count of rows which are read ahead from the IDBAccess.
	 * 
	 * @return the count of rows which are read ahead from the IDBAccess.
	 */
	public int getReadAhead()
	{
		return iReadAheadRowCount;
	}

	/**
	 * Sets the count of rows which are read ahead from the IDBAccess.
	 * 
	 * @param pReadAheadRowCount
	 *            the row count to read ahead from the IDBAccess
	 */
	public void setReadAhead(int pReadAheadRowCount)
	{
		if (pReadAheadRowCount >= 0)
		{
			iReadAheadRowCount = pReadAheadRowCount;
		}
	}			
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isOpen()
	{
		return bIsOpen;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void open() throws ModelException
	{
		if (!isOpen())
		{
			if (getName() == null)
			{
				throw new ModelException("DataBook Name is null!");				
			}

			if (rdRowDefinition.getColumnCount() == 0)
			{
				throw new ModelException("RowDefintion in DataBook contains no columns!");				
			}
			
			//use default values for choice cell editors
			if (UIFactoryManager.getFactory() != null)
			{
				ColumnDefinition coldef;
				
				for (int i = 0, anz = rdRowDefinition.getColumnCount(); i < anz; i++)
				{
					coldef = rdRowDefinition.getColumnDefinition(i);
					Object[] allowedValues = coldef.getAllowedValues();
					
					if (coldef.getDataType().getCellEditor() == null && allowedValues != null && allowedValues.length > 0)
					{
						IChoiceCellEditor editor = UIChoiceCellEditor.getDefaultChoiceCellEditor(allowedValues);
						
						if (editor != null)
						{
							coldef.getDataType().setCellEditor(editor);
						}
						else
						{
							ColumnDefinition cdEnum = coldef.clone();
							cdEnum.setAllowedValues(null);
							String columnName = cdEnum.getName();
							
							MemDataBook mdbEnum = new MemDataBook();
							mdbEnum.setName("enum_" + columnName);
							mdbEnum.getRowDefinition().addColumnDefinition(cdEnum);
							mdbEnum.open();
							
							for (int j = 0; j < allowedValues.length; j++)
							{
								mdbEnum.insert(false);
								mdbEnum.setValue(columnName, allowedValues[j]);
							}
							mdbEnum.saveAllRows();
							
							coldef.getDataType().setCellEditor(new UILinkedCellEditor(
									new ReferenceDefinition(new String[] {columnName}, mdbEnum, new String[] {columnName})));
						}
					}
				}
			}
			
			// check PK Columns in RowDefinition
			String[] saPKColumnNames = rdRowDefinition.getPrimaryKeyColumnNames();		
			for (int i = 0; saPKColumnNames != null && i < saPKColumnNames.length; i++)
			{
				if (rdRowDefinition.getColumnDefinitionIndex(saPKColumnNames[i]) < 0)
				{
					throw new ModelException("Primary key column '" + saPKColumnNames[i] + 
							                 "' doesn't exist in RowDefinition of this DataBook!");
				}
			}

			ReferenceDefinition refdefMaster = getMasterReference();
			if (refdefMaster != null)
			{
				IDataBook dbMaster = refdefMaster.getReferencedDataBook();
				if (!isSelfJoined() && !dbMaster.isOpen())
				{
					throw new ModelException("The master DataBook has to be opened first! - " + dbMaster.getName());
				}
				// check if Master Reference Definition is correct.
				String[] saColumns = refdefMaster.getColumnNames();
				for (int i = 0; i < saColumns.length; i++)
				{
					if (rdRowDefinition.getColumnDefinitionIndex(saColumns[i]) < 0)
					{
						throw new ModelException("Column '" + saColumns[i] + 
								                 "' doesn't exist in detail DataBook '" + getName() + "' !");
					}
				}
				String[] saRefColumns = refdefMaster.getReferencedColumnNames();
				for (int i = 0; i < saRefColumns.length; i++)
				{
					if (dbMaster.getRowDefinition().getColumnDefinitionIndex(saRefColumns[i]) < 0)
					{
						throw new ModelException("Column '" + saRefColumns[i] + 
								                 "' doesn't exist in master DataBook '" + 
								                 dbMaster.getName() + "' !");
					}
				}
				if (isSelfJoined())
				{
					if (getRootReference() != null)
					{
						String[] columnNames = getRootReference().getColumnNames();
						saTreeRootMasterColumnNames = new String[columnNames.length]; 
						
						for (int i = 0; i < columnNames.length; i++)
						{
							int index = ArrayUtil.indexOf(getMasterReference().getColumnNames(), columnNames[i]);
							
							if (index < 0)
							{
								throw new ModelException("Column name " + columnNames[i] + " of root reference is not in master reference!");
							}
							saTreeRootMasterColumnNames[i] = getMasterReference().getReferencedColumnNames()[index];
						}
						
						getRootReference().getReferencedDataBook().addDetailDataBook(this);
					}
					else
					{
						saTreeRootMasterColumnNames = null;
					}
				}
				else
				{
					refdefMaster.getReferencedDataBook().addDetailDataBook(this);
					refdefMaster.setConnected();
				}
			}
			
			if (getDataSource() != null)
			{
				if (!getDataSource().isOpen())
				{
					throw new ModelException("DataSource is not open!");								
				}						
				getDataSource().addDataBook(this);
			}
			
			rdRowDefinition.addDataBook(this);
			
			iSelectedRowIndex = -1;
			bMasterChanged = true;
			bIsOpen = true;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void close()
	{
		if (isOpen())
		{
			bIsOpen 		          = false;
			dpCurrentDataPage	      = null;
			htDataPagesCache.clear();							
			iSelectedRowIndex 	      = -1;
			oaStorage				  = null;

			if (getMasterReference() != null)
			{
				getMasterReference().getReferencedDataBook().removeDetailDataBook(this);
			}
			
			if (dsDataSource != null)
			{
				dsDataSource.removeDataBook(this);
			}
			if (rdRowDefinition != null)
			{
				rdRowDefinition.removeDataBook(this);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized int getRowCount() throws ModelException
	{
		if (isOpen())
		{
			sync();

			return dpCurrentDataPage.getRowCount();
		}
		return 0;
	}


	/**
	 * {@inheritDoc}
	 */
	public synchronized int getSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			return -1;			
		}
		sync();
		return iSelectedRowIndex;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSelectedRow(int pDataRowIndex) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}

		sync();
		
		if (iSelectedRowIndex != pDataRowIndex)
		{		
			invokeSaveEditingControls();
			
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			
			saveDataRowLevelDetails(null);

			if (eventBeforeRowSelected != null)
			{
				eventBeforeRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_ROW_SELECTED, drOld));
			}

			setSelectedRowInternal(pDataRowIndex);
			
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized String getSelectedColumn() throws ModelException
	{
		// sync() not mandatory !
		return sSelectedColumn;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSelectedColumn(String pColumnName) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}

		sync();
		
		if (pColumnName != null)
		{
			rdRowDefinition.getColumnDefinition(pColumnName); // Check if the column exists.
		}
		
		if ((sSelectedColumn == null && pColumnName != null)
				|| (sSelectedColumn != null && !sSelectedColumn.equals(pColumnName)))
		{
			if (eventBeforeColumnSelected != null)
			{
				eventBeforeColumnSelected.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_COLUMN_SELECTED, null));
			}
			
			sSelectedColumn = pColumnName;
			
			invokeRepaintListeners();
			
			if (eventAfterColumnSelected != null)
			{
				eventAfterColumnSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_COLUMN_SELECTED, null));
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isDeleteCascade()
	{
		return bDeleteCascade;
	}

	/**
	 * {@inheritDoc}
	 */
	public void setDeleteCascade(boolean pDeleteCascade)
	{
		bDeleteCascade = pDeleteCascade;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized int insert(boolean pBeforeRow) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}		
		sync();
		if (dpCurrentDataPage == dpEmptyDataPage)
		{
			throw new ModelException("Master DataBook has no selected row! - " + 
					                 getMasterReference().getReferencedDataBook().getName());
		}
		if (!isInsertEnabled())
		{
			throw new ModelException("Insert isn't allowed!");			
		}		
		
		invokeSaveEditingControls();

		saveDataRowLevel(null);

		// #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
		// saveDataRowLevel can set the dpCurrentDataPage to null, because notifyMasterChange rehash the details, and force lazy refetch!
		sync();

		// #374 - If an insert happens, after that more rows have to be fetched, then wrong rows will be fetched
		// if first insert without anything fetched, begin fetch, to prevent that a 
		// cursor deliveres the new inserted row, what happens, if the fetch is started after inserting a row.
		if (!dpCurrentDataPage.isAllFetched() && dpCurrentDataPage.getRowCount() == 0)
		{
			getDataRow(0);
		}
		
		IDataRow drOld = getDataRow(iSelectedRowIndex);			
		if (eventBeforeInserting != null)
		{
			eventBeforeInserting.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_INSERTING, null));
		}
		
		int iDataRowIndex = insertInternal(pBeforeRow);
		
		setSelectedRowInternal(iDataRowIndex);
		copyMasterColumnsToCurrentDetail();

		if (eventAfterInserting != null)
		{
			eventAfterInserting.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_INSERTING, null));
		}
		if (eventAfterRowSelected != null)
		{
			eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
		}
		
		return iDataRowIndex;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void update() throws ModelException
	{
		if (!isUpdating() && !isInserting() && !isDeleting())
		{
			if (!isOpen())
			{
				throw new ModelException("DataBook isn't open! - " + sName);			
			}
			sync();
			if (!isUpdateEnabled())
			{
				throw new ModelException("Update isn't allowed!");			
			}				
			
			saveDataRowLevel(this);

			IDataRow drOld = getDataRow(iSelectedRowIndex); 
			if (eventBeforeUpdating != null)
			{
				eventBeforeUpdating.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_UPDATING, drOld));
			}
			
			if (isLockAndRefetchEnabled())
			{
				executeLockAndRefetch();
			}
			
			setUpdating();
			setDetailsChangedInMasterBook(); 

			if (eventAfterUpdating != null)
			{
				eventAfterUpdating.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_UPDATING, drOld));
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void delete() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			return;			
		}		
		if (!isDeleteEnabled())
		{
			throw new ModelException("Delete isn't allowed!");
		}				
		
		IDataRow drOld = getDataRow(iSelectedRowIndex);
		if (eventBeforeDeleting != null)
		{
			eventBeforeDeleting.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_DELETING, drOld));
		}
		if (isInserting() && eventBeforeRestore != null)
		{
			eventBeforeRestore.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RESTORE, drOld));
		}
		
		// #507 move after eventsa above.
		// restore details, if cascade delete. 
		// Otherwise try to save the details, if WriteBackIsolationLevel == DATA_ROW 	
		restoreSaveAllDetails();		
		if (!isInserting())
		{
			restoreSelectedRow();
		}
		else 
		{
			// #375 - delete, cancelEditing missing 
			// -> cancel editing has to be called, to ensure, that the editors are not saving after delete.
			invokeCancelEditingControls();
		}		

		if (!isInserting())
		{
			// if this is an exiting row, then try to lock it first, that no other can modify it.
			if (isLockAndRefetchEnabled())
			{			
				executeLockAndRefetch();
			}
		}
		boolean bRowRemoved = deleteInternal();		

		if (bRowRemoved)
		{
			// correct selected row, if the last row got deleted
			correctSelectedRowAfterDelete();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}

			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}		

		if (eventAfterDeleting != null)
		{
			eventAfterDeleting.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_DELETING, drOld));
		}
		
		// correct selected row
		// #505 - delete should should check the WritebackIsolation level
		if (!bRowRemoved && WriteBackIsolationLevel.DATA_ROW.equals(getWritebackIsolationLevel())) 
		{
			// store depended DataBooks, if the WritebackIsolationLevel == IDataSource.DATA_ROW
			try
			{
				saveDataRowLevel(null);
			}
			catch (ModelException modelException)
			{
				if (isInserting() || isUpdating() || isDeleting())
				{		
					restore();
				}
		        throw modelException;
			}
		}		
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void deleteAllDataRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();

		fetchAll();
		for (int i = getRowCount() - 1; i >= 0; i--)
		{
			setSelectedRow(i);
			delete();
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyDetailChanged()
	{
		try
		{
			// check if one detail is changed.
			boolean bDetailChanged = false;
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				if (auDetailDataBooks.get(i).get() != null)
				{
					int [] iChanges = auDetailDataBooks.get(i).get().getChangedDataRows();
					
					if (iChanges != null && iChanges.length > 0)
					{
						bDetailChanged = true;
						break;
					}
				}
			}
			
			// set the detail change state correct.
			extentStorage();
			if (bDetailChanged)
			{
				oaStorage[rdRowDefinition.getColumnCount() + 2] = ONE;				
				dpCurrentDataPage.addChange(iSelectedRowIndex);
			}
			else
			{
				// remove Details changed flag.
				oaStorage[rdRowDefinition.getColumnCount() + 2] = null;				
				dpCurrentDataPage.removeChange(iSelectedRowIndex);
			}
			dpCurrentDataPage.setDataRowInternal(iSelectedRowIndex, oaStorage);
		}
		catch (ModelException e)
		{
			throw new RuntimeException(e);
		}

		setDetailsChangedInMasterBook();
	}
		
	/**
	 * {@inheritDoc}
	 */
	public synchronized IChangeableDataRow getDataRow(int pDataRowIndex) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (pDataRowIndex < 0)
		{
			return null;
		}
		sync();
		
		return getDataRowInternal(pDataRowIndex);		
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isDeleting() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isDeleting();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isUpdating() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isUpdating();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isInserting() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		// getSelectedRow does sync();
		if (getSelectedRow() < 0)
		{
			return false;
		}		
		return super.isInserting();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isDetailChanged() throws ModelException
	{
		if (!isOpen())
		{
			return false;			
		}
		sync();
		return super.isDetailChanged();
	}

	/**
	 * {@inheritDoc}
	 */
	public ICondition getFilter()
	{
		return cFilter;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setFilter(ICondition pFilter) throws ModelException
	{
		if (isOpen())
		{
			//#615
			//memfilter means: all rows fetched and if datasource level is set, don't save
			//on filter change.
			//If you have a pseudo master-link view (no master reference between databooks, but master sets filter to
			//filter details). In this case, don't save on "pseudo master" change.
//			if (!isMemFilter() || getWritebackIsolationLevel() != WriteBackIsolationLevel.DATASOURCE)
			{
//TODO [RH] temporary reverted because of save problems				
				saveAllRows();
			}
		}
		// copy Filter, to preserve, User changes during filtering.
		if (pFilter != null)
		{
			cFilter = pFilter.clone();	
		}
		else
		{
			cFilter = null;
		}
		
		if (isOpen())
		{
			if (isMemFilter())
			{
				clearFilterSortInMemDataPages();
				invokeRepaintListeners();		
			}
			else
			{			
				if (getSelectionMode() == SelectionMode.DESELECTED)
				{
					reload();
				}
				else
				{
					reload(SelectionMode.FIRST_ROW);
				}
			}
		}			
	}

	/**
	 * {@inheritDoc}
	 */
	public SortDefinition getSort()
	{
		return sSort;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void setSort(SortDefinition pSort) throws ModelException
	{
		if (isOpen())
		{
			saveAllRows();
		}
		// copy Filter, to preserve, User changes during filtering.
		if (pSort != null)
		{
			sSort = pSort.clone();		
		}
		else
		{
			sSort = null;		
		}
		
		if (isOpen())
		{
			if (isMemSort())
			{
				// #357 - setSort on MemDataBook with setMemSort(true), doesn't setSelectedRow correct after sort 
				handleStoreSelection(iSelectionMode);
				
				clearFilterSortInMemDataPages();				
				invokeRepaintListeners();
			}
			else
			{
				reload();
			}
		}			
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void reload() throws ModelException
	{
		reload(getSelectionMode());
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void reload(SelectionMode pSelectionMode) throws ModelException
	{
		if (isOpen())
		{
			handleStoreSelection(pSelectionMode);
			
			if (eventBeforeReload != null)
			{
				eventBeforeReload.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RELOAD, null));
			}
			
			if (!htDataPagesCache.isEmpty() || dpCurrentDataPage != null)
			{
				restoreAllRows();
			}
			
			executeRefresh();		

			// Notify Master changed has to be before AFTER_RELOAD event, to prevent undefined states in event.
			notifyMasterChanged();			
			
			if (eventAfterReload != null)
			{
				eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized void restoreSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();	
		if (getSelectedRow() < 0)
		{
			return;
		}	
		
		invokeCancelEditingControls();

		if (isInserting() || isUpdating() || isDeleting())
		{		
			restore();
		}

		invokeRepaintListeners();			
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void restoreAllRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (dpCurrentDataPage == null)
		{
			return;
		}
		sync();	

		invokeCancelEditingControls();

		if (getSelectedRow() >= 0 && (isInserting() || isUpdating() || isDeleting()))
		{
			restore();
		}

		// #492 - MemDataBook.restoreAllRows set wrong SelectedRow :: save the current row after the restore e.g is Inserting....
        int iOldSelectedRow = iSelectedRowIndex;        

        int[] iaChangedRows = dpCurrentDataPage.getChangedDataRows();

        //otherwise, it would be possible that the row is not available e.g. [5, 6, 7,... 100, 0, 1, 2, 3, 4]
        //an Exception occurs with row 100
		Arrays.sort(iaChangedRows);
		
		for (int i = iaChangedRows.length - 1; i >= 0; i--)
		{
			setSelectedRowInternal(iaChangedRows[i]);
			restore();
		}
		if (iSelectedRowIndex != iOldSelectedRow)
		{
			setSelectedRow(iOldSelectedRow);
		}
		invokeRepaintListeners();		
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void saveSelectedRow() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();	
		if (getSelectedRow() < 0)
		{
			return;
		}	

        invokeSaveEditingControls();
		if (isDeleting())
		{
			IDataRow drOld = getDataRow(iSelectedRowIndex);
			if (eventBeforeDeleted != null)
			{
				eventBeforeDeleted.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_DELETED, drOld));
			}
			
			// delete details first
    		if (isSelfJoined())
    		{
	        	if (!isDeleteCascade() && isWritebackEnabled())
	        	{
	    			if (treePath != null)
	    			{
	    				removeDataPage(null, treePath.getChildPath(getSelectedRow()));
	    			}
	    			else
	    			{
	    				removeDataPage(null, null);
	    			}
	        	}
    		}
			if (auDetailDataBooks != null)
			{
				for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
				{
					IDataBook dataBook = auDetailDataBooks.get(i).get();
					if (dataBook != null)
					{
			        	if (dataBook.isDeleteCascade())
			        	{
			        		dataBook.saveAllRows();
			        	}
			        	else
			        	{
			        		dataBook.restoreAllRows();
			        		
			        		if (isWritebackEnabled())
			        		{
			        			dataBook.removeDataPage(this, null);
			        		}
			        	}
					}
		        }					
			}
			
	        executeDelete();
	        store();

	        // #378 - Exception in AfterDeleted Event, if an insert() is called in the user event 
	        // correct selected row before the AfterDeleted Event, if the last row got deleted
			correctSelectedRowAfterDelete();
			
			if (eventAfterDeleted != null)
			{
				eventAfterDeleted.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_DELETED, drOld));
			}

			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
		else if (isUpdating())
		{
			IDataRow drOld = getOriginalRow();
			if (eventBeforeUpdated != null)
			{
				eventBeforeUpdated.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_UPDATED, drOld));
			}
			
			// #163 - if no writeable column is affected, don't update to the server! 
			// -> isWriteable() columns just came from server over the getMetaData() call in RemoteDataBook.
			if (isWritableColumnChanged())
			{
				executeUpdate();				
			}
	        store();
			if (eventAfterUpdated != null)
			{
				eventAfterUpdated.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_UPDATED, drOld));
			}
			invokeMasterChangedDetailsListeners();
		}
		else if (isInserting())
		{
			if (eventBeforeInserted != null)
			{
				eventBeforeInserted.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_INSERTED, null));
			}
			// if insert Master first
			if (getMasterReference() != null)
			{
		        ReferenceDefinition dbParentReference = isSelfJoined() ? getRootReference() : getMasterReference();
				if (dbParentReference != null && dbParentReference.getReferencedDataBook().isInserting())
				{
					dbParentReference.getReferencedDataBook().saveSelectedRow();
					// #315 - saveSelectedRow() bug with more then one leave with isInsertung state in DATASOURCE level
					// -> sync all details , before we save, so that bMaster changed is false again, and the changes of this level can be 
					// propagated to the details!
					syncDetails();
				}
				else
				{
					// copy PK (=reference columns) to detail.
					copyMasterColumnsToCurrentDetail();
				}
			}

			executeInsert();
	        store();
					        
			if (eventAfterInserted != null)
			{
				eventAfterInserted.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_INSERTED, null));
			}
			
			invokeMasterChangedDetailsListeners();
			
			// #288 - master DataRow rehash not working in DATASOUCE Level over more then 1 hierarchy ! - bug fixed.
			setUID(null);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void saveAllRows() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		if (htDataPagesCache.isEmpty() && dpCurrentDataPage == null)
		{
			//no pages, no changes return
			return;
		}
		sync();	

		// Editors has to save first to recognize changed Rows!
		invokeSaveEditingControls();
		
		int[] iaChangedRows = dpCurrentDataPage.getChangedDataRows();

		if (iaChangedRows.length > 0)
		{
	        //otherwise, it would be possible that the row is not available e.g. [5, 6, 7,... 100, 0, 1, 2, 3, 4]
	        //an Exception occurs with row 100
			Arrays.sort(iaChangedRows);
			
			int iOldSelectedRowIndex = getSelectedRow();
			for (int i = iaChangedRows.length - 1; i >= 0; i--)
			{
				int iRow = iaChangedRows[i];
				if (iSelectedRowIndex != iRow)
				{		
					setSelectedRowInternal(iRow);
				}
				saveSelectedRow();
			}
			if (iOldSelectedRowIndex >= getRowCount())
			{
				if (getSelectionMode() == SelectionMode.DESELECTED)
				{
					iOldSelectedRowIndex = -1;		
				}
				else
				{
					iOldSelectedRowIndex = getRowCount() - 1;						
				}
			}		
			if (iSelectedRowIndex != iOldSelectedRowIndex)
			{		
				setSelectedRowInternal(iOldSelectedRowIndex);
			}				
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isOutOfSync()
	{
		return dpCurrentDataPage == null || (rdMasterReference != null && bMasterChanged);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean isAllFetched() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		return dpCurrentDataPage.isAllFetched();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void fetchAll() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (dpCurrentDataPage == dpEmptyDataPage)
		{
			return;
		}
		dpCurrentDataPage.fetchAll();
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataRow getMasterDataRow() throws ModelException
	{
		if (!isOpen())
		{
			return null;	
		}
		sync();
		return dpCurrentDataPage.getMasterDataRow();
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public void addDetailDataBook(IDataBook pDetailDataBook)
	{
		if (auDetailDataBooks == null)
		{
			auDetailDataBooks = new ArrayUtil<WeakReference<IDataBook>>();
		}
		else
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				if (auDetailDataBooks.get(i).get() == null)
				{
					auDetailDataBooks.remove(i);
				}
			}			
		}
		auDetailDataBooks.add(new WeakReference(pDetailDataBook));
	}

	/**
	 * {@inheritDoc}
	 */
	public void removeDetailDataBook(IDataBook pDetailDataBook)
	{
		if (auDetailDataBooks != null)
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook == null || dataBook == pDetailDataBook)
				{
					auDetailDataBooks.remove(i);
				}
			}
		}
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook[] getDetailDataBooks()
	{
		ArrayUtil<IDataBook> result = new ArrayUtil<IDataBook>();
		if (auDetailDataBooks != null)
		{
			for (int i = auDetailDataBooks.size() - 1; i >= 0; i--)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook == null)
				{
					auDetailDataBooks.remove(i);
				}
				else
				{
					result.add(0, dataBook);
				}
			}
		}
		return result.toArray(new IDataBook[result.size()]);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized int[] getChangedDataRows() throws ModelException
	{
		if (!isOpen())
		{
			return null;			
		} 
		if (dpCurrentDataPage == null)
		{
			return null;
		}
		
		// #286 - no DataPage to master row exists, then return null -> no changes can exits - performance tuning!
		if (getMasterReference() != null && !isSelfJoined())
		{
			IDataBook dbMaster = getMasterReference().getReferencedDataBook();
			IDataRow drMaster = getMasterDataRow(dbMaster);
			if (htDataPagesCache.get(drMaster) == null)
			{
				if (dbMaster.getUID() != null)
				{
					if (htDataPagesCache.get(dbMaster.getUID()) == null)
					{
						return null;
					}
				}
				else
				{
					return null;
				}
			}			
		}
		
		sync();
		
		return dpCurrentDataPage.getChangedDataRows();
	}	
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized IDataPage getDataPage(IDataRow pMasterDataRow) throws ModelException
	{
		if (getMasterReference() == null)
		{
			sync();
			return dpCurrentDataPage;
		}
		else
		{
			return getDataPageIntern(pMasterDataRow);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public synchronized boolean hasDataPage(IDataRow pMasterDataRow) throws ModelException
	{
		if (getMasterReference() == null)
		{
			if (pMasterDataRow == null)
			{
				return true;
			}
			return false;
		}
		else
		{
			IDataRow drMaster = getMasterDataRow(pMasterDataRow);
			MemDataPage dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			

			if (dpNewCurrentDataPage == null && pMasterDataRow instanceof IChangeableDataRow)
			{
				if (((IChangeableDataRow)pMasterDataRow).getUID() != null)
				{
					dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)pMasterDataRow).getUID());
				}
			}
		
			return dpNewCurrentDataPage != null;
		}
	}

	/**
	 * Returns true if an <code>IDataPage</code> to specified root row and TreePath
	 * from the master <code>DataBook</code> exists.
	 * 
	 * @param pRootDataRow	the root <code>IDataRow</code> of the <code>DataBook</code> to check.
	 * @param pTreePath		the <code>TreePath</code> to use.
	 * @throws ModelException 
	 * 			  if the pRootDataRow don't contains the root columns from the DataBook.
	 * @return true if an <code>IDataPage</code> to specified root row and TreePath
	 * 		   from the master <code>DataBook</code> exists.
	 */
	public synchronized boolean hasDataPage(IDataRow pRootDataRow, TreePath pTreePath) throws ModelException
	{
		IDataRow drMaster = getMasterDataRowFromRootDataRow(pRootDataRow);
		MemDataPage dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			
		
		if (dpNewCurrentDataPage == null && pRootDataRow instanceof IChangeableDataRow)
		{
			if (((IChangeableDataRow)pRootDataRow).getUID() != null)
			{
				dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)pRootDataRow).getUID());
			}
		}
		
		if (pTreePath == null)
		{
			return dpNewCurrentDataPage != null;
		}		

		for (int i = 0; i < pTreePath.length(); i++)
		{
			int j = pTreePath.get(i);
			if (j < dpNewCurrentDataPage.getRowCount())
			{
				drMaster = getMasterDataRow(dpNewCurrentDataPage.getDataRow(j));
				dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			
				
				if (dpNewCurrentDataPage == null && drMaster instanceof IChangeableDataRow)
				{
					if (((IChangeableDataRow)drMaster).getUID() != null)
					{
						dpNewCurrentDataPage = htDataPagesCache.get(((IChangeableDataRow)drMaster).getUID());
					}
				}			
				
				if (dpNewCurrentDataPage == null)
				{
					return false;
				}
			}
			else
			{
				return false;
			}
		}
		return true;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeRowSelected()
	{
		if (eventBeforeRowSelected == null)
		{
			eventBeforeRowSelected = new DataBookHandler();
		}
		return eventBeforeRowSelected;
	}

	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterRowSelected()
	{
		if (eventAfterRowSelected == null)
		{
			eventAfterRowSelected = new DataBookHandler();
		}
		return eventAfterRowSelected;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeInserting()
	{
		if (eventBeforeInserting == null)
		{
			eventBeforeInserting = new DataBookHandler();
		}
		return eventBeforeInserting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterInserting()
	{
		if (eventAfterInserting == null)
		{
			eventAfterInserting = new DataBookHandler();
		}
		return eventAfterInserting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeInserted()
	{
		if (eventBeforeInserted == null)
		{
			eventBeforeInserted = new DataBookHandler();
		}
		return eventBeforeInserted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterInserted()
	{
		if (eventAfterInserted == null)
		{
			eventAfterInserted = new DataBookHandler();
		}
		return eventAfterInserted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeUpdating()
	{
		if (eventBeforeUpdating == null)
		{
			eventBeforeUpdating = new DataBookHandler();
		}
		return eventBeforeUpdating;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterUpdating()
	{
		if (eventAfterUpdating == null)
		{
			eventAfterUpdating = new DataBookHandler();
		}
		return eventAfterUpdating;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeUpdated()
	{
		if (eventBeforeUpdated == null)
		{
			eventBeforeUpdated = new DataBookHandler();
		}
		return eventBeforeUpdated;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterUpdated()
	{
		if (eventAfterUpdated == null)
		{
			eventAfterUpdated = new DataBookHandler();
		}
		return eventAfterUpdated;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeDeleting()
	{
		if (eventBeforeDeleting == null)
		{
			eventBeforeDeleting = new DataBookHandler();
		}
		return eventBeforeDeleting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterDeleting()
	{
		if (eventAfterDeleting == null)
		{
			eventAfterDeleting = new DataBookHandler();
		}
		return eventAfterDeleting;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeDeleted()
	{
		if (eventBeforeDeleted == null)
		{
			eventBeforeDeleted = new DataBookHandler();
		}
		return eventBeforeDeleted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterDeleted()
	{
		if (eventAfterDeleted == null)
		{
			eventAfterDeleted = new DataBookHandler();
		}
		return eventAfterDeleted;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeRestore()
	{
		if (eventBeforeRestore == null)
		{
			eventBeforeRestore = new DataBookHandler();
		}
		return eventBeforeRestore;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterRestore()
	{
		if (eventAfterRestore == null)
		{
			eventAfterRestore = new DataBookHandler();
		}
		return eventAfterRestore;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeReload()
	{
		if (eventBeforeReload == null)
		{
			eventBeforeReload = new DataBookHandler();
		}
		return eventBeforeReload;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterReload()
	{
		if (eventAfterReload == null)
		{
			eventAfterReload = new DataBookHandler();
		}
		return eventAfterReload;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventBeforeColumnSelected()
	{
		if (eventBeforeColumnSelected == null)
		{
			eventBeforeColumnSelected = new DataBookHandler();
		}
		return eventBeforeColumnSelected;
	}

	/**
	 * {@inheritDoc}
	 */
	public DataBookHandler eventAfterColumnSelected()
	{
		if (eventAfterColumnSelected == null)
		{
			eventAfterColumnSelected = new DataBookHandler();
		}
		return eventAfterColumnSelected;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchNext(pCondition);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition, int pRowNum) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchNext(pCondition, pRowNum);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchPrevious(pCondition);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition, int pRowNum) throws ModelException
	{
		sync();
		
		return dpCurrentDataPage.searchPrevious(pCondition, pRowNum);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getRowIndex()
	{
		try
		{
			sync();
			return iSelectedRowIndex;
		}
		catch (ModelException e)
		{
			return -1;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IDataPage getDataPage()
	{
		try
		{
			sync();
			return dpCurrentDataPage;
		}
		catch (ModelException e)
		{
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void finalize() throws Throwable
	{
		try
		{
			close();
		}
		catch (Throwable pThrowable)
		{
			// Silent close();
		}
		super.finalize();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Object getValue(String pColumnName) throws ModelException
	{
		if (!isOpen())
		{
			return null;
		}
		sync();
		
		return super.getValue(pColumnName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setValue(String pColumnName, Object pValue) throws ModelException
	{				
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			throw new ModelException("No selected row!");			
		}		
		if (isDeleting())
		{
			throw new ModelException("Row is already deleted!");			
		}		
		update();

		// the order of setValues was switched. Its important that a saveSelectedRow within a
		// values changed event stores the new and not the old row!
		setValueInternal();
		setWritableColumnChanged(pColumnName);
		super.setValue(pColumnName, pValue);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized Object[] getValues(String[] pColumnNames) throws ModelException
	{
		if (!isOpen())
		{
			return null;	
		}
		sync();			

		return super.getValues(pColumnNames);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setValues(String[] pColumnNames, Object[] pValues) throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		if (iSelectedRowIndex < 0)
		{
			throw new ModelException("No selected row!");			
		}		
		if (isDeleting())
		{
			throw new ModelException("Row is already deleted!");			
		}		
		update();

		// the order of setValues was switched. Its important that a saveSelectedRow within a
		// values changed event stores the new and not the old row!
		setValueInternal();	
		
		if (pColumnNames == null)
		{
			pColumnNames = rdRowDefinition.getColumnNames();
		}
		
		boolean bWritableChanged = isWritableColumnChanged();
		
		for (int i = 0; !bWritableChanged && i < pColumnNames.length; i++)
		{
			bWritableChanged = setWritableColumnChanged(pColumnNames[i]);
		}
		
		super.setValues(pColumnNames, pValues);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized String toString()
	{
		StringBuilder sbResult = new StringBuilder();
		
		sbResult.append(getClass().getName());
		sbResult.append(" ::");
		sbResult.append(sName);
		sbResult.append(":: \n  isOpen         = ");
		sbResult.append(bIsOpen);
		sbResult.append("\n");
		
		if (rdRowDefinition != null)
		{
			sbResult.append("  ");
			sbResult.append(rdRowDefinition);
		}
		if (rdMasterReference != null)
		{
			sbResult.append("  ");
			sbResult.append(rdMasterReference);
		}
		if (cFilter != null)
		{
			sbResult.append("  Filter         =");
			sbResult.append(cFilter);
		}	
		if (auDetailDataBooks != null)
		{
			sbResult.append("  DetailBooks    =");
			
			for (int i = 0; i < auDetailDataBooks.size(); i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					sbResult.append(" ");
					sbResult.append(dataBook.getName());
				}
			}					
		}
		
		if (isOpen())
		{
			try
			{
				sync();
			}
			catch (ModelException modelException)
			{
				return sbResult.toString() + " :: " + modelException.getMessage();
			}
				
			sbResult.append("\n  SelectedRow    = ");
			sbResult.append(iSelectedRowIndex);
			sbResult.append("\nCurrent DataPage:\n  ");
			sbResult.append(dpCurrentDataPage);
			
			Enumeration<MemDataPage> dpCacheElements = htDataPagesCache.elements();
			Enumeration<Object> dpCacheKeys = htDataPagesCache.keys();
			if (dpCacheElements.hasMoreElements())
			{
				sbResult.append("DataPagesCache:\n");
				
				while (dpCacheElements.hasMoreElements())
				{
					MemDataPage dpElement = dpCacheElements.nextElement();
					Object drKey     = dpCacheKeys.nextElement();
					if (dpElement != dpCurrentDataPage)
					{
						sbResult.append("  [");
						sbResult.append(drKey);
						sbResult.append("] = ");
						sbResult.append(dpElement);
					}
					else
					{
						sbResult.append("  [");
						sbResult.append(drKey);
						sbResult.append("] = Current DataPage\n");
					}
				}
			}
		}
		
        return sbResult.toString();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * It constructs a new MemDataPage. Its every time used, when a new MemDataPage is 
	 * necessary.
	 * Derived classed should override this and create the needed IDataPage.
	 *  
	 * @param pMasterDataRow	the master IDataRow for this IDataPage.
	 * @return the new MemDataPage to use.
	 */
	protected MemDataPage createDataPage(IDataRow pMasterDataRow)
	{
		return new MemDataPage(this, pMasterDataRow);
	}

	/**
	 * Returns true if the sort is handled in memory.
	 *
	 * @return true if the sort is handled in memory.
	 */
	protected boolean isMemSort()
	{
		return bMemSort;
	}

	/**
	 * Sets if the sort is handled in memory.
	 *
	 * @param pMemSort true if the sort is handled in memory. 
	 */
	protected void setMemSort(boolean pMemSort)
	{
		bMemSort = pMemSort;
	}

	/**
	 * Returns true if the filter is handled in memory.
	 *
	 * @return true if the filter is handled in memory.
	 */
	protected boolean isMemFilter()
	{
		return bMemFilter;
	}

	/**
	 * Sets if the sort is handled in memory.
	 *
	 * @param pMemFilter true if the filter is handled in memory. 
	 */
	protected void setMemFilter(boolean pMemFilter)
	{
		bMemFilter = pMemFilter;
	}	
	
	/**
	 * Returns true if the RemoteDataBook should write back the changes to the AbstractStorage.
	 * 
	 * @return true if the RemoteDataBook should write back the changes to the AbstractStorage.
	 */
	public boolean isWritebackEnabled()
	{
		return bWritebackEnabled;
	}
	
	/**
	 * Sets if the RemoteDataBook should write back the changes to the AbstractStorage.
	 * 
	 * @param pWritebackEnabled determines if the RemoteDataBook should write back the changes to the AbstractStorage.
	 */
	public void setWritebackEnabled(boolean pWritebackEnabled)
	{
		bWritebackEnabled = pWritebackEnabled;
	}	
	
	/**
	 * Returns the selection that is stored at the last reload.
	 * 
	 * @return the selection that is stored at the last reload.
	 */
	private IDataRow getStoredSelection()
	{
		return auStoredSelection;
	}
	
	/**
	 * Sets the the current selection before the reload.
	 * 
	 * @param pStoredSelection	the current selection as IDataRow (PK columns) to store 
	 */
	private void setStoredSelection(IDataRow pStoredSelection)
	{
		auStoredSelection = pStoredSelection;		
	}
		
	/**
	 * Returns the row count in <code>IDBAccess</code> for this RemoteDataBook.
	 * 
	 * @return the row count of the DataBook.
	 * @throws ModelException
	 * 				the the detail DataBook couldn't synchronized with the master DataBook
	 */
	public synchronized int getEstimatedRowCount() throws ModelException
	{
		if (!isOpen())
		{
			throw new ModelException("DataBook isn't open! - " + sName);			
		}
		sync();
		
		return dpCurrentDataPage.getEstimatedRowCount();
	}	
	
	/**
	 * Its called when the inserting DataRow (new row) should be inserted.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. store the new row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during insert.
	 */
	protected void executeInsert() throws ModelException
	{
	}
	
	/**
	 * Its called when the updating DataRow (changed row) should be updated
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. update the old row with the new row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during update.
	 */
	protected void executeUpdate() throws ModelException
	{
	}
	
	/**
	 * Its called when the deleting DataRow (delete row) should be deleted.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. delete the row in the storage.
	 * 
	 * @throws ModelException	if an ModelException occur during delete.
	 */
	protected void executeDelete() throws ModelException
	{
	}

	/**
	 * Its called before the DataRow (existing row) will be changed.
	 * Derived classed should override this to implemented the needed functionality.
	 * e.g. lock and refetch the actual values of the row from storage
	 * 
	 * @throws ModelException	if an ModelException occur during insert.
	 */
	protected void executeLockAndRefetch() throws ModelException
	{
	}
	
	/**
	 * It invokes for all registered IConrols the notifyRepaint() method and calls for
	 * all detail IDataBooks the the notifyMasterChanged method to determine that the 
	 * master has changed.
	 */
	private void invokeMasterChangedDetailsListeners()
	{	
		invokeRepaintListeners();				
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.notifyMasterChanged();
				}
			}
		}
	}	
	
	/**
	 * Sync the DataBook an all it detail DataBooks. Its necessary in DATASOURCE level for saving cascades over varies levels. 
	 *
	 * @throws ModelException if the sync isn't possible.
	 */
	private void syncDetails() throws ModelException
	{	
		sync();
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.getSelectedRow(); // calls a sync!!!
				}
			}
		}
	}	

	/**
	 * Returns the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * 
	 * @param pMasterDataRow	the full/complete master IDataRow to use.
	 * @return internal the master row only with the primary key columns - like defined in
	 * the getMasterReference().getReferencedColumns().
	 * @throws ModelException	if the master row couldn't created.
	 */
	private IDataRow getMasterDataRow(IDataRow pMasterDataRow) throws ModelException
	{
		return pMasterDataRow.createDataRow(getMasterReference().getReferencedColumnNames());
	}
	
	/**
	 * Returns the MemDataPage to the specified master IDataRow. If the none existing,
	 * it will be a new one created and in the MemDataPage Hashtable stored.
	 *   
	 * @param pMasterDataRow	the master IDataRow to use.
	 * @return the MemDataPage to the specified master IDataRow.
	 * @throws ModelException	if master row or the MemDataPage couldn't created.
	 */
	private MemDataPage getDataPageIntern(IDataRow pMasterDataRow) throws ModelException
	{
		// #336 - First check if a DataPage with the UID exists.
		MemDataPage dpNewCurrentDataPage = null;
		if (pMasterDataRow instanceof ChangeableDataRow)
		{
			Object oUID = ((ChangeableDataRow)pMasterDataRow).getUID();
			if (oUID != null)
			{
				dpNewCurrentDataPage = htDataPagesCache.get(oUID);			
				if (dpNewCurrentDataPage != null)
				{
					return dpNewCurrentDataPage;
				}
			}
		}
		IDataRow drMaster = getMasterDataRow(pMasterDataRow);
		dpNewCurrentDataPage = htDataPagesCache.get(drMaster);			

		// init DataPage, if for this master row doesn't exists a DataPage in Cache
		if (dpNewCurrentDataPage == null)
		{		
			IChangeableDataRow cdrMasterDataRow = null;
			if (pMasterDataRow instanceof IChangeableDataRow)
			{
				if (((IChangeableDataRow)pMasterDataRow).getUID() != null)
				{
					cdrMasterDataRow = (IChangeableDataRow)pMasterDataRow;
					dpNewCurrentDataPage = htDataPagesCache.get(cdrMasterDataRow.getUID());					
				}
			}			

			if (dpNewCurrentDataPage == null)
			{
				dpNewCurrentDataPage = createDataPage(drMaster);
				if (cdrMasterDataRow != null && cdrMasterDataRow.getUID() != null)
				{
					// #290- if the master DataBook is in DATASOURCE level && Inserting() no fetch is necessary, because no rows can exists!
					ReferenceDefinition masterRef = getMasterReference();
					if (masterRef != null 
							&& masterRef.getReferencedDataBook().getWritebackIsolationLevel() == IDataBook.WriteBackIsolationLevel.DATASOURCE)
					{
						dpNewCurrentDataPage.setAllFetched(true);
					}
					
					htDataPagesCache.put(cdrMasterDataRow.getUID(), dpNewCurrentDataPage);	
					   
				}
				else
				{
					htDataPagesCache.put(drMaster, dpNewCurrentDataPage);
				}
			}
		}
		return dpNewCurrentDataPage;
	}

	/**
	 * Returns the the IChangeableDataRow synchronized for the specified row index.
	 * 
	 * @param pDataRowIndex	the row index to use.
	 * @return the the IChangeableDataRow synchronized for the specified row index
	 * @throws ModelException if the IChangeableDataRow couldn't get from the MemDataPage.
	 */
	private synchronized IChangeableDataRow getDataRowInternal(int pDataRowIndex) throws ModelException
	{
		if (pDataRowIndex < 0 || dpCurrentDataPage == dpEmptyDataPage)
		{
			return null;
		}
		
		return dpCurrentDataPage.getDataRow(pDataRowIndex);
	}

	/**
	 * It correct after the sync() call the selected row under consideration of the 
	 * Selection Mode and the the stored selection of the last reload.
	 * 
	 * @throws ModelException	if the row couldn't selected.
	 */
	private void correctSelectedRow() throws ModelException
	{
		IDataRow drCurrent = getStoredSelection();
		
		if (getSelectionMode() == SelectionMode.DESELECTED)
		{
			setSelectedRowInternal(-1);						
		}			
		else if (getSelectionMode() == SelectionMode.CURRENT_ROW && drCurrent != null)
		{
			if (drCurrent == this) // if the databook was deselectd the reload functon stores this to identify, that it was deselected.
			{
				setSelectedRowInternal(-1);	
			}
			else
			{
				// find the same row by StoredSelection
				int i = 0;
				IDataRow drCompare = getDataRow(i);
	
				String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
				if (asPKColumns == null)
				{
					asPKColumns = rdRowDefinition.getColumnNames();
				}
				
				while (drCompare != null && !drCurrent.equals(drCompare, asPKColumns))
				{
					i++;
					drCompare = getDataRow(i);
				}
				if (drCompare != null)
				{
					setSelectedRowInternal(i);
				}
				else
				{
					setSelectedRowInternal(0);						
				}
			}
		}
		else
		{
			setSelectedRowInternal(0);						
		}
		if (eventAfterRowSelected != null)
		{
			eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, null));
		}
	}

	/**
	 * Corrects the selected row after delete on the same or one row index before. 
	 * if the DESELECTED and the last row is deleted, it deselects.
	 * 
	 * @throws ModelException if selected row couldn't changed.
	 */
	private void correctSelectedRowAfterDelete() throws ModelException
	{
		int iNewRowIndex = iSelectedRowIndex;
		if (iSelectedRowIndex >= getRowCount())
		{
			if (getSelectionMode() == SelectionMode.DESELECTED)
			{
				iNewRowIndex = -1;		
			}
			else
			{
				iNewRowIndex = getRowCount() - 1;						
			}
		}		
		setSelectedRowInternal(iNewRowIndex);
	}
		
	/**
	 * It synchronize the MemDataBook, at the first access after the (master) change.
	 * It gets the corresponding MemDataPage from the Hastable. If not possible it will 
	 * be created, or an empty MemDataPage will be filled in when the master has no
	 * selected row. It also does the lazy load after the open()/reload call.  
	 * 
	 * @throws ModelException	if the new MemDataPage couldn't determined.
	 */
	private void sync() throws ModelException
	{
		// init the CurrentDataPage for a Master DataBook
	    if (dpEmptyDataPage == null)
	    {
	    	dpEmptyDataPage = new MemDataPage(null, null);
	    }
	   
		if (rdMasterReference == null)
		{
			if (dpCurrentDataPage == null)
			{
				dpCurrentDataPage = dpEmptyDataPage;
				iSelectedRowIndex = -1;

				dpCurrentDataPage  = createDataPage(null);
				correctSelectedRow();
			}
		}
		else if (bMasterChanged)
		{
			bMasterChanged = false;
			
			if (dpCurrentDataPage == null)
			{
				dpCurrentDataPage = dpEmptyDataPage;
				iSelectedRowIndex = -1;
				treePath = new TreePath();
			}

			ReferenceDefinition dbParentReference = isSelfJoined() ? getRootReference() : getMasterReference();
			if (dbParentReference != null && dbParentReference.getReferencedDataBook().getSelectedRow() < 0)
			{
				dpCurrentDataPage = dpEmptyDataPage;
				iSelectedRowIndex = -1;
				treePath = new TreePath();

				setSelectedRowInternal(-1);
			}
			else
			{
				if (isSelfJoined())
				{
					if (dpCurrentDataPage != dpEmptyDataPage
							&& dbParentReference != null
							&& !getMasterDataRowFromRootDataRow(dbParentReference.getReferencedDataBook()).equals(
									dpCurrentDataPage.getMasterDataRow(), saTreeRootMasterColumnNames))
					{
						treePath = new TreePath();
					}
					
					IDataPage dpNewCurrentDataPage = getDataPage(treePath);

					if (dpNewCurrentDataPage != dpCurrentDataPage)
					{
						treePath = new TreePath();
					
						dpCurrentDataPage = dpEmptyDataPage;
						iSelectedRowIndex = -1;
	
						dpCurrentDataPage = (MemDataPage)getDataPage(treePath);
						
						if (getSelectionMode() == SelectionMode.DESELECTED)
						{
							setSelectedRowInternal(-1);
						}
						else
						{
							setSelectedRowInternal(0);
						}
					}
				}
				else
				{
					MemDataPage dpNewCurrentDataPage = getDataPageIntern(getMasterReference().getReferencedDataBook());
					
					// different master row, so change the current DataPage
					if (dpNewCurrentDataPage != dpCurrentDataPage)
					{
						dpCurrentDataPage = dpEmptyDataPage;
						iSelectedRowIndex = -1;
						treePath = new TreePath();

						dpCurrentDataPage = dpNewCurrentDataPage;
						correctSelectedRow();
					}
				}
			}
		}
	}	

	/**
	 * It will be called, when the MemDataBook will be completely refreshed. 
	 * In the MemDataBook, it does nothing, because all is in memory and not restoreable.
	 * Derived classes maybe want to clear the memory.
	 * 
	 * @throws ModelException	if an ModelExcpetion happen during refresh
	 */
	protected void executeRefresh()  throws ModelException
	{
		clearFilterSortInMemDataPages();
		invokeRepaintListeners();		
	}
	
	/**
	 * It clears the AbstractStorage of the MemDataBook. Derived classed maybe need it.
	 */
	protected void clear()
	{
		dpCurrentDataPage	      = null;
		htDataPagesCache.clear();							
		iSelectedRowIndex 	      = -1;
		oaStorage 				  = null;
	}
	
	/**
	 * Sets the selected row with throwing any changed events.
	 * 
	 * @param pDataRowIndex		the new row index to use.
	 * @throws ModelException	if new row couldn't determined or selected.
	 */
	protected void setSelectedRowInternal(int pDataRowIndex) throws ModelException
	{
		IDataRow drNewCurrent = null;
		
		try
		{
			drNewCurrent = getDataRow(pDataRowIndex);
		}
		finally
		{
			if (drNewCurrent != null)
			{
				oaStorage = dpCurrentDataPage.getDataRowInternal(pDataRowIndex);
				iSelectedRowIndex = pDataRowIndex;
			}
			else
			{
				oaStorage = new Object[rdRowDefinition.getColumnCount()];
				iSelectedRowIndex = -1;
			}
		}

		invokeMasterChangedDetailsListeners();				
	}
	
	/**
	 * It inserts a new before or after the selected row and returns the row index where it is
	 * inserted. 
	 * 
	 * @param pBeforeRow	determines if the row is inserted before (true) or after (false) the current row.
	 * @return the row index, where the row is inserted.
	 * @throws ModelException	if a new row couldn't inserted.
	 */
	private int insertInternal(boolean pBeforeRow) throws ModelException
	{
		int iDataRowIndex = getSelectedRow();
		if (!pBeforeRow && getRowCount() > 0)
		{
			iDataRowIndex++;
		}		
		if (iDataRowIndex < 0)
		{
			iDataRowIndex = 0;
		}

		// create new data row, set Inserting and UID and set storage of the base DataRow to this 
		oaStorage = createNewRow();
		setInserting();
		setUID(Integer.valueOf(iUID++));
		
		// insert new Row in current DataPage
		dpCurrentDataPage.insert(iDataRowIndex, oaStorage); 
		
		// set master in master, that the details changed.
		setDetailsChangedInMasterBook();
								
		invokeMasterChangedDetailsListeners();				
		
		return iDataRowIndex;
	}
	
	/**
	 * It deletes the current selected row and returns true if the row is removed from
	 * the memory or false if the row is only marked as deleting. The first case happens if
	 * the row is inserting and not stored, then it will be removed.
	 * 
	 * @return true if the row is removed from
	 * the memory or false if the row is only marked as deleting.
	 * @throws ModelException	if the delete operation fails.
	 */
	private boolean deleteInternal() throws ModelException
	{
		// delete details
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
		        	if (dataBook.isDeleteCascade())
		        	{
		        		dataBook.deleteAllDataRows();
		        	}
		        	else
		        	{
		        		dataBook.restoreAllRows();
		        	}
				}
			}
		}
        
        boolean bResult;
		// if the current DataRow is new inserted, then remove it from storage
		if (isInserting())
		{
			dpCurrentDataPage.delete(iSelectedRowIndex);
			bResult = true;
		}
		else
		{
	        // Otherwise mark current DataRow as deleting
			setDeleting(); 
			dpCurrentDataPage.addChange(iSelectedRowIndex);
			dpCurrentDataPage.setDataRowInternal(iSelectedRowIndex, oaStorage);			
			bResult = false;
		}
		
		setDetailsChangedInMasterBook();
		invokeRepaintListeners();			
		
		return bResult;
	}

	/**
	 * It sets the values without throwing an event.
	 * 
	 * @param pColumnNames		the column names to use. 
	 * @param pValues			the values to use
	 * @throws ModelException	if the value couldn't set into the memory.
	 */
	protected void setValuesInternal(String[] pColumnNames, Object[] pValues) throws ModelException
	{
		setValueInternal();
		
		if (pColumnNames.length != pValues.length)
		{
			throw new ModelException("Number of columns and values are different!");
		}
		
		boolean bChanged = false;
		
		for (int i = 0; i < pColumnNames.length; i++)
		{
			if (!bChanged)
			{
				bChanged = setWritableColumnChanged(pColumnNames[i]);
			}
			
			// set the DataRow value
			super.setValueDRInternal(pColumnNames[i], pValues[i]);
		}		
	} 
	
	/**
	 * It sets the value without throwing an event. It uses the mechanism, 
	 * that oaStorage will be set in the setValue() of the DataRow.
	 * This method is only allowed to call together with the setValue of the DataRow.
	 * 
	 * @throws ModelException	if the oaStorage couldn't set into the MemDataPage.
	 */
	protected void setValueInternal() throws ModelException
	{
		dpCurrentDataPage.setDataRowInternal(iSelectedRowIndex, oaStorage);		
	}
	
	/**
	 * It sets in the master IDataBook, that this detail has changed with the detailChanged()
	 * method.
	 */
	private void setDetailsChangedInMasterBook()
	{
		if (rdMasterReference != null && !isSelfJoined())
		{
			IDataBook dbMaster = (IDataBook)rdMasterReference.getReferencedDataBook();
			dbMaster.notifyDetailChanged();
		}
	}
	
	/**
	 * After a insert of a new detail row, the master reference columns will be copied
	 * from the master to this detail row.
	 * 
	 * @throws ModelException	if the values couldn't copied 
	 */
	private void copyMasterColumnsToCurrentDetail() throws ModelException
	{
		if (rdMasterReference != null)
		{
			Object[] oaMasterValues = dpCurrentDataPage.getMasterDataRow().getValues(rdMasterReference.getReferencedColumnNames());
			setValuesInternal(rdMasterReference.getColumnNames(), oaMasterValues);
			
			if (isSelfJoined() && rdTreeRootReference != null)
			{
				oaMasterValues = rdTreeRootReference.getReferencedDataBook().getValues(rdTreeRootReference.getReferencedColumnNames());
				setValuesInternal(rdTreeRootReference.getColumnNames(), oaMasterValues);
			}
		}
	}		
	
	/**
	 * Its saves all required changes, if the write back isolation level is DATAROW.
	 * It determines the root IDataBook and call saveDetails(). 
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDataRowLevel(IDataBook pExcludeDataBook) throws ModelException
	{
		// get Root Master first
        IDataBook dbRoot = this;
        ReferenceDefinition dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();

        // if this databook has WritebackIsolationLevel DATA_ROW, it has to ensure that all Details and all Masters 
        // with WritebackIsolationLevel DATA_ROW are saved too, because in DATA_ROW level only one datarow over all 
        // should be edited.
        // if this databook has WritebackIsolationLevel DATA_SOURCE, only all details with WritebackIsolationLevel DATA_ROW
        // are saved. The master is explizitly not saved.
        // in general it is guaranteed, if a databook with WritebackIsolationLevel DATA_ROW is saved, all its details,
        // ignoring the WritebackIsolationLevel, are saved.
        while (dbRoot.getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW && dbParentReference != null)
		{
			dbRoot = dbParentReference.getReferencedDataBook();
			
			dbParentReference = dbRoot.isSelfJoined() ? dbRoot.getRootReference() : dbRoot.getMasterReference();
		}

		// then save all details to this root DataBook, except the current DataBook
		((MemDataBook)dbRoot).saveDataRowLevelDetails(pExcludeDataBook);
	}
	
	/**
	 * It saves all required changes, if the write back isolation level is DATA_ROW.
	 * It searches the current row and then hierarchical down over all details to the last detail if there is a 
	 * data book with WriteBackIsolationLevel DATA_ROW.
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDataRowLevelDetails(IDataBook pExcludeDataBook) throws ModelException
	{		
		if (getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
		{
			saveDetails(pExcludeDataBook);
		}
		else if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					((MemDataBook)dataBook).saveDataRowLevelDetails(pExcludeDataBook);
				}
			}
		}
	}
	
	/**
	 * It saves all required changes ignoring the write back isolation level.
	 * It saves current row and then hierarchical down over all details to the last detail.
	 * 
	 * @param pExcludeDataBook	the IDataBook to exclude in the save process
	 * @throws ModelException	if the changes couldn't saved.
	 */
	private void saveDetails(IDataBook pExcludeDataBook) throws ModelException
	{		
		if (this != pExcludeDataBook)
		{
			saveAllRows();
		}
		
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					((MemDataBook)dataBook).saveDetails(pExcludeDataBook);
				}
			}
		}
	}
	
	/**
	 * It restores all details, if cascade delete. 
     * Otherwise it tries to save the details, if WriteBackIsolationLevel == DATA_ROW.
     *  	
	 * @throws ModelException if the save or restore fails.
	 */
	private void restoreSaveAllDetails() throws ModelException
	{
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
		        	if (dataBook.isDeleteCascade())
		        	{
		        		dataBook.restoreAllRows();
		        		((MemDataBook)dataBook).restoreSaveAllDetails();
		        	}
		        	else
		        	{
		        		if (dataBook.getWritebackIsolationLevel() == WriteBackIsolationLevel.DATA_ROW)
		        		{
		        			dataBook.saveAllRows();
		        		}
		        		((MemDataBook)dataBook).restoreSaveAllDetails();
		        	}
				}
			}
		}
	}
	
	/**
	 * Stores the selection for a correct, reload, resort, ...
	 * 
	 * @param pSelectionMode 	the SelectionMode to use. 
	 * @throws ModelException if getSelectedRow throw an exception.
	 */
	private void handleStoreSelection(SelectionMode pSelectionMode) throws ModelException
	{
		if (dpCurrentDataPage != null)
		{
			if (pSelectionMode == SelectionMode.CURRENT_ROW)
			{
				if (getSelectedRow() < 0) // Mark that the databook is deselected.
				{
					if (dpCurrentDataPage.getRowCount() == 0)
					{
						setStoredSelection(null);
					}
					else
					{
						setStoredSelection(this);
					}
				}
				else
				{
					// save current DataRow, PK cols or all
					String[] asPKColumns = rdRowDefinition.getPrimaryKeyColumnNames();
					if (asPKColumns == null)
					{
						asPKColumns = rdRowDefinition.getColumnNames();
					}
					if (asPKColumns != null && asPKColumns.length > 0)
					{
						setStoredSelection(getDataRow(iSelectedRowIndex).createDataRow(asPKColumns));
					}
				}
			}
			else
			{
				setStoredSelection(null);
			}
		}
	}
	
	/**
	 * Sets the internal Unique Identifier for the DataRow (is the DataBook itself), who 
	 * aren't INSERTED to the AbstractStorage.<br>
	 * Thats necessary for DataBooks with WritebackIsolationLevel == DATABOOK, and insert more 
	 * the one new empty master row.
	 * 
     * @param pUID the new Unique Identifier
	 */
	private void setUID(Object pUID)
	{
		extentStorage();
		oaStorage[rdRowDefinition.getColumnCount()] = pUID;
	}
	
	/**
	 * It marks this row (is the DataBook itself) as INSERTING.
	 */
	private void setInserting()
	{
		extentStorage();
		oaStorage[rdRowDefinition.getColumnCount() + 1] = ChangeableDataRow.INSERTING;			
	}
	
	/**
	 * It marks this row (is the DataBook itself) as UPDATING.
	 */
	private void setUpdating()
	{
		int iColumnCount = rdRowDefinition.getColumnCount();

		// save the original value only if the old row has no changes
		Object[] oaNewRow = new Object[iColumnCount * 2 + INTERNAL_OFFSET];
		System.arraycopy(oaStorage, 0, oaNewRow, 0, oaStorage.length); // copy all the information!!!
		System.arraycopy(oaStorage, 0, oaNewRow, INTERNAL_OFFSET + iColumnCount, iColumnCount);
		
		if (oaStorage.length > iColumnCount + 2)
		{
			oaNewRow[iColumnCount + 2] = oaStorage[iColumnCount + 2]; // copy Details changed flag
		}
		
		// set new Object[] into the Row
		oaStorage = oaNewRow;
		oaStorage[iColumnCount + 1] = UPDATING;
		
		dpCurrentDataPage.addChange(iSelectedRowIndex);		
	}

	/**
	 * It marks this row (is the DataBook itself) as WRITEABLE_COLUMN_CHANGED.
	 * 
	 * @param pColumn the column that has changed.
	 * @return <code>true</code> if the current row is now set changed, <code>false</code> otherwise
	 * @throws ModelException if the column didn't found.
	 */
	private boolean setWritableColumnChanged(String pColumn) throws ModelException
	{
		// #163 - IChangeableDataRow should support isWritableColumnChanged 
		// set the state here correct!
		if (isUpdating() && rdRowDefinition.getColumnDefinition(pColumn).isWritable())
		{
			oaStorage[rdRowDefinition.getColumnCount() + 1] = WRITABLE_COLUMN_CHANGED;
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * It marks this row (is the DataBook itself) as DELETING.
	 */
	private void setDeleting()
	{
		extentStorage();
		oaStorage[rdRowDefinition.getColumnCount() + 1] = DELETING;			
	}

	/**
	 * It extends the internal storage, if the first change in the ChangeableDataRow 
	 * (is the DataBook itself) is made.
	 */
	private void extentStorage()
	{
		int iColumnCount = rdRowDefinition.getColumnCount();
		if (oaStorage.length <= iColumnCount)
		{
			// extend the AbstractStorage and copy the current values into it.
			Object[] oaNewRow = new Object[iColumnCount + INTERNAL_OFFSET];
			System.arraycopy(oaStorage, 0, oaNewRow, 0, iColumnCount);
			oaStorage = oaNewRow;
        }
	}
	
	/**
	 * It clears all changes in the ChangeableDataRow (is the DataBook itself), and use 
	 * the original values before the changes. 
	 * 
	 * @throws ModelException	if the isUpdating() check doesn't work
	 */
	private void restoreChangeableDataRow() throws ModelException
	{
		int iColumnCount = rdRowDefinition.getColumnCount();

		if (isUpdating())
		{
			Object[] oaNewRow = new Object[iColumnCount];
			System.arraycopy(oaStorage, INTERNAL_OFFSET + iColumnCount, 
					         oaNewRow, 0, 
					         iColumnCount);
			
			// set new Object[] into the Row
			oaStorage = oaNewRow;
		}
		else 
		{
			Object[] oaNewRow = new Object[iColumnCount];
			System.arraycopy(oaStorage, 0, 
					         oaNewRow, 0, 
					         iColumnCount);
			
			// set new Object[] into the Row
			oaStorage = oaNewRow;
		}
		dpCurrentDataPage.setDataRowInternal(iSelectedRowIndex, oaStorage);				
	}
	
	/**
	 * It restores all detail rows.
	 * 
	 * @throws ModelException	if the change couldn't restored in memory.
	 */
	private void restoreAllDetails() throws ModelException
	{
		if (auDetailDataBooks != null)
		{
			for (int i = 0, size = auDetailDataBooks.size(); i < size; i++)
			{
				IDataBook dataBook = auDetailDataBooks.get(i).get();
				if (dataBook != null)
				{
					dataBook.restoreAllRows();
				}
			}
		}
	}

	/**
	 * It restores the changes on the selected row and remove the change state of 
	 * this row.
	 * 
	 * @throws ModelException	if the change couldn't restored in memory.
	 */
	private void restore() throws ModelException
	{		
		IDataRow drOld = getDataRow(iSelectedRowIndex);
		if (eventBeforeRestore != null)
		{
			eventBeforeRestore.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RESTORE, drOld));
		}

		if (isInserting())
		{
			// #376 - restore row fails in DATASOURCE level with more then one detail level with isInserting rows
			// restoreDetails first, then delete row from page.
			restoreAllDetails();

			dpCurrentDataPage.delete(iSelectedRowIndex);
			setSelectedRowInternal(iSelectedRowIndex - 1);
			setDetailsChangedInMasterBook();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}
			if (eventAfterRowSelected != null)
			{
				eventAfterRowSelected.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_ROW_SELECTED, drOld));
			}
		}
		else
		{
			dpCurrentDataPage.removeChange(iSelectedRowIndex);		
			restoreChangeableDataRow();		
			setDetailsChangedInMasterBook();
			
			if (eventAfterRestore != null)
			{
				eventAfterRestore.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RESTORE, drOld));
			}
		}
	}
	
	/**
	 * It stores the changes in memory as stored. That means the changes will be made and
	 * the change state of the current DataRow (the DataBook it self) will be removed.
	 * 
	 * @throws ModelException	if the isDeleting() check doesn't work
	 */
	private void store() throws ModelException
	{
		if (isDeleting())
		{
			dpCurrentDataPage.delete(iSelectedRowIndex);
			return;
		}	
		dpCurrentDataPage.removeChange(iSelectedRowIndex);		
		
		// It clears all changes in the ChangeableDataRow, but prevents the actual values.
		int iColumnCount = rdRowDefinition.getColumnCount();

		Object[] oaNewRow = new Object[iColumnCount + INTERNAL_OFFSET];
		System.arraycopy(oaStorage, 0, 
				         oaNewRow, 0, 
				         iColumnCount + 1);

		// set new Object[] into the Row
		oaStorage = oaNewRow;
		dpCurrentDataPage.setDataRowInternal(iSelectedRowIndex, oaStorage);		
		
		setDetailsChangedInMasterBook();
	}

	/**
	 * It clears all mem filter sorts in the MemDataPages.
	 * 
	 * @throws ModelException if a ModelException occurs.
	 */
	protected void clearFilterSortInMemDataPages() throws ModelException
	{
		if (eventBeforeReload != null)
		{
			eventBeforeReload.dispatchEvent(new DataBookEvent(this, ChangedType.BEFORE_RELOAD, null));
		}
		
		if (getMasterReference() != null)
		{
			Enumeration<MemDataPage> enumDataPages = htDataPagesCache.elements();
			while (enumDataPages.hasMoreElements())
			{
				enumDataPages.nextElement().clear();
			}
		}
		else
		{
			if (!isOutOfSync())
			{
				dpCurrentDataPage.clear();
			}
		}		

		// correctSelectedRow has to be before AFTER_RELOAD event, to prevent undefined states in event.
		if (!isOutOfSync())
		{
			correctSelectedRow();
		}
		if (eventAfterReload != null)
		{
			eventAfterReload.dispatchEvent(new DataBookEvent(this, ChangedType.AFTER_RELOAD, null));
		}
	}
	
	/**
	 * Creates a new empty row with default values.
	 * 
	 * @return an object array as new row
	 */
	protected Object[] createNewRow()
	{
		int iCount = rdRowDefinition.getColumnCount();
		
		Object[] oRow = new Object[iCount + INTERNAL_OFFSET];

		for (int i = 0, anz = iCount; i < anz; i++)
		{
			oRow[i] = rdRowDefinition.getColumnDefinition(i).getDefaultValue();
		}

		return oRow;
	}

	/**
	 * Sets the databook readonly but does not automatically save.
	 * 
	 * @param pReadOnly <code>true</code> to set the databook readonly, <code>false</code> otherwise
	 */
	public void setReadOnlyWithoutSave(boolean pReadOnly)
	{
		bReadOnly = pReadOnly;
		
		invokeRepaintListeners();
	}
	
}	// MemDataBook
