package com.shitu.epathmap.ui.widget;

import android.content.Context;
import android.graphics.Camera;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Path;
import android.support.annotation.Nullable;
import android.util.AttributeSet;
import android.view.View;

import com.sails.engine.LocationRegion;
import com.sails.engine.SAILS;
import com.sails.engine.SAILSMapView;
import com.shitu.epathmap.ui.utils.CoordinateConversion;
import com.shitu.epathmap.ui.utils.DeviceUtils;
import com.shitu.location.epathmap.utils.IpsConstants;

import java.util.ArrayList;
import java.util.List;

public class GuideLineView extends View {


    private Paint paint;
    private CoordinateConversion conversion;
    private ArrayList<FPoint> mPoints;
    private Paint textPaint;
    private float angle;
    private Camera camera;

    private int beginOffset = 90;
    private double latitude;
    private double longitude;
    private long useTime;
    private boolean needDraw;
    private SAILSMapView mapView;
    private float lineUnit;

    public GuideLineView(Context context) {
        this(context,null);
    }

    public GuideLineView(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public GuideLineView(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);

        init(context);
    }

    private void init(Context context) {
        paint = new Paint(Paint.ANTI_ALIAS_FLAG);
        paint.setStyle(Paint.Style.STROKE);
        paint.setColor(Color.parseColor("#4400FF7F"));
        paint.setStrokeWidth(DeviceUtils.dpToPixel(getContext(), 48));
//        paint.setStrokeCap(Paint.Cap.ROUND);
        paint.setStrokeJoin(Paint.Join.ROUND);

        lineUnit = DeviceUtils.dpToPixel(getContext(), 32);



        textPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        textPaint.setTextSize(DeviceUtils.dpToPixel(getContext(),16));
        textPaint.setColor(Color.BLACK);
        textPaint.setTextAlign(Paint.Align.CENTER);

        conversion = new CoordinateConversion();

        camera = new Camera();
    }


    @Override
    protected void onDraw(Canvas canvas) {

        if(needDraw){
            postInvalidateDelayed(16);
        }

        if(mPoints == null || mPoints.size() == 0){
            return;
        }

        if(mapView != null){
            angle = mapView.getRotationAngle();
        }

        Path path = new Path();
        path.moveTo(0,0);


        StringBuffer sb = new StringBuffer();

        for (FPoint point : mPoints) {
            path.lineTo(point.x*lineUnit,point.y*lineUnit);
            sb.append(point.x).append(",").append(point.y).append(" ");
        }

        int width = getWidth();
        int height = getHeight();
//        path.lineTo(100,100);
//        path.lineTo(300,400);
//        path.lineTo(600,0);

        int save = canvas.save();

        camera.save();

        camera.rotateX(45);

        canvas.translate(width>>1,height>>1);

        camera.applyToCanvas(canvas);

        camera.restore();

        canvas.translate(- width>>1, - height>>1);

        canvas.translate(width>>1,height*2/3);

        canvas.rotate(angle);

        canvas.drawPath(path,paint);

        canvas.restoreToCount(save);

//        canvas.drawText(mPoints.size()+"",width/2,height/2,textPaint);
//        canvas.drawText(useTime+"",width/2,height/2,textPaint);
//        canvas.drawText(sb.toString(),width/2,height/2,textPaint);

    }

    public void updatePath(SAILS sails, List<SAILS.GeoNode> list){

        long startTime = System.currentTimeMillis();

        removeSimilarPoints(sails, list);

        ArrayList points = new ArrayList<FPoint>();

        int size = list.size();
        if(size < 2){
            return;
        }

        latitude = sails.getLatitude();
        longitude = sails.getLongitude();

        int count = Math.min(size,4);

        SAILS.GeoNode node1 = list.get(0);
        SAILS.GeoNode node2 = list.get(1);

        float[] pt1 = conversion.latLon2M(node1.latitude, node1.longitude);
        float[] pt2 = conversion.latLon2M(node2.latitude, node2.longitude);
        float[] firstPoint = conversion.latLon2M(latitude, longitude);

        float[] truePoint = getSimilarPoint(pt1[0], pt1[1], pt2[0], pt2[1], firstPoint[0], firstPoint[1]);

//        int[] firstPoint = conversion.latLon2M(firstNode.latitude, firstNode.longitude) ;
        for (int i = 1; i < count; i++) {
            SAILS.GeoNode geoNode = list.get(i);
            float[] floats = conversion.latLon2M(geoNode.latitude, geoNode.longitude);
            points.add(new FPoint(floats[0] - truePoint[0],truePoint[1] - floats[1]));
        }

        mPoints = points;

        useTime = System.currentTimeMillis() - startTime;

    }

    private static class FPoint{
        float x;
        float y;

        public FPoint(float x, float y) {
            this.x = x;
            this.y = y;
        }
    }

    /**
     * 将距离小于2米的点合并
     *
     * @param geoNodeList
     */
    private void removeSimilarPoints(SAILS sails, List<SAILS.GeoNode> geoNodeList) {
        if (geoNodeList == null) {
            return;
        }
        for (int i = 0; i < geoNodeList.size() - 1; i++) {
            SAILS.GeoNode geoNode = geoNodeList.get(i);
            SAILS.GeoNode nextGeoNode = geoNodeList.get(i + 1);
            double distance = getDistance(sails, geoNode, nextGeoNode);
            if (distance < IpsConstants.SIMILAR_POINTS_RADIUS && geoNode.floornumber == nextGeoNode.floornumber) {
                LocationRegion belongsRegion = nextGeoNode.BelongsRegion;
                if (belongsRegion != null) {
                    if (belongsRegion.self != 0) {
                        continue;
                    }
                }
                geoNodeList.remove(i + 1);
                i = -1;
            }
        }

    }

    private double mapDistanceByLngLat;

    private double getDistance(SAILS sails, SAILS.GeoNode geoNode1, SAILS.GeoNode geoNode2) {
        try{
            mapDistanceByLngLat = sails.getMapDistanceByLngLat(geoNode1.longitude, geoNode1.latitude, geoNode2.longitude, geoNode2.latitude);
            return mapDistanceByLngLat;
        }catch (Exception e){
            return mapDistanceByLngLat;
        }

    }

    public void updateAngle(float angle){
        this.angle = angle;

//        invalidate();

    }


    /** @ brief 根据两点求出垂线过第三点的直线的交点
     @ param pt1 直线上的第一个点
     @ param pt2 直线上的第二个点
     @ param pt3 定位的点
     @ return 返回点到直线上与定位点比较相似的点
     */
    private float[] getSimilarPoint(float p1x, float p1y, float p2x,float p2y,float p3x,float p3y) {
        float[] result = new float[2];

        float dx = (p2x - p1x)/50;
        float dy = (p2y - p1y)/50;

        float offsetX = 0;
        float offsetY = 0;

        double preDistance = Double.MAX_VALUE;

        for (int i = 0; i < 50; i++) {
            double distance = getDistance(p1x + offsetX, p1y + offsetY, p3x, p3y);
            if(distance > preDistance){
                result[0] = p1x + offsetX;
                result[1] = p1y + offsetY;
                return result;
            }
            preDistance = distance;
            offsetX += dx;
            offsetY += dy;
        }

        result[0] = p2x;
        result[1] = p2y;
        return result;
    }


    public double getDistance(float x1,float y1,float x2, float y2){
        double _x = Math.abs(x1 - x2);
        double _y = Math.abs(y1 - y2);
        return Math.sqrt(_x*_x+_y*_y);
    }


    public void startDraw(SAILSMapView mapView){
        this.mapView = mapView;
        needDraw = true;
        invalidate();
    }

    public void stopDraw(){
        needDraw = false;
        mapView = null;
    }



}
