package com.segway.robot.sdk.vision.utils;

import android.os.SystemClock;

import com.segway.robot.sdk.base.log.Logger;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;


/**
 * Created by gaominghui on 2019/3/14.
 ***/
public class DevicesUtils {


    private static final String TAG = DevicesUtils.class.getSimpleName();

    public static final String LEFT_BOTTOM = "usb1/1-2/1-2.3";
    public static final String RIGHT_BOTTOM = "usb1/1-2/1-2.4";
    public static final String LEFT_TOP = "usb2/2-4/2-4";

    public static final String LEFT_BOTTOM_RK = "-1.2";
    public static final String LEFT_TOP_RK = "-1.4";
    public static final String LEFT_TOP_RK2 = "-1.3";
    public static final String RIGHT_BOTTOM_RK= "-1.1";

    public static final String KERNEL_PORT_INFO_LEFT_BOTTOM = "1-2.3";
    public static final String KERNEL_PORT_INFO_RIGHT_BOTTOM = "1-2.4";
    public static final String KERNEL_PORT_INFO_TOP = "2-4";

    public static final int DS5_LEFT = 1;
    public static final int DS5_RIGHT = 2;
    public static final int DEPTH_TOP = 3;
    public static final int ALL_DEVICES = 0;

    private static final int CHECK_INTERVAL = 500;
    private static final int GAP_THRESHOLD = 1000;


    /**
     * @param depthType use DS5_LEFT,DS5_RIGHT,DEPTH_TOP to specify the depth you want check ，
     *                  pass ALL_DEVICES to check if there is any disconnection.
     * @return
     */
    public static boolean isDepthDevicesDisconnected(int depthType){
        boolean isLeftOn = false , isRightOn = false, isTopOn = false;

        String devPath = "/proc/bus/input/devices";
        List<String> lines = readFileByLine(devPath);

        switch (depthType){
            case DS5_LEFT:
                for (String line : lines){
                    if (line.contains(LEFT_BOTTOM) || line.contains(LEFT_BOTTOM_RK)){
                        Logger.d(TAG," ds5_left return false." + line);
                        return false;
                    }
                }
                break;
            case DS5_RIGHT:
                for (String line : lines){
                    if (line.contains(RIGHT_BOTTOM) || line.contains(RIGHT_BOTTOM_RK)){
                        Logger.d(TAG," ds5_right return false.");
                        return false;
                    }
                }
                break;
            case DEPTH_TOP:
                for (String line : lines){
                    if (line.contains(LEFT_TOP) || line.contains(LEFT_TOP_RK) || line.contains(LEFT_TOP_RK2)){
                        Logger.d(TAG," ds5_top return false.");
                        return false;
                    }
                }
                break;
            case ALL_DEVICES:
                for (String line : lines){
                    if (line.contains(LEFT_BOTTOM) || line.contains(LEFT_BOTTOM_RK)){
                        isLeftOn = true;
                        continue;
                    }
                    if (line.contains(RIGHT_BOTTOM) || line.contains(RIGHT_BOTTOM_RK)){
                        isRightOn = true;
                        continue;
                    }
                    if (line.contains(LEFT_TOP) || line.contains(LEFT_TOP_RK) || line.contains(LEFT_TOP_RK2)){
                        isTopOn = true;
                    }
                }
                Logger.d(TAG," check all devices return " + !(isLeftOn && isRightOn && isTopOn));
                return !(isLeftOn && isRightOn && isTopOn);
            default:
                Logger.e(TAG," unknown depth type of " + depthType);
                break;
        }
        return true;
    }

    private static List<String> readFileByLine(String devPath) {
        List<String> resStrings = new ArrayList<>();
        File file = new File(devPath);
        if (file.exists()) {
            FileInputStream fis = null;
            BufferedReader bufferedReader = null;
            InputStreamReader isr = null;
            try {
                fis = new FileInputStream(file);
                isr = new InputStreamReader(fis);
                bufferedReader = new BufferedReader(isr);
                String line;
                while ((line = bufferedReader.readLine()) != null) {
                    //Logger.d(TAG, line);
                    resStrings.add(line);
                }
            } catch (Exception e) {
                Logger.i(TAG,"doNothing2.");
            } finally {
                try {
                    if (bufferedReader != null) bufferedReader.close();
                    if (isr != null) isr.close();
                    if (fis != null) fis.close();
                } catch (IOException e1) {Logger.w(TAG,"doNothing1.");}
            }
        }
        return resStrings;
    }


    /**
     * @param depthType pass static constant field in com.segway.robot.sdk.vision.utils.DevicesUtils
     *                   DS5_LEFT,DS5_RIGHT,DEPTH_TOP
     * @return
     */
    public static boolean isDisconnectionEverLoggedInKernel(int depthType) {
        long elapsedRealtime = SystemClock.elapsedRealtime();
        List<String> errorInfoList = getErrorInfoList();
        for (String info : errorInfoList){
            switch (depthType){
                case DS5_LEFT:
                    if (info != null && info.contains(KERNEL_PORT_INFO_LEFT_BOTTOM)){
                        if (checkTime(info,elapsedRealtime)) return true;
                    }
                    break;
                case DS5_RIGHT:
                    if (info != null && info.contains(KERNEL_PORT_INFO_RIGHT_BOTTOM)){
                        if (checkTime(info,elapsedRealtime)) return true;
                    }
                    break;
                case DEPTH_TOP:
                    if (info != null && info.contains(KERNEL_PORT_INFO_TOP)){
                        if (checkTime(info,elapsedRealtime)) return true;
                    }
                    break;
                default:
                    break;
            }
        }
        return false;
    }

    private static boolean checkTime(String info,long elapsedRealtime) {

        String kernelTime = info.substring(info.indexOf("[") +1, info.indexOf("]"));
        double disconnectTime = 0;
        try{
            double time = Double.valueOf(kernelTime.trim());
            disconnectTime = time * 1000;
        } catch (Exception e){
            Logger.w(TAG,"exception in transfer log time. " + e.getLocalizedMessage());
        }
        Logger.d(TAG," elapsedRealtime="+elapsedRealtime+", disconnectTime="+disconnectTime+" time gap: " + Math.abs(elapsedRealtime - disconnectTime - CHECK_INTERVAL));
        if (Math.abs(elapsedRealtime - disconnectTime - CHECK_INTERVAL) < GAP_THRESHOLD){
            return true;
        }
        return false;
    }

    private static String getErrorInfo() {
        BufferedReader reader = null;
        try {
            String[] cmd = new String[]{"sh", "-c", "adb shell dmesg | grep \'USB disconnect\'"};
            Process process = Runtime.getRuntime().exec(cmd);
            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            StringBuffer output = new StringBuffer();
            int read;
            char[] buffer = new char[1024];
            while ((read = reader.read(buffer)) > 0) {
                output.append(buffer, 0, read);
            }
            return output.toString();
        } catch (IOException e) {
            e.printStackTrace();
            return e.getMessage();
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    private static List<String> getErrorInfoList() {
        BufferedReader reader = null;
        List<String> outputs = new ArrayList<>();
        try {
            String[] cmd = new String[]{"sh", "-c", "adb shell dmesg | grep \'USB disconnect\'"};
            Process process = Runtime.getRuntime().exec(cmd);
            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            String line;
            while((line = reader.readLine()) != null){
                outputs.add(line);
            }
        } catch (IOException e) {
            Logger.e(TAG," exception :" + e.getLocalizedMessage(),e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return outputs;
    }
}
