package com.segway.robot.sdk.vision.internal.ipc;

import com.segway.robot.sdk.base.log.Logger;

import java.io.IOException;

/**
 * Created by ark338 on 2016/11/7.
 * Holds an array of MemoryFileBuffer
 */
public class MemoryFileBufferQueue {
    private static final String TAG = "MemoryFileBufferQueue";
    private static final int BUFFER_SIZE = 5;
    private MemoryFileBuffer[] mMemoryFileBuffers = new MemoryFileBuffer[BUFFER_SIZE];
    private long[] mMemoryFileBufferUseTime = new long[BUFFER_SIZE];
    private int mImageSize;
    private int mLogCount;
    private int mAcquired;
    private int mReleased;


    public MemoryFileBufferQueue(int imageSize) throws IOException {
        mImageSize = imageSize;
        for (int i = 0; i < mMemoryFileBuffers.length; i++) {
            mMemoryFileBuffers[i] = new MemoryFileBuffer(imageSize, i);
        }
    }

    public MemoryFileBuffer next() {
        mLogCount++;
        if(mLogCount > 100) {
            mLogCount = 0;
            Logger.v(TAG, countUsage());
        }

        for (int i = 0; i < BUFFER_SIZE; i++) {
            MemoryFileBuffer memoryFileBuffer = mMemoryFileBuffers[i];
            if (!memoryFileBuffer.isDirty()) {
                memoryFileBuffer.setDirty(true);
                mAcquired++;
                mMemoryFileBufferUseTime[i] = System.currentTimeMillis();
                return memoryFileBuffer;
            }
        }

        // get the least recent used buffer and force close it
        int index = 0;
        long least = Long.MAX_VALUE;
        for (int i = 0; i < BUFFER_SIZE; i++) {
            if (mMemoryFileBufferUseTime[i] < least) {
                least = mMemoryFileBufferUseTime[i];
                index = i;
            }
        }

        Logger.w(TAG, "memory file buffer used up, swap new, index = " + index);

        try {
            MemoryFileBuffer newMemoryFileBuffer = new MemoryFileBuffer(mImageSize, index);
            MemoryFileBuffer memoryFileBuffer = mMemoryFileBuffers[index];
            memoryFileBuffer.getImageMemoryFile().close();
            memoryFileBuffer.getInfoMemoryFile().close();
            mMemoryFileBuffers[index] = newMemoryFileBuffer;
            mMemoryFileBufferUseTime[index] = System.currentTimeMillis();
            return newMemoryFileBuffer;
        } catch (IOException e) {
            Logger.e(TAG, "create new memory file error", e);
        }

        return null;
    }

    private String countUsage() {
        StringBuffer stringBuffer = new StringBuffer();
        stringBuffer.append("count usage size = (" + mImageSize + ") usage:");
        int count = 0;
        for (MemoryFileBuffer memoryFileBuffer : mMemoryFileBuffers) {
            if (memoryFileBuffer.isDirty()) {
                count++;
                stringBuffer.append("[1]");
            } else {
                stringBuffer.append("[0]");
            }
        }
        stringBuffer.append(" buffer used:" + count);
        stringBuffer.append("\n")
                .append("acquired:" + mAcquired)
                .append(" released:" + mReleased)
                .append(" delta:" + (mAcquired - mReleased));
        return stringBuffer.toString();
    }

    public void releaseBuffer(int index) {
        mReleased++;
        mMemoryFileBuffers[index].setDirty(false);
    }

    public void close() {
        Logger.i(TAG, "closing...");
        for (MemoryFileBuffer memoryFileBuffer : mMemoryFileBuffers) {
            memoryFileBuffer.getImageMemoryFile().close();
            memoryFileBuffer.getInfoMemoryFile().close();
            memoryFileBuffer.setDirty(false);
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }
}
