package com.segway.robot.sdk.vision.internal.socket;

import android.net.LocalSocket;
import android.net.LocalSocketAddress;

import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.vision.ImageStreamCallback;
import com.segway.robot.sdk.vision.frame.FrameInfo;
import com.segway.robot.sdk.vision.internal.framebuffer.FrameBuffer;
import com.segway.robot.sdk.vision.internal.framebuffer.RecyclableFrame;
import com.segway.robot.sdk.vision.stream.StreamInfo;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by ark338 on 16/3/15.
 */
public class ImageReaderThreadManager {
    private static final String TAG = "ImageReaderThreadManage";
    private Map<Integer, Thread> mWorkThreadMap = new HashMap<>();

    private static ImageReaderThreadManager mInstance;
    private ImageReaderThreadManager(){}
    public static synchronized ImageReaderThreadManager getInstance() {
        if (mInstance == null) {
            mInstance = new ImageReaderThreadManager();
        }
        return mInstance;
    }

    public synchronized void addConnection(final StreamInfo streamInfo,
                           String address, final ImageStreamCallback callback) throws IOException {
        final LocalSocket localSocket = new LocalSocket();
        try {
            localSocket.connect(new LocalSocketAddress(address));
            final InputStream inputStream = localSocket.getInputStream();
            Thread workThread = new Thread("Image Transfer Thread, Type:" + streamInfo.getStreamType()){
                @Override
                public void run() {
                    int bufferLength = (int)(streamInfo.getPixelBytes() * streamInfo.getHeight() * streamInfo.getWidth());
                    ByteBuffer frameInfoBuffer = ByteBuffer.allocate(FrameInfo.SIZE);
                    ByteBuffer imageBuffer = ByteBuffer.allocate(bufferLength);
                    try {
                        while (!isInterrupted()) {
                            try {
                                // read frame info
                                int cnt = 0;
                                frameInfoBuffer.rewind();
                                while (cnt < FrameInfo.SIZE && !isInterrupted()) {
                                    int len = inputStream.read(frameInfoBuffer.array(), cnt, FrameInfo.SIZE - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                // read image
                                cnt = 0;
                                imageBuffer.rewind();
                                while (cnt < bufferLength && !isInterrupted()) {
                                    int len = inputStream.read(imageBuffer.array(), cnt, bufferLength - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                FrameInfo frameInfo = FrameInfo.fromByteBuffer(frameInfoBuffer);
                                callback.onNewImage(frameInfo, imageBuffer);
                            } catch (IOException e) {
                                Logger.v(TAG, "socket exception:" + e.getMessage());
                                break;
                            }
                        }
                    } finally {
                        try {
                            localSocket.close();
                        } catch (IOException e) {
                        }
                    }
                }
            };

            mWorkThreadMap.put(streamInfo.getStreamType(), workThread);

            workThread.start();
        } catch (IOException e) {
            Logger.e(TAG, "connect socket error!", e);
            throw e;
        }
    }

    public synchronized void addConnection(final StreamInfo streamInfo,
                                           String address, final FrameBuffer frameBuffer) throws IOException {
        final LocalSocket localSocket = new LocalSocket();
        try {
            localSocket.connect(new LocalSocketAddress(address));
            final InputStream inputStream = localSocket.getInputStream();
            Thread workThread = new Thread("Image Transfer Thread, Type:" + streamInfo.getStreamType()){
                @Override
                public void run() {
                    ByteBuffer frameInfoBuffer = ByteBuffer.allocate(FrameInfo.SIZE);
                    int imageSize = (int) (streamInfo.getPixelBytes() * streamInfo.getWidth() * streamInfo.getHeight());
                    try {
                        while (!isInterrupted()) {
                            try {
                                // read frame info
                                int cnt = 0;
                                frameInfoBuffer.rewind();
                                while (cnt < FrameInfo.SIZE && !isInterrupted()) {
                                    int len = inputStream.read(frameInfoBuffer.array(), cnt, FrameInfo.SIZE - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                FrameInfo frameInfo = FrameInfo.fromByteBuffer(frameInfoBuffer);
                                RecyclableFrame recyclableFrame = RecyclableFrame.create(imageSize, frameInfo);
                                // read image
                                ByteBuffer imageBuffer = recyclableFrame.getByteBuffer();
                                cnt = 0;
                                imageBuffer.rewind();
                                while (cnt < imageSize && !isInterrupted()) {
                                    int len = inputStream.read(imageBuffer.array(), cnt, imageSize - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }
                                imageBuffer.rewind();
                                frameBuffer.add(recyclableFrame);
                            } catch (IOException e) {
                                Logger.v(TAG, "socket exception:" + e.getMessage());
                                break;
                            }
                        }
                    } finally {
                        try {
                            localSocket.close();
                        } catch (IOException e) {
                        }
                    }
                }
            };

            mWorkThreadMap.put(streamInfo.getStreamType(), workThread);

            workThread.start();
        } catch (IOException e) {
            Logger.e(TAG, "connect socket error!", e);
            throw e;
        }
    }

    public synchronized void removeConnection(StreamInfo streamInfo) {
        Thread workThread = mWorkThreadMap.get(streamInfo.getStreamType());
        if (workThread != null) {
            workThread.interrupt();
            mWorkThreadMap.remove(streamInfo.getStreamType());
        }
    }

}
