package com.segway.robot.sdk.vision.internal.framebuffer;

import android.util.SparseArray;

import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.time.StampedData;
import com.segway.robot.sdk.vision.frame.FrameInfo;
import com.segway.robot.sdk.vision.stream.Resolution;

import java.nio.ByteBuffer;
import java.util.Stack;

/**
 * Recyclable frame
 */
public class RecyclableFrame extends BaseFrame {
    static SparseArray<Stack<RecyclableFrame>> mFramePool = new SparseArray<>();
    ByteBuffer mByteBuffer;
    FrameInfo mFrameInfo;
    private int mBufferSize;

    private RecyclableFrame (int bufferSize, FrameInfo frameInfo) {
        super();
        mBufferSize = bufferSize;
        mFrameInfo = frameInfo;
        mByteBuffer = ByteBuffer.allocateDirect(bufferSize);
    }

    public void reuse(FrameInfo frameInfo) {
        super.reuse();
        mFrameInfo = frameInfo;
    }

    @Override
    public ByteBuffer getByteBuffer() {
        mByteBuffer.rewind();
        return mByteBuffer;
    }

    @Override
    public FrameInfo getInfo() {
        return mFrameInfo;
    }

    @Override
    public void release() {
        synchronized (RecyclableFrame.class) {
            super.release();
            mFramePool.get(mBufferSize).push(this);
        }
    }

    public synchronized static RecyclableFrame create(int bufferSize, FrameInfo frameInfo) {
        if (mFramePool.get(bufferSize) == null) {
            mFramePool.put(bufferSize, new Stack());
        }

        if (mFramePool.get(bufferSize).empty()) {
            return new RecyclableFrame(bufferSize, frameInfo);
        }

        RecyclableFrame recyclableFrame = mFramePool.get(bufferSize).pop();
        recyclableFrame.reuse(frameInfo);
        return recyclableFrame;
    }

    @Override
    public long getTimestamp() {
        return mFrameInfo.getIMUTimeStamp();
    }

    @Override
    public long diff(StampedData source) {
        return source.getTimestamp() - getTimestamp();
    }

    /**
     * pack a RecyclableFrame using ByteBuffer.
     *
     */
    public static RecyclableFrame packFrame(ByteBuffer frame, long timestamp, int streamType, long num,
                                            int width, int height, int bytesPerPixel, int pixelFormat) {
        //Logger.d("RecyclableFrame","  timestamp->" + timestamp + " type : " + streamType);
        RecyclableFrame recyclableFrame;
        int imageSize = frame.capacity();
        int realSize = width * height * bytesPerPixel;

        if (imageSize != realSize) {
            Logger.e("RecyclableFrame" , "jniOnNewFrame->right imageSize = " + imageSize + "; realSize=" +
                    realSize);
            return null;
        }

        int resolution = Resolution.toVisionServiceResolution(width, height);
        FrameInfo frameInfo = new FrameInfo(streamType, resolution, pixelFormat);
        frameInfo.setFrameNum((int) num);
        frameInfo.setPlatformTimeStamp(timestamp);
        frameInfo.setExposure(0);
        frameInfo.setStride(0);
        recyclableFrame = RecyclableFrame.create(imageSize, frameInfo);
        // read image
        recyclableFrame.getByteBuffer().put(frame);
        //Logger.d("RecycleableFrame","  packFrame timestamp -> "+"type = "+ streamType+" ts ->" + recyclableFrame.getInfo().getPlatformTimeStamp() +
        //"recyclableFrame.getTimeStamp() " + recyclableFrame.getTimestamp() + "systemPlatform timestamp -> " + System.currentTimeMillis());

        return recyclableFrame;
    }

}
