package com.segway.robot.sdk.vision.imu;


import android.content.Context;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.util.Log;
import android.util.SparseArray;

import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.vision.calibration.RS2Intrinsic;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;


/**
 * Created by gaominghui on 2018/10/15.
 ***/
public class IMUSensor implements IIMUSensor, Handler.Callback {

    private static String TAG = IMUSensor.class.getSimpleName();
    private SensorManager mSensorManager;
    private final static int MAX_FRAME = 10;
    private final static int FRAME_LENGTH = 48;
    private static final int TYPE_GYR = 1;
    private static final int TYPE_ACC = 2;
    private final ByteBuffer mByteBufferEvents;
    private final AtomicBoolean isOpen = new AtomicBoolean(false);
    private Context mContext;
    private volatile int mIntCursor = 0;
    private volatile AtomicLong mLongSequence = new AtomicLong(0L);
    private Sensor mSensorAcc;
    private Sensor mSensorGyr;

    public static final int ADD_IMU = 5;
    public static final int REMOVE_IMU = 6;
    public static final int NOTIFY_IMU = 7;
    public static final int OPEN_IMU = 8;
    public static final int CLOSE_IMU = 9;

    private SensorEventListener mSensorEventListenerACC;
    private SensorEventListener mSensorEventListenerGyr;
    private Handler mIMUHandler = null;
    private final SparseArray<IIMUDataCallback> mIMUMap = new SparseArray<>();


    private IMUSensor(@NonNull Context context) {
        this.mContext = context;
        mByteBufferEvents = ByteBuffer.allocate(MAX_FRAME * FRAME_LENGTH);
        mByteBufferEvents.order(ByteOrder.LITTLE_ENDIAN);
    }

    public static IMUSensor newInstance(Context context) {
        return new IMUSensor(context);
    }

    @Override
    public void open() {
        synchronized (isOpen) {
            if (mIMUHandler == null) {
                HandlerThread thread = new HandlerThread("imu_callback");
                thread.start();
                mIMUHandler = new Handler(thread.getLooper(), this);
            }
        }
        mIMUHandler.sendEmptyMessage(OPEN_IMU);
    }

    private void doOpen() {
        Logger.i(TAG, "doOpen() isOpen()=" + isOpen.get());
        if (isOpen.get()) {
            Logger.w(TAG, "open: IMUSensor is already open.");
            return;
        }

        mSensorManager = (SensorManager) mContext.getSystemService(Context.SENSOR_SERVICE);
        if (mSensorManager == null) {
            Log.e(TAG, "IMUSensor: sensorManager is null.");
            return;
        }
        mSensorAcc = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        mSensorGyr = mSensorManager.getDefaultSensor(Sensor.TYPE_GYROSCOPE);

        mSensorEventListenerACC = new SensorEventListener() {
            @Override
            public void onSensorChanged(SensorEvent event) {
//                Logger.d(TAG,"imu== type: " + TYPE_ACC + " timestamp: " + event.timestamp );
                synchronized (mByteBufferEvents) {
                    inflateIMUFrame(event, TYPE_ACC);
                    if (mIntCursor == MAX_FRAME - 1) {
                        callNotifyIMUData();
                    } else {
                        mIntCursor++;
                    }
                }
            }

            @Override
            public void onAccuracyChanged(Sensor sensor, int accuracy) {
                Log.d(TAG, "onAccuracyChanged: ");
            }
        };

        mSensorEventListenerGyr = new SensorEventListener() {
            @Override
            public void onSensorChanged(SensorEvent event) {
                //Logger.d(TAG,"imu== type: " + TYPE_GYR + " timestamp: " + event.timestamp );
                synchronized (mByteBufferEvents) {
                    inflateIMUFrame(event, TYPE_GYR);
                    if (mIntCursor == MAX_FRAME - 1) {
                        callNotifyIMUData();
                    } else {
                        mIntCursor++;
                    }
                }
            }

            @Override
            public void onAccuracyChanged(Sensor sensor, int accuracy) {

            }
        };

        mSensorManager.registerListener(mSensorEventListenerACC, mSensorAcc, 1000 * 1000 / 208);
        mSensorManager.registerListener(mSensorEventListenerGyr, mSensorGyr, 1000 * 1000 / 208);
        Logger.d(TAG, "open: registerListeners.");
        isOpen.set(true);
    }

    private void doClose() {
        Log.i(TAG, "doClose: isOpen()=" + isOpen.get());
        if (!isOpen.get()) {
            Logger.w(TAG, "release: IMUSensor is not open.");
            return;
        }
        if (mSensorManager != null) {
            mSensorManager.unregisterListener(mSensorEventListenerACC);
            mSensorManager.unregisterListener(mSensorEventListenerGyr);
            Logger.d(TAG, "release: unregisterListeners.");
        }
        mSensorManager = null;
        mSensorAcc = null;
        mSensorGyr = null;
        mSensorEventListenerGyr = null;
        mSensorEventListenerACC = null;
        isOpen.set(false);
        mIMUMap.clear();

        mIMUHandler.removeCallbacksAndMessages(null);
        mIMUHandler.getLooper().quit();
        mIMUHandler = null;
    }

    private void callNotifyIMUData() {
        mIntCursor = 0;
        byte[] resBytes = new byte[mByteBufferEvents.capacity()];
        mByteBufferEvents.get(resBytes);
        ByteBuffer resBuffer = ByteBuffer.wrap(resBytes);
        Message msg = Message.obtain();
        msg.what = NOTIFY_IMU;
        msg.obj = resBuffer;
        msg.arg1 = FRAME_LENGTH;
        msg.arg2 = MAX_FRAME;
        mIMUHandler.sendMessage(msg);
        mByteBufferEvents.clear();
    }

    private void inflateIMUFrame(SensorEvent event, int type) {
        int frameOffset = mIntCursor * FRAME_LENGTH;
        mByteBufferEvents.putLong(frameOffset, event.timestamp);
        mByteBufferEvents.putLong(frameOffset + 8, mIntCursor);
        mByteBufferEvents.putInt(frameOffset + 16, type);
        mByteBufferEvents.putFloat(frameOffset + 24, event.values[0]);
        mByteBufferEvents.putFloat(frameOffset + 28, event.values[1]);
        mByteBufferEvents.putFloat(frameOffset + 32, event.values[2]);
        mByteBufferEvents.putLong(frameOffset + 40, event.timestamp / 1000);
    }

    @Override
    public void close() {
        this.release();
    }

    @Override
    public void release() {
        if (mIMUHandler != null) {
            mIMUHandler.sendEmptyMessage(CLOSE_IMU);
        }
    }

    public synchronized boolean isOpen() {
        return isOpen.get();
    }

    @Override
    public void addIMUDataCallback(int pid, IIMUDataCallback.Stub callback) {
        if (mIMUHandler != null) {
            mIMUHandler.sendMessage(mIMUHandler.obtainMessage(ADD_IMU, pid, pid, callback));
        }
    }

    @Override
    public void removeIMUDataCallback(int pid) {
        if (mIMUHandler != null) {
            mIMUHandler.sendMessage(mIMUHandler.obtainMessage(REMOVE_IMU, pid, pid));
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        Log.w(TAG, "finalize: " + this.getClass().getSimpleName());
    }

    @Override
    public RS2Intrinsic getRS2Intrinsic() {
        return null;
    }

    @Override
    public boolean handleMessage(Message msg) {
        switch (msg.what) {
            case ADD_IMU:
                mIMUMap.put(msg.arg1, (IIMUDataCallback) msg.obj);
                break;
            case REMOVE_IMU:
                mIMUMap.remove(msg.arg1);
                break;
            case OPEN_IMU:
                doOpen();
                break;
            case CLOSE_IMU:
                doClose();
                break;
            case NOTIFY_IMU:
                notifyIMUData(msg);
                break;
            default:
        }
        return true;
    }

    private void notifyIMUData(Message msg) {
        for (int i = 0; i < mIMUMap.size(); i++) {
            IIMUDataCallback callback = mIMUMap.valueAt(i);
            try {
                if (callback == null) continue;
                callback.onNewData(((ByteBuffer) msg.obj).array(), msg.arg1, msg.arg2);
            } catch (RemoteException e) {
                Logger.e(TAG, "notifyIMUData: remote exception.");
                mIMUMap.removeAt(i);
                break;
            }
        }
    }
}
