package com.segway.robot.sdk.vision.person;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.os.Parcel;
import android.os.Parcelable;

import com.segway.robot.sdk.vision.tools.DataWithConfidence;

/**
 * Person is an instance generated by Person Detection.
 */
public class Person implements Parcelable{
    private int m_id;
    private DataWithConfidence<Rect> m_boundingRect;
    private DataWithConfidence<PointF> m_imageMassCenter;
    private DataWithConfidence<Point3DF> m_worldMassCenter;

    /**
     * Constructor of a person.
     * @param id the person ID.
     * @param boundingRect the rectangle bounding a person on an image, with confidence.
     * @param image the central point of a person in 2D space, with confidence.
     * @param world the central point of a person in 3D space, with confidence.
     */
    public Person(int id, DataWithConfidence<Rect> boundingRect, DataWithConfidence<PointF> image, DataWithConfidence<Point3DF> world) {
        this.m_id = id;
        this.m_boundingRect = boundingRect;
        this.m_imageMassCenter = image;
        this.m_worldMassCenter = world;
    }

    /**
     * Get the person ID.
     * @return the person ID.
     */
    public int getId() {
        return this.m_id;
    }

    /**
     * Get the rectangle bounding a person on an image, with confidence.
     * @return the rectangle bounding a person on an image, with confidence.
     */
    public DataWithConfidence<Rect> getBounds() {
        return this.m_boundingRect;
    }

    /**
     * Get the central point of a person in 2D space, with confidence.
     * @return the central point of a person in 2D space, with confidence.
     */
    public DataWithConfidence<PointF> getImageMassCenter() {
        return this.m_imageMassCenter;
    }

    /**
     * Get the central point of a person in 3D space, with confidence.
     * @return the central point of a person in 3D space, with confidence.
     */
    public DataWithConfidence<Point3DF> getWorldMassCenter() {
        return this.m_worldMassCenter;
    }

    /**
     * Draw a person rectangle on a Bitmap.
     * @param bitmap an image in Bitmap.
     */
    public void drawFaceRect(Bitmap bitmap) {
        Canvas canvas = new Canvas(bitmap);
        Paint paint = new Paint();
        Rect rect = (Rect) m_boundingRect.data;
        paint.setColor(Color.GREEN);
        paint.setTextSize(30f);
        paint.setStrokeWidth(5);
        paint.setStyle(Paint.Style.STROKE);
        canvas.drawRect(rect, paint);
        canvas.drawText(Integer.toString(m_id), rect.left, rect.top + 15, paint);
    }

    protected Person(Parcel in) {
        m_id = in.readInt();
        m_boundingRect = in.readParcelable(DataWithConfidence.class.getClassLoader());
        m_imageMassCenter = in.readParcelable(DataWithConfidence.class.getClassLoader());
        m_worldMassCenter = in.readParcelable(DataWithConfidence.class.getClassLoader());
    }

    /**
     * Create a person from Parcel.
     */
    public static final Creator<Person> CREATOR = new Creator<Person>() {
        @Override
        public Person createFromParcel(Parcel in) {
            return new Person(in);
        }

        @Override
        public Person[] newArray(int size) {
            return new Person[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(m_id);
        dest.writeParcelable(m_boundingRect, flags);
        dest.writeParcelable(m_imageMassCenter, flags);
        dest.writeParcelable(m_worldMassCenter, flags);
    }

}
