package com.segway.robot.sdk.vision.internal.framebuffer;

import android.util.SparseArray;

import com.segway.robot.sdk.base.time.StampedData;
import com.segway.robot.sdk.vision.frame.FrameInfo;

import java.nio.ByteBuffer;
import java.util.Stack;

/**
 * Recyclable frame
 */
public class RecyclableFrame extends BaseFrame {
    static SparseArray<Stack<RecyclableFrame>> mFramePool = new SparseArray<>();
    ByteBuffer mByteBuffer;
    FrameInfo mFrameInfo;
    private int mBufferSize;

    private RecyclableFrame (int bufferSize, FrameInfo frameInfo) {
        super();
        mBufferSize = bufferSize;
        mFrameInfo = frameInfo;
        mByteBuffer = ByteBuffer.allocateDirect(bufferSize);
    }

    public void reuse(FrameInfo frameInfo) {
        super.reuse();
        mFrameInfo = frameInfo;
    }

    @Override
    public ByteBuffer getByteBuffer() {
        mByteBuffer.rewind();
        return mByteBuffer;
    }

    @Override
    public FrameInfo getInfo() {
        return mFrameInfo;
    }

    @Override
    public void release() {
        synchronized (RecyclableFrame.class) {
            super.release();
            mFramePool.get(mBufferSize).push(this);
        }
    }

    public synchronized static RecyclableFrame create(int bufferSize, FrameInfo frameInfo) {
        if (mFramePool.get(bufferSize) == null) {
            mFramePool.put(bufferSize, new Stack());
        }

        if (mFramePool.get(bufferSize).empty()) {
            return new RecyclableFrame(bufferSize, frameInfo);
        }

        RecyclableFrame recyclableFrame = mFramePool.get(bufferSize).pop();
        recyclableFrame.reuse(frameInfo);
        return recyclableFrame;
    }

    @Override
    public long getTimestamp() {
        return mFrameInfo.getIMUTimeStamp();
    }

    @Override
    public long diff(StampedData source) {
        return source.getTimestamp() - getTimestamp();
    }
}
