package com.segway.robot.sdk.vision;

import android.annotation.SuppressLint;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.graphics.Rect;
import android.os.Binder;
import android.os.Environment;
import android.os.IBinder;
import android.os.ParcelFileDescriptor;
import android.os.RemoteException;
import android.util.Log;
import android.util.SparseArray;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.view.SurfaceView;

import com.segway.robot.algo.dts.BaseControlCommand;
import com.segway.robot.algo.dts.DTSPerson;
import com.segway.robot.algo.dts.FollowTrajectory;
import com.segway.robot.algo.dts.IPersonDetectionListener;
import com.segway.robot.algo.dts.IPersonTrackingListener;
import com.segway.robot.algo.dts.IPersonTrackingWithTrajectoryListener;
import com.segway.robot.algo.dts.IPlannerPersonTackingListener;
import com.segway.robot.algo.dts.PersonDetectListener;
import com.segway.robot.algo.dts.PersonDetectionProfile;
import com.segway.robot.algo.dts.PersonTrackingListener;
import com.segway.robot.algo.dts.PersonTrackingProfile;
import com.segway.robot.algo.dts.PersonTrackingWithPlannerListener;
import com.segway.robot.algo.dts.PersonTrackingWithTrajectoryListener;
import com.segway.robot.algo.generaldts.DTSObject;
import com.segway.robot.algo.generaldts.GeneralTrackingListener;
import com.segway.robot.algo.generaldts.GeneralTrackingProfile;
import com.segway.robot.algo.generaldts.IGeneralTrackingListener;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;
import com.segway.robot.sdk.vision.calibration.ColorDepthCalibration;
import com.segway.robot.sdk.vision.calibration.MotionModuleCalibration;
import com.segway.robot.sdk.vision.calibration.RS2Intrinsic;
import com.segway.robot.sdk.vision.error.DTSError;
import com.segway.robot.sdk.vision.frame.DepthFrameInfo;
import com.segway.robot.sdk.vision.frame.FishEyeFrameInfo;
import com.segway.robot.sdk.vision.frame.Frame;
import com.segway.robot.sdk.vision.frame.FrameInfo;
import com.segway.robot.sdk.vision.imu.IIMUDataCallback;
import com.segway.robot.sdk.vision.imu.IMUCallbackBundle;
import com.segway.robot.sdk.vision.imu.IMUDataCallback;
import com.segway.robot.sdk.vision.internal.ImageTransferType;
import com.segway.robot.sdk.vision.internal.VisionServiceState;
import com.segway.robot.sdk.vision.internal.framebuffer.CountableFrame;
import com.segway.robot.sdk.vision.internal.framebuffer.FrameBuffer;
import com.segway.robot.sdk.vision.internal.framebuffer.RecyclableFrame2;
import com.segway.robot.sdk.vision.internal.ipc.MemoryFileBuffer;
import com.segway.robot.sdk.vision.internal.ipc.MemoryFileBufferCallback;
import com.segway.robot.sdk.vision.internal.socket.FrameClientThread;
import com.segway.robot.sdk.vision.internal.socket.ImageReaderThreadManager;
import com.segway.robot.sdk.vision.params.VisionParamID;
import com.segway.robot.sdk.vision.person.FindPersonsHandler;
import com.segway.robot.sdk.vision.person.Person;
import com.segway.robot.sdk.vision.person.PersonDetectCallback;
import com.segway.robot.sdk.vision.stream.ExtDepthStreamFactory;
import com.segway.robot.sdk.vision.stream.FrameRate;
import com.segway.robot.sdk.vision.stream.PixelFormat;
import com.segway.robot.sdk.vision.stream.Resolution;
import com.segway.robot.sdk.vision.stream.StreamInfo;
import com.segway.robot.sdk.vision.stream.StreamType;
import com.segway.robot.sdk.vision.stream.StreamType.VisionStreamType;

import java.io.ByteArrayInputStream;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * the entry of the vision service.
 */
public class VisionServiceManager implements ServiceBinder {
    private static final String TAG = "VisionServiceManager";
    private static final String SERVICE_PACKAGE_NAME = "com.segway.robot.host.coreservice.vision";
    private static final String SERVICE_CLASS_NAME = "com.segway.robot.host.coreservice.vision.AprSenseService";
    private static final String LOGGER_CONFIG_FILE_PATH = Environment.getExternalStorageDirectory().getPath() + "/logger/VisionSDK.config";
    private static final int SURFACE_HOLDER_CALLBACK_TAG_KEY = 0x29cad07c;
    private static final int SURFACE_HOLDER_PROFILE_TAG_KEY = 0x29cad07d;
    private static final int IMU_DATA_SIZE = 800;
    private static VisionServiceManager mInstance;
    private Context mContext;
    private BindStateListener mListener;
    private Object mDummy = new Object();
    private IAprSenseService mAprSenseAIDLService;

    private AtomicBoolean mDTSEnabled = new AtomicBoolean(false);
    private PersonDetectListener mPersonDetectListener = null;
    private PersonTrackingListener mPersonTrackingListener = null;
    private PersonTrackingWithPlannerListener mPlannerPersonTrackingListener = null;
    private PersonTrackingWithTrajectoryListener mPersonTrackingWithTrajectoryListener = null;
    private GeneralTrackingListener mGeneralTrackingListener = null;

    final private List<VisionServiceCallback> mVisionServiceCallbacks = new ArrayList<>();
    final private List<AprSenseStateCallback> mAprSenseStateCallback = new ArrayList<>();
    final private List<PersonDetectCallback> mPersonDetectCallback = new ArrayList<>();
    final private List<IMUCallbackBundle> mIMUDataCallbacks = new ArrayList<>();
    final private SparseArray<Object> mImageCallbackMap = new SparseArray<>();
    final private SparseArray<Object> mPreviewMap = new SparseArray<>();
    final private SparseArray<FrameBuffer> mFrameBufferMap = new SparseArray<>();
    final private SparseArray<FrameClientThread> mFrameTransferMap = new SparseArray<>();
    final private SparseArray<SharedMemoryHolder> mFrameTransferSharedMemoryMap = new SparseArray<>();

    final private SparseArray<ByteBuffer> mMemoryFileBufferCacheArray = new SparseArray<>();

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            // get remote service stub
            mAprSenseAIDLService = IAprSenseService.Stub.asInterface(service);

            afterOnServiceConnected();
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            mAprSenseAIDLService = null;
            notifyServiceDisconnected();
            mListener.onUnbind("Service disconnected.");
            mContext = null;
            synchronized (mFrameTransferMap) {
                for (int i = 0; i < mFrameTransferMap.size(); i++) {
                    FrameClientThread frameClientThread = mFrameTransferMap.valueAt(i);
                    frameClientThread.disconnect();
                }
                mFrameTransferMap.clear();
            }

            synchronized (mFrameTransferSharedMemoryMap) {
                for (int i = 0; i < mFrameTransferSharedMemoryMap.size(); i++) {
                    SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.valueAt(i);
                    sharedMemoryHolder.parcelFileDescriptor = null;
                    try {
                        sharedMemoryHolder.inputStream.close();
                    } catch (IOException e) {
                    }
                    sharedMemoryHolder.inputStream = null;

                }
                mFrameTransferSharedMemoryMap.clear();
            }

            synchronized (mImageCallbackMap) {
                mImageCallbackMap.clear();
            }

            synchronized (mFrameBufferMap) {
                for (int i = 0; i < mFrameBufferMap.size(); i++) {
                    FrameBuffer frameBuffer = mFrameBufferMap.valueAt(i);
                    frameBuffer.release();
                }
                mFrameBufferMap.clear();
                Logger.d(TAG, "onServiceDisconnect() mFrameBufferMap clear()");
            }

            synchronized (mPreviewMap) {
                mPreviewMap.clear();
            }

            synchronized (mIMUDataCallbacks) {
                mIMUDataCallbacks.clear();
            }

            synchronized (mMemoryFileBufferCacheArray) {
                mMemoryFileBufferCacheArray.clear();
            }

            Logger.d(TAG, "VisionService disconnected.");
        }
    };

    private IPersonDetectionListener.Stub mIPersonDetectionListener = new IPersonDetectionListener.Stub() {
        @Override
        public void onPersonDetected(DTSPerson[] person) throws RemoteException {
            if (mPersonDetectListener != null) {
                mPersonDetectListener.onPersonDetected(person);
            }
        }

        @Override
        public void onPersonDetectionResult(DTSPerson[] person) throws RemoteException {
            if (mPersonDetectListener != null) {
                mPersonDetectListener.onPersonDetectionResult(person);
            }
        }

        @Override
        public void onPersonDetectionError(int errorCode, String message) throws RemoteException {
            if (mPersonDetectListener != null) {
                mPersonDetectListener.onPersonDetectionError(errorCode, message);
            }
        }
    };

    private IPersonTrackingListener mIPersonTrackingListener = new IPersonTrackingListener.Stub() {
        @Override
        public void onPersonTracking(DTSPerson person) throws RemoteException {
            if (mPersonTrackingListener != null) {
                mPersonTrackingListener.onPersonTracking(person);
            }
        }

        @Override
        public void onPersonTrackingResult(DTSPerson person) throws RemoteException {
            if (mPersonTrackingListener != null) {
                mPersonTrackingListener.onPersonTrackingResult(person);
            }
        }

        @Override
        public void onPersonTrackingError(int error, String reason) throws RemoteException {
            if (mPersonTrackingListener != null) {
                mPersonTrackingListener.onPersonTrackingError(error, reason);
            }
        }
    };

    private IPlannerPersonTackingListener mIPlannerPersonTackingListener = new IPlannerPersonTackingListener.Stub() {
        @Override
        public void onPlannerPersonTrackingResult(DTSPerson person, BaseControlCommand command) throws RemoteException {
            if (mPlannerPersonTrackingListener != null) {
                mPlannerPersonTrackingListener.onPersonTrackingWithPlannerResult(person, command);
            }
        }

        @Override
        public void onPlannerPersonTrackingError(int error, String reason) throws RemoteException {
            if (mPlannerPersonTrackingListener != null) {
                mPlannerPersonTrackingListener.onPersonTrackingWithPlannerError(error, reason);
            }
        }
    };

    private IPersonTrackingWithTrajectoryListener mIPersonTrackingWithTrajectoryListener = new IPersonTrackingWithTrajectoryListener.Stub() {
        @Override
        public void onPersonTrackingWithTrajectoryResult(DTSPerson person, FollowTrajectory followTrajectory) throws RemoteException {
            if (mPersonTrackingWithTrajectoryListener != null) {
                mPersonTrackingWithTrajectoryListener.onPersonTrackingWithTrajectoryResult(person, followTrajectory);
            }
        }

        @Override
        public void onPersonTrackingWithTrajectoryError(int errorCode, String message) throws RemoteException {
            if (mPersonTrackingWithTrajectoryListener != null) {
                mPersonTrackingWithTrajectoryListener.onPersonTrackingWithTrajectoryError(errorCode, message);
            }
        }
    };

    private IGeneralTrackingListener mIGeneralTrackingListener = new IGeneralTrackingListener.Stub() {
        @Override
        public void onObjectTrackingResult(DTSObject dtsObject) throws RemoteException {
            if (mGeneralTrackingListener != null) {
                mGeneralTrackingListener.onObjectTrackingResult(dtsObject);
            }
        }

        @Override
        public void onObjectTrackingError(int error, String msg) throws RemoteException {
            if (mGeneralTrackingListener != null) {
                mGeneralTrackingListener.onObjectTrackingError(error, msg);
            }
        }
    };

    protected void afterOnServiceConnected() {
        try {
            // add internal AprSense Service Callback
            mAprSenseAIDLService.addCallback(mAprSenseServiceCallback);

            // add FindPersonHandler
            mAprSenseAIDLService.addFindPersonHandler(mFindPersonHandler);
        } catch (RemoteException e) {
            String error = "Cannot add the internal VisionService callback, err = " + e.getMessage();

            // notify error
            notifyRealSenseError(VisionServiceError.APR_SENSE_START_ERROR, error);

            // disconnect to remote service
            mListener.onUnbind(error);
            unbindService();
            mContext = null;

            Logger.e(TAG, error, e);
            return;
        }

        // check version
        try {
            Version serviceVersion = mAprSenseAIDLService.getVersion();
            getVersion().check("VersionInfo", serviceVersion);
        } catch (VersionMismatchException e) {
            String error = "Version mismatch: " + e.getMessage();

            // disconnect to remote service
            mListener.onUnbind(error);
            unbindService();
            mContext = null;
            Logger.e(TAG, error, e);
            return;
        } catch (Exception e) {
            String error = "Cannot get VisionService version, err = " + e.getMessage();

            // disconnect to remote service
            mListener.onUnbind(error);
            unbindService();
            mContext = null;
            Logger.e(TAG, error, e);
            return;
        }

        // register client
        try {
            mAprSenseAIDLService.registerClient(new Binder(), mContext.getPackageName());
        } catch (Exception e) {
            String error = "Register client to service error";
            Logger.e(TAG, error);

            // disconnect to remote service
            mListener.onUnbind(error);
            unbindService();
            mContext = null;

            return;
        }

        mListener.onBind();

        notifyServiceConnected();
        Logger.d(TAG, "AprSenseService connected.");
    }

    private VisionServiceManager() {
    }

    /**
     * Get an instance of VisionServiceManager.
     *
     * @return an instance of VisionServiceManager.
     */
    static synchronized VisionServiceManager getInstance() {
        if (mInstance == null) {
            mInstance = new VisionServiceManager();
        }
        return mInstance;
    }

    /**
     * Add VisionServiceCallback. This callback receives remote service connection state changes.
     *
     * @param visionServiceCallback a callback instance
     */
    void addVisionServiceCallback(VisionServiceCallback visionServiceCallback) {
        if (visionServiceCallback == null) {
            return;
        }

        synchronized (mVisionServiceCallbacks) {
            if (!mVisionServiceCallbacks.contains(visionServiceCallback)) {
                mVisionServiceCallbacks.add(visionServiceCallback);
            }
        }
    }

    /**
     * Remove VisionServiceCallback.
     *
     * @param visionServiceCallback a callback instance.
     */
    void removeVisionServiceCallback(VisionServiceCallback visionServiceCallback) {
        if (visionServiceCallback == null) {
            return;
        }

        synchronized (mVisionServiceCallbacks) {
            mVisionServiceCallbacks.remove(visionServiceCallback);
        }
    }

    /**
     * Add AprSenseStateCallback. This callback receives the AprSense state changes.
     *
     * @param aprSenseStateCallback a callback instance.
     */
    void addRealSenseStateCallback(AprSenseStateCallback aprSenseStateCallback) {
        if (aprSenseStateCallback == null) {
            return;
        }

        synchronized (mAprSenseStateCallback) {
            if (!mAprSenseStateCallback.contains(aprSenseStateCallback)) {
                mAprSenseStateCallback.add(aprSenseStateCallback);
            }
        }
    }

    /**
     * Remove AprSenseStateCallback.
     *
     * @param aprSenseStateCallback a callback instance.
     */
    void removeRealSenseStateCallback(AprSenseStateCallback aprSenseStateCallback) {
        if (aprSenseStateCallback == null) {
            return;
        }

        synchronized (mAprSenseStateCallback) {
            mAprSenseStateCallback.remove(aprSenseStateCallback);
        }
    }

    /**
     * Connect to the vision service.
     *
     * @param context  any Android Context.
     * @param listener the connection state callback.
     * @return true if the connection is successful.
     */
    @Override
    public synchronized boolean bindService(Context context, BindStateListener listener) {
        if (context == null) {
            Logger.e(TAG, "illegalStateException bindService context = null.");
            throw new IllegalArgumentException("Context cannot be null!");
        }

        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Listener cannot be null!");
        }

        if (mContext != null) {
            return true;
        }

        context = context.getApplicationContext();

        // connect service
        Intent startServiceIntent = new Intent();
        startServiceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        startServiceIntent.putExtra("PackageNameFPC", context.getPackageName());
        boolean result = context.bindService(startServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);

        // save param on success
        if (result) {
            mContext = context.getApplicationContext();
            mListener = listener;
            Logger.i(TAG, getVersion().toString());
        }

        return result;
    }

    /**
     * Disconnect from the vision service.
     */
    @Override
    public synchronized void unbindService() {
        if (mContext == null) {
            return;
        }

        synchronized (mFrameTransferMap) {
            for (int i = 0; i < mFrameTransferMap.size(); i++) {
                FrameClientThread frameClientThread = mFrameTransferMap.valueAt(i);
                frameClientThread.disconnect();
            }
            mFrameTransferMap.clear();
        }

        synchronized (mFrameTransferSharedMemoryMap) {
            for (int i = 0; i < mFrameTransferSharedMemoryMap.size(); i++) {
                SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.valueAt(i);
                sharedMemoryHolder.parcelFileDescriptor = null;
                try {
                    sharedMemoryHolder.inputStream.close();
                } catch (IOException e) {
                }
                sharedMemoryHolder.inputStream = null;

            }
            mFrameTransferSharedMemoryMap.clear();
        }

        synchronized (mImageCallbackMap) {
            for (int i = 0; i < mImageCallbackMap.size(); i++) {
                int streamType = mImageCallbackMap.keyAt(i);
                // 2016/12/2 Because image streaming var local socket is abandoned,
                // 2016/12/2 So I just stop the streaming var shared memory
                try {
                    mAprSenseAIDLService.stopImageTransferMemoryFileBuffer(streamType);
                } catch (RemoteException e) {
                    Logger.w(TAG, "try to stop image streaming while unbind but caught exception:", e);
                }
            }
            mImageCallbackMap.clear();
        }

        synchronized (mPreviewMap) {
            for (int i = 0; i < mPreviewMap.size(); i++) {
                int streamType = mPreviewMap.keyAt(i);
                try {
                    mAprSenseAIDLService.stopPreview(streamType);
                } catch (RemoteException e) {
                    Logger.w(TAG, "try to stop image preview while unbind but caught exception:", e);
                }
            }
            mPreviewMap.clear();
        }

        synchronized (mMemoryFileBufferCacheArray) {
            mMemoryFileBufferCacheArray.clear();
        }

        synchronized (mIMUDataCallbacks) {
            mIMUDataCallbacks.clear();
        }

        if (mContext != null) {
            // unregister client
            if (mAprSenseAIDLService != null) {
                try {
                    mAprSenseAIDLService.unregisterClient(mContext.getPackageName());
                } catch (RemoteException e) {
                    Logger.e(TAG, "unregister client error", e);
                }
            } else {
                Logger.e(TAG, "unbindService: vision service may be crashed");
            }
            // unbind service
            mContext.unbindService(mServiceConnection);
            mContext = null;
        }
    }

    @Override
    public boolean isBind() {
        return mContext != null;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mListener;
    }

    /**
     * Return the service connection state.
     *
     * @return service connect state
     */
    synchronized boolean isServiceConnected() {
        return mContext != null;
    }

    /**
     * Start AprSense
     *
     * @param aprSenseStateCallback AprSenseStateCallback which can be null.
     */
    synchronized void startAprSense(AprSenseStateCallback aprSenseStateCallback) {
        checkConnected();

        try {
            if (mAprSenseAIDLService.isAprSenseRunning()) {
                throw new VisionServiceException("AprSense is already started.");
            }
            mAprSenseAIDLService.startAprSense();
            addRealSenseStateCallback(aprSenseStateCallback);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop AprSense.
     */
    synchronized void stopAprSense() {
        checkConnected();

        try {
            if (!mAprSenseAIDLService.isAprSenseRunning()) {
                throw new VisionServiceException("AprSense is not running");
            }
            mAprSenseAIDLService.stopAprSense();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Return the AprSense state.
     *
     * @return true if AprSense is running.
     */
    boolean isAprSenseRunning() {
        checkConnected();

        try {
            return mAprSenseAIDLService.isAprSenseRunning();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Set and enable the stream profile for AprSense. This Function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     * For the color camera and the depth camera, each can set only one stream profile at a time.
     *
     * @param streamType  the stream type, see the StreamType.
     * @param resolution  the image resolution, see the VisionResolution.Color and VisionResolution.Depth.
     * @param fps         the image frame rate, see the FrameRate.
     * @param pixelFormat the pixel format, see the PixelFormat.
     */
    void enableStream(@VisionStreamType int streamType,
                      @Resolution.VisionResolution int resolution,
                      @FrameRate.VisionFrameRate int fps,
                      @PixelFormat.VisionPixelFormat int pixelFormat) throws VisionServiceException {
        checkConnected();

        try {
            mAprSenseAIDLService.enableStream(streamType, resolution, fps, pixelFormat);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }

    }

    /**
     * Set and enable the stream profile for AprSense. This Function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     * For the color camera and the depth camera, each can set only one stream profile at a time.
     *
     * @param streamInfo see the StreamInfo.
     */
    void enableStream(StreamInfo streamInfo) {
        checkConnected();

        int resolution = Resolution.toVisionServiceResolution(streamInfo.getWidth(), streamInfo.getHeight());
        if (resolution == 0) {
            throw new VisionServiceException("The width or height is not supported.");
        }

        try {
            mAprSenseAIDLService.enableStream(streamInfo.getStreamType(), resolution,
                    streamInfo.getFps(), streamInfo.getPixelFormat());
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Disable all the streams for AprSense. This function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     */
    void cleanStreams() throws VisionServiceException {
        checkConnected();

        try {
            mAprSenseAIDLService.cleanStreams();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get the activated stream profile for AprSense.
     *
     * @return StreamInfo array.
     */
    StreamInfo[] getActivatedStreamProfiles() throws VisionServiceException {
        checkConnected();

        try {
            return mAprSenseAIDLService.getActivatedStreamProfile();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start the selected stream preview. The image will be displayed on the input surface view directly.
     * The stream profile must be enabled before AprSense is started.
     *
     * @param surfaceView the SurfaceView to display preview image.
     * @param streamInfo  the StreamInfo to select stream that will be displayed.
     * @param showPerson  If it is set to be true, the person rectangle will be drawn on the preview.
     */
    void startPreview(final SurfaceView surfaceView, final StreamInfo streamInfo, final boolean showPerson) throws VisionServiceException {
        checkConnected();
        checkAprSenseStarted();

        SurfaceHolder surfaceHolder = surfaceView.getHolder();
        surfaceHolder.setFixedSize(streamInfo.getWidth(), streamInfo.getHeight());
        SurfaceHolder.Callback callback = new SurfaceHolder.Callback() {
            @Override
            public void surfaceCreated(SurfaceHolder holder) {
                Logger.i(TAG, "surfaceCreated id = " + surfaceView.getId());
                try {
                    preview(streamInfo.getStreamType(), holder.getSurface(), showPerson);
                } catch (VisionServiceException e) {
                    notifyRealSenseError(VisionServiceError.PREVIEW_ERROR, e.getMessage());
                }
            }

            @Override
            public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
                Logger.i(TAG, "surfaceChanged id = " + surfaceView.getId()
                        + " format = " + format + " width = " + width + " height = " + width);
            }

            @Override
            public void surfaceDestroyed(SurfaceHolder holder) {
                Logger.i(TAG, "surfaceDestroyed id = " + surfaceView.getId());
            }
        };
        surfaceView.setTag(SURFACE_HOLDER_CALLBACK_TAG_KEY, callback);
        surfaceView.setTag(SURFACE_HOLDER_PROFILE_TAG_KEY, streamInfo);

        surfaceHolder.addCallback(callback);
    }

    /**
     * Stop the preview.
     *
     * @param surfaceView the SurfaceView to stop the preview.
     */
    void stopPreview(final SurfaceView surfaceView) {
        checkConnected();
        checkAprSenseStarted();

        SurfaceHolder surfaceHolder = surfaceView.getHolder();
        SurfaceHolder.Callback callback = (SurfaceHolder.Callback) surfaceView.getTag(SURFACE_HOLDER_CALLBACK_TAG_KEY);
        StreamInfo streamInfo = (StreamInfo) surfaceView.getTag(SURFACE_HOLDER_PROFILE_TAG_KEY);
        if (callback != null) {
            surfaceHolder.removeCallback(callback);
        }
        try {
            mAprSenseAIDLService.stopPreview(streamInfo.getStreamType());
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start the selected image in a stream.
     *
     * @param streamInfo the selected stream.
     * @param callback   the image callback.
     */
    void startImageStream(StreamInfo streamInfo, ImageStreamCallback callback) {
        checkConnected();

        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("The stream profile cannot be null.");
        }

        if (callback == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("The image stream callback cannot be null.");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamInfo.getStreamType()) != null) {
                throw new VisionServiceException("The stream is duplicated.");
            }
            mImageCallbackMap.put(streamInfo.getStreamType(), mDummy);
        }

        try {
            String address = mAprSenseAIDLService.setupImageTransferSocket(streamInfo.getStreamType(),
                    ImageTransferType.PUSH);
            ImageReaderThreadManager.getInstance().addConnection(streamInfo, address, callback);
        } catch (Exception e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * stream to sdk buffer
     *
     * @param streamInfo Info of stream
     */
    void startImageStreamToBuffer(final StreamInfo streamInfo) {
        checkConnected();

        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("stream profile can't be null");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamInfo.getStreamType()) != null) {
                throw new VisionServiceException("stream duplicated");
            }
            mImageCallbackMap.put(streamInfo.getStreamType(), mDummy);
        }

        FrameBuffer frameBuffer;
        synchronized (mFrameBufferMap) {
            if (mFrameBufferMap.get(streamInfo.getStreamType()) == null) {
                mFrameBufferMap.put(streamInfo.getStreamType(), new FrameBuffer());
                Logger.d(TAG, "startImageStreamToBuffer() mFrameBufferMap.put type : " + streamInfo.getStreamType());
            }

            frameBuffer = mFrameBufferMap.get(streamInfo.getStreamType());
        }

        try {
            String address = mAprSenseAIDLService.setupImageTransferSocket(streamInfo.getStreamType(),
                    ImageTransferType.PUSH);
            ImageReaderThreadManager.getInstance().addConnection(streamInfo, address, frameBuffer);
        } catch (Exception e) {
            throw new VisionServiceException(e);
        }
    }

    void startImageTransferMemoryFileBuffer(final StreamInfo streamInfo, final ImageStreamCallback callback) {
        checkConnected();

        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("The stream profile cannot be null.");
        }

        if (callback == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("The image stream callback cannot be null.");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamInfo.getStreamType()) != null) {
                throw new VisionServiceException("The stream is duplicated.");
            }
            mImageCallbackMap.put(streamInfo.getStreamType(), mDummy);
        }

        try {
            mAprSenseAIDLService.startImageTransferMemoryFileBuffer(streamInfo.getStreamType(), new MemoryFileBufferCallback.Stub() {
                long prevTimeStamp;

                @Override
                public void onNewImage(MemoryFileBuffer memoryFileBuffer) throws RemoteException {
                    try {
                        ByteBuffer imageBuffer = getMappedBufferFromMemoryFile(
                                streamInfo.getStreamType(),
                                memoryFileBuffer.getIndex(),
                                0,
                                memoryFileBuffer.getImageFD().getFileDescriptor(),
                                memoryFileBuffer.getImageSize());
                        ByteBuffer infoBuffer = getMappedBufferFromMemoryFile(
                                streamInfo.getStreamType(),
                                memoryFileBuffer.getIndex(),
                                1,
                                memoryFileBuffer.getInfoFD().getFileDescriptor(),
                                memoryFileBuffer.getInfoSize());
                        if (imageBuffer == null || infoBuffer == null) {
                            Logger.e(TAG, "get mapped buffer error");
                            releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }
                        FrameInfo frameInfo = FrameInfo.fromByteBuffer(infoBuffer);

                        long ts = frameInfo.getIMUTimeStamp();
                        if (frameInfo.getStreamType() != streamInfo.getStreamType()
                                || frameInfo.getPixelFormat() != streamInfo.getPixelFormat()) {
                            Logger.e(TAG, "_error frame detected! frame info = " + frameInfo.toString());
                            synchronized (mMemoryFileBufferCacheArray) {
                                mMemoryFileBufferCacheArray.clear();
                            }
                            releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }
                        if (ts > 0 && ts <= prevTimeStamp) {
                            Logger.w(TAG, "_get frame ts(" + ts + ") low than previous(" + prevTimeStamp + "), clean share mem cache and drop frame");
                            synchronized (mMemoryFileBufferCacheArray) {
                                mMemoryFileBufferCacheArray.clear();
                            }
                            releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }

                        prevTimeStamp = ts;

                        callback.onNewImage(frameInfo, imageBuffer);
                        // TODO: 2016/11/7 auto close feature
                        releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                    } finally {
                        try {
                            memoryFileBuffer.getImageFD().close();
                        } catch (IOException ignored) {
                        }
                        try {
                            memoryFileBuffer.getInfoFD().close();
                        } catch (IOException ignored) {
                        }
                    }
                }
            });
        } catch (RemoteException e) {
            Logger.e(TAG, "startImageTransferMemoryFileBuffer error", e);
            throw new VisionServiceException(e);
        }
    }

    /**
     * stream to sdk buffer using memory file
     *
     * @param streamInfo Info of stream
     */
    void startImageStreamToBufferByMemoryFile(final StreamInfo streamInfo) {
        checkConnected();

        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("stream profile can't be null");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamInfo.getStreamType()) != null) {
                throw new VisionServiceException("stream duplicated");
            }
            mImageCallbackMap.put(streamInfo.getStreamType(), mDummy);
        }

        FrameBuffer frameBuffer;
        synchronized (mFrameBufferMap) {
            if (mFrameBufferMap.get(streamInfo.getStreamType()) == null) {
                mFrameBufferMap.put(streamInfo.getStreamType(), new FrameBuffer());
                Logger.d(TAG, "startImageStreamToBufferByMemoryFilem() FrameBufferMap.put type: " + streamInfo.getStreamType());
            }

            frameBuffer = mFrameBufferMap.get(streamInfo.getStreamType());
        }

        final FrameBuffer finalFrameBuffer = frameBuffer;

        final RecyclableFrame2.FrameReleaseHandler frameReleaseHandler =
                new RecyclableFrame2.FrameReleaseHandler() {
                    @Override
                    public void releaseFrame(int streamType, int index) {
                        IAprSenseService stub = mAprSenseAIDLService;
                        if (stub == null) {
                            Log.w(TAG, "mAprSenseAIDLService is null while releaseFrame service may be disconnected");
                            return;
                        }
                        try {
                            stub.releaseMemoryFileBuffer(streamType, index);
                        } catch (RemoteException e) {
                            Logger.e(TAG, "Release memory file error", e);
                        }
                    }
                };

        try {
            synchronized (mMemoryFileBufferCacheArray) {
                mMemoryFileBufferCacheArray.clear();
            }

            mAprSenseAIDLService.startImageTransferMemoryFileBuffer(streamInfo.getStreamType(), new MemoryFileBufferCallback.Stub() {
                long prevTimeStamp;

                @Override
                public void onNewImage(MemoryFileBuffer memoryFileBuffer) throws RemoteException {
                    try {
                        ByteBuffer imageBuffer = getMappedBufferFromMemoryFile(
                                streamInfo.getStreamType(),
                                memoryFileBuffer.getIndex(),
                                0,
                                memoryFileBuffer.getImageFD().getFileDescriptor(),
                                memoryFileBuffer.getImageSize());
                        ByteBuffer infoBuffer = getMappedBufferFromMemoryFile(
                                streamInfo.getStreamType(),
                                memoryFileBuffer.getIndex(),
                                1,
                                memoryFileBuffer.getInfoFD().getFileDescriptor(),
                                memoryFileBuffer.getInfoSize());
                        if (imageBuffer == null || infoBuffer == null) {
                            Logger.e(TAG, "get mapped buffer error");
                            releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }
                        FrameInfo frameInfo = FrameInfo.fromByteBuffer(infoBuffer);
                        if (frameInfo.getStreamType() != streamInfo.getStreamType()) {
                            Logger.w(TAG, "Illegal frame detected, " +
                                    " stream type = " + frameInfo.getStreamType() +
                                    " index = " + memoryFileBuffer.getIndex() +
                                    " clean share mem cache and drop frame");
                            synchronized (mMemoryFileBufferCacheArray) {
                                mMemoryFileBufferCacheArray.clear();
                            }
                            releaseMemoryFileBuffer(frameInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }

                        long ts = frameInfo.getIMUTimeStamp();
                        if (frameInfo.getStreamType() != streamInfo.getStreamType()
                                || frameInfo.getPixelFormat() != streamInfo.getPixelFormat()) {
                            Logger.e(TAG, "__error frame detected! frame info = " + frameInfo.toString());
                            synchronized (mMemoryFileBufferCacheArray) {
                                mMemoryFileBufferCacheArray.clear();
                            }
                            releaseMemoryFileBuffer(streamInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }
                        if (ts > 0 && ts <= prevTimeStamp) {
                            Logger.w(TAG, "__get frame ts(" + ts + ") low than previous(" + prevTimeStamp + "), clean share mem cache and drop frame");
                            synchronized (mMemoryFileBufferCacheArray) {
                                mMemoryFileBufferCacheArray.clear();
                            }
                            releaseMemoryFileBuffer(frameInfo.getStreamType(), memoryFileBuffer.getIndex());
                            return;
                        }

                        prevTimeStamp = ts;

                        RecyclableFrame2 recyclableFrame2 = RecyclableFrame2.create(streamInfo.getStreamType(),
                                memoryFileBuffer.getIndex(), imageBuffer.capacity(), frameInfo, imageBuffer, frameReleaseHandler);
                        finalFrameBuffer.add(recyclableFrame2);
                    } finally {
                        try {
                            memoryFileBuffer.getImageFD().close();
                        } catch (IOException ignored) {
                        }
                        try {
                            memoryFileBuffer.getInfoFD().close();
                        } catch (IOException ignored) {
                        }
                    }
                }
            });
        } catch (RemoteException e) {
            Logger.e(TAG, "startImageTransferMemoryFileBuffer error", e);
            throw new VisionServiceException(e);
        }
    }

    void stopImageTransferMemoryFileBuffer(int streamType) {
        checkConnected();

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamType) == null) {
                Logger.w(TAG, "stopImageTransferMemoryFileBuffer: stream " + streamType + " is not start transfer.");
                return;
            }
            mImageCallbackMap.remove(streamType);
        }

        try {
            mAprSenseAIDLService.stopImageTransferMemoryFileBuffer(streamType);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }

        synchronized (mMemoryFileBufferCacheArray) {
            mMemoryFileBufferCacheArray.clear();
        }

        synchronized (mFrameBufferMap) {
            FrameBuffer frameBuffer = mFrameBufferMap.get(streamType);
            if (frameBuffer != null) {
                frameBuffer.release();
                mFrameBufferMap.remove(streamType);
            }
        }
    }

    void releaseMemoryFileBuffer(int streamType, int index) {
        checkConnected();
        try {
            mAprSenseAIDLService.releaseMemoryFileBuffer(streamType, index);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    private ByteBuffer getMappedBufferFromMemoryFile(int type, int index, int info, FileDescriptor fileDescriptor, int size) {
        ByteBuffer mappedByteBuffer;
        int key = type << 8 | index << 1 | info;
        synchronized (mMemoryFileBufferCacheArray) {
            mappedByteBuffer = mMemoryFileBufferCacheArray.get(key);
        }
        if (mappedByteBuffer != null) {
            mappedByteBuffer.rewind();
            return mappedByteBuffer;
        }

        FileInputStream fileInputStream = null;
        try {
            fileInputStream = new FileInputStream(fileDescriptor);
            mappedByteBuffer = fileInputStream.getChannel().map(FileChannel.MapMode.READ_ONLY, 0, size);
            synchronized (mMemoryFileBufferCacheArray) {
                mMemoryFileBufferCacheArray.put(key, mappedByteBuffer);
            }
        } catch (IOException e) {
            Logger.e(TAG, "map buffer from memory file error", e);
            return null;
        } finally {
            if (fileInputStream != null) {
                try {
                    fileInputStream.close();
                } catch (IOException ignored) {
                }
            }
        }

        mappedByteBuffer.rewind();
        return mappedByteBuffer;
    }

    /**
     * Stop the image stream.
     *
     * @param streamInfo the selected stream.
     */
    @SuppressLint("WrongConstant")
    void stopImageStream(StreamInfo streamInfo) {
        checkConnected();

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamInfo.getStreamType()) == null) {
                throw new VisionServiceException("The stream is not started " + streamInfo.getStreamType());
            }
            mImageCallbackMap.remove(streamInfo.getStreamType());
        }

        synchronized (mFrameBufferMap) {
            FrameBuffer frameBuffer = mFrameBufferMap.get(streamInfo.getStreamType());
            if (frameBuffer != null) {
                frameBuffer.release();
                mFrameBufferMap.remove(streamInfo.getStreamType());
            }
        }

        try {
            ImageReaderThreadManager.getInstance().removeConnection(streamInfo);
            mAprSenseAIDLService.stopImageTransfer(streamInfo.getStreamType(),
                    ImageTransferType.PUSH);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get latest frame with tid larger than previousTid
     *
     * @param streamInfo  stream profile
     * @param previousTid tid of the previous frame
     * @return Frame object for success or null for no new frame
     */
    Frame getLatestFrameForStream(StreamInfo streamInfo, long previousTid) {
        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException getLatestFrameForStream streamInfo = null.");
            throw new IllegalArgumentException("stream profile can't be null");
        }

        return getLatestFrameForStream(streamInfo.getStreamType(), previousTid);
    }

    Frame getLatestFrameForStream(@VisionStreamType int streamType, long previousTid) {
        FrameBuffer frameBuffer;

        synchronized (mFrameBufferMap) {
            frameBuffer = mFrameBufferMap.get(streamType);
        }

        if (frameBuffer == null) {
            Logger.e(TAG, "getLatersFrameForStream frameBuffer = null. ");
            throw new IllegalArgumentException("stream image transfer not initialized.");
        }

        return frameBuffer.getLatest(previousTid);
    }

    /**
     * Return frame got from getLatestFrameForStream to buffer
     * This is very important for the buffer queue is limited
     *
     * @param streamInfo stream profile
     * @param frame      frame to return
     */
    void returnFrameToStream(StreamInfo streamInfo, Frame frame) {
        if (streamInfo == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("stream profile can't be null");
        }

        if (frame == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("can not return null frame");
        }

        returnFrameToStream(streamInfo.getStreamType(), frame);
    }

    void returnFrameToStream(int streamType, Frame frame) {
        FrameBuffer frameBuffer;
        if (frame == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("can not return null frame");
        }

        if (frame.getInfo().getStreamType() != streamType) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("try to return frame type = " + frame.getInfo().getStreamType() + " to buffer type = " + streamType);
        }

        synchronized (mFrameBufferMap) {
            frameBuffer = mFrameBufferMap.get(streamType);
        }

        if (frameBuffer == null) {
            Logger.w(TAG, "return image for stream " + streamType + " but not initialized");
            ((CountableFrame) frame).lockCountDown();
            return;
        }

        frameBuffer.returnFrame(frame);
    }

    /**
     * Enable the frame transfer by StreamInfo.
     *
     * @param profile the selected stream.
     */
    void enableFrameTransfer(StreamInfo profile) {
        checkConnected();
        synchronized (mFrameTransferMap) {
            if (mFrameTransferMap.get(profile.getStreamType()) != null) {
                throw new VisionServiceException("The frame profile is duplicated.");
            }
        }

        String address;
        try {
            address = mAprSenseAIDLService.setupImageTransferSocket(profile.getStreamType(), ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException("An error occurs in seting up the image transfer socket.", e);
        }

        FrameClientThread frameClientThread = new FrameClientThread(profile);
        frameClientThread.connectServer(address);
        // TODO: connect failed
        synchronized (mFrameTransferMap) {
            mFrameTransferMap.put(profile.getStreamType(), frameClientThread);
        }
    }

    /**
     * Disable the frame transfer by StreamInfo.
     *
     * @param profile the selected stream.
     */
    void disableFrameTransfer(StreamInfo profile) {
        checkConnected();
        synchronized (mFrameTransferMap) {
            FrameClientThread frameClientThread = mFrameTransferMap.get(profile.getStreamType());
            if (frameClientThread != null) {
                frameClientThread.disconnect();
                mFrameTransferMap.remove(profile.getStreamType());
            }
        }

        try {
            mAprSenseAIDLService.stopImageTransfer(profile.getStreamType(),
                    ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Enable frame transfer by StreamInfo
     *
     * @param profile Stream selected
     */
    void enableFrameTransferMemoryFile(StreamInfo profile) {
        checkConnected();
        synchronized (mFrameTransferSharedMemoryMap) {
            if (mFrameTransferSharedMemoryMap.get(profile.getStreamType()) != null) {
                throw new VisionServiceException("frame profile duplicated");
            }
        }

        try {
            ParcelFileDescriptor parcelFileDescriptor =
                    mAprSenseAIDLService.setupImageTransferSharedMemory(
                            profile.getStreamType(), ImageTransferType.PULL);
            FileInputStream fileInputStream = new FileInputStream(parcelFileDescriptor.getFileDescriptor());
            SharedMemoryHolder sharedMemoryHolder = new SharedMemoryHolder();
            sharedMemoryHolder.parcelFileDescriptor = parcelFileDescriptor;
            sharedMemoryHolder.inputStream = fileInputStream;
            synchronized (mFrameTransferSharedMemoryMap) {
                mFrameTransferSharedMemoryMap.put(profile.getStreamType(), sharedMemoryHolder);
            }
        } catch (RemoteException e) {
            throw new VisionServiceException("setup image transfer socket error", e);
        }
    }

    /**
     * Disable frame transfer by StreamInfo
     *
     * @param profile Stream selected
     */
    void disableFrameTransferMemoryFile(StreamInfo profile) {
        checkConnected();
        synchronized (mFrameTransferSharedMemoryMap) {
            SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.get(profile.getStreamType());
            if (sharedMemoryHolder != null) {
                sharedMemoryHolder.parcelFileDescriptor = null;
                try {
                    sharedMemoryHolder.inputStream.close();
                } catch (IOException e) {
                }
                sharedMemoryHolder.inputStream = null;
                mFrameTransferSharedMemoryMap.remove(profile.getStreamType());
            }
        }

        // TODO: 16/6/23
        try {
            mAprSenseAIDLService.stopImageTransfer(profile.getStreamType(),
                    ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get the fish eye frame.
     *
     * @param streamInfo the selected stream which must be enabled by enableFrameTransfer.
     * @param buffer     the image data to fill in the buffer.
     * @return the frame information.
     */
    FishEyeFrameInfo getFishEyeFrame(StreamInfo streamInfo, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            Logger.e(TAG, "illegalStateException buffer = null.");
            throw new IllegalArgumentException("The buffer is null.");
        }

        FrameInfo frameInfo = getFrame(streamInfo, buffer);
        if (frameInfo != null) {
            return new FishEyeFrameInfo(frameInfo);
        }

        return null;
    }

    /**
     * Get the depth frame.
     *
     * @param streamInfo the selected frame which must be enabled by enableFrameTransfer.
     * @param buffer     the image data to fill in the buffer.
     * @return the frame information.
     */
    DepthFrameInfo getDepthFrame(StreamInfo streamInfo, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            Logger.e(TAG, "illegalStateException buffer = null");
            throw new IllegalArgumentException("The buffer is null.");
        }

        FrameInfo frameInfo = getFrame(streamInfo, buffer);
        if (frameInfo != null) {
            return new DepthFrameInfo(frameInfo);
        }

        return null;
    }

    /**
     * Get the color frame which is temporary solution.
     *
     * @param streamInfo the selected frame which must be enabled by enableFrameTransfer.
     * @param buffer     the image data to fill in the buffer.
     * @return the frame information.
     */
    @Deprecated
    FrameInfo getColorFrame(StreamInfo streamInfo, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            Logger.e(TAG, "illegalStateException buffer = null.");
            throw new IllegalArgumentException("The buffer is null.");
        }

        return getFrame(streamInfo, buffer);
    }

    FrameInfo getFrameSharedMemory(StreamInfo profile, ByteBuffer buffer) {
        SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.get(profile.getStreamType());
        if (sharedMemoryHolder == null) {
            throw new VisionServiceException("stream " + profile.getStreamType() + " not enabled");
        }

        try {
            FrameInfo frameInfo = mAprSenseAIDLService.getFrameMemoryFile(profile.getStreamType(),
                    profile.getResolution(), profile.getPixelFormat());

            if (frameInfo == null) {
                return null;
            }

            try {
                buffer.rewind();
                FileInputStream inputStream = sharedMemoryHolder.inputStream;
                int cnt = inputStream.getChannel().position(0).read(buffer);
            } catch (IOException e) {
                throw new VisionServiceException(e);
            }

            return frameInfo;
        } catch (RemoteException e) {
            // TODO: dell with connect failed
            throw new VisionServiceException("get Frame failed", e);
        }
    }

    private FrameInfo getFrame(StreamInfo profile, ByteBuffer buffer) {
        FrameInfo frameInfo;
        FrameClientThread frameClientThread;
        synchronized (mFrameTransferMap) {
            frameClientThread = mFrameTransferMap.get(profile.getStreamType());
        }

        if (frameClientThread == null) {
            throw new VisionServiceException("frame transfer for stream " + profile.getStreamType()
                    + " not started");
        }

        try {
            frameInfo = mAprSenseAIDLService.getFrame(profile.getStreamType(),
                    profile.getResolution(), profile.getPixelFormat());
        } catch (RemoteException e) {
            // TODO: dell with connect failed
            throw new VisionServiceException("get Frame failed", e);
        }

        if (frameInfo == null) {
            return null;
        }

        int length = (int) (PixelFormat.getPixelBytes(profile.getPixelFormat())
                * profile.getHeight() * profile.getWidth());
        frameClientThread.readFrame(buffer, length);

        // wait for transfer complete
        synchronized (frameClientThread) {
            while (!frameClientThread.isTransferEnded()) {
                try {
                    frameClientThread.wait();
                } catch (InterruptedException e) {
                }
            }
        }

        if (!frameClientThread.isReadSuccess()) {
            Logger.e(TAG, "read frame not success, check log for details");
            throw new VisionServiceException("read frame failed");
        }
        return frameInfo;
    }

    /**
     * Start person detection.
     *
     * @param callback the detected person callback which can be null.
     * @deprecated use new {@link #startDetectingPerson(PersonDetectListener)} and
     * new {@link #stopDetectingPerson()} to replace it.
     */
    @Deprecated
    void startPersonDetection(PersonDetectCallback callback) {
        checkConnected();

        try {
            mAprSenseAIDLService.startPersonDetection();
            if (callback != null) {
                mPersonDetectCallback.add(callback);
            }
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop person detection and clean all the callbacks.
     *
     * @deprecated use new {@link #startDetectingPerson(PersonDetectListener)} and
     * new {@link #stopDetectingPerson()} to replace it.
     */
    @Deprecated
    void stopPersonDetection() {
        checkConnected();
        checkAprSenseStarted();

        mPersonDetectCallback.clear();
        try {
            mAprSenseAIDLService.stopPersonDetection();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start detecting person.
     *
     * @param listener the detected person listener {@link PersonDetectListener} which can't be
     *                 null.
     */
    void startDetectingPerson(PersonDetectListener listener) {
        startDetectingPerson(null, listener);
    }

    /**
     * Start detecting person with profile. Not available now.
     *
     * @param profile  the profile of person detection
     * @param listener the detected person listener {@link PersonDetectListener} which can't be null.
     */
    void startDetectingPerson(PersonDetectionProfile profile, PersonDetectListener listener) {
        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("PersonDetectListener can't be null!");
        }

        checkDtsStarted();
        try {
            mPersonDetectListener = listener;
            mAprSenseAIDLService.startDetectingPersonWithProfile(profile, mIPersonDetectionListener);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop detecting person.
     * If stop detecting person, any callback from {@link PersonDetectListener} will be stopped.
     */
    void stopDetectingPerson() {
        checkAprSenseStarted();
        checkDtsStarted();
        mPersonDetectListener = null;
        try {
            mAprSenseAIDLService.stopDetectingPerson();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start tracking on a selected person.
     *
     * @param personId the person ID which is got from the person detection callback.
     * @deprecated use new {@link #startPersonTracking(DTSPerson, long, PersonTrackingListener)} and
     * {@link #stopPersonTracking()} to replace it.
     */
    @Deprecated
    void startPersonTracking(int personId) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.startPersonTracking(personId);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop tracking person.
     *
     * @param personId the person ID which is got from the person detection callback.
     * @deprecated use new {@link #startPersonTracking(DTSPerson, long, PersonTrackingListener)} and
     * {@link #stopPersonTracking()} to replace it.
     */
    @Deprecated
    void stopPersonTracking(int personId) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.stopPersonTracking(personId);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Return person find now.
     *
     * @return the person list to de found
     * @deprecated use new {@link #detectPersons(long)} method to replace it.
     */
    @Deprecated
    List<Person> findPersons() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.findPersons();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Load the person database.
     *
     * @param databasePath the person database path.
     * @deprecated this is not needed again.
     */
    // TODO: not tested
    void loadPersonDatabase(String databasePath) {
        checkConnected();

        try {
            mAprSenseAIDLService.loadRecognitionDatabase(databasePath);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Save the person database.
     *
     * @param databasePath the person database path.
     * @deprecated this is not needed again.
     */
    // TODO: not tested
    void savePersonDatabase(String databasePath) {
        checkConnected();

        try {
            mAprSenseAIDLService.saveRecognitionDatabase(databasePath);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get the AprSense timestamp.
     *
     * @return the 64bit timestamp in ms.
     */
    long getAprSenseTimeStamp() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getAprSenseTimeStamp();
        } catch (RemoteException e) {
            throw new VisionServiceException("An exception occurs in getting the AprSense timestamp! ", e);
        }
    }

    /**
     * Get Depth Focal Length X
     *
     * @return float parameter
     */
    float getDepthFocalLengthX() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_FOCAL_LENGTH_X);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Get Depth Focal Length Y
     *
     * @return float parameter
     */
    float getDepthFocalLengthY() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_FOCAL_LENGTH_Y);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }


    /**
     * Get Depth Principal Point X
     *
     * @return float parameter
     */
    float getDepthPrincipalPointX() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_PRINCIPAL_POINT_X);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Get Depth Principal Point Y
     *
     * @return float parameter
     */
    float getDepthPrincipalPointY() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_PRINCIPAL_POINT_Y);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Add the IMU data callback.
     *
     * @param callback
     */
    void addIMUDataCallback(IMUDataCallback callback) {
        checkConnected();

        synchronized (mIMUDataCallbacks) {
            for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                if (bundle.callback == callback) {
                    return;
                }
            }

            IMUCallbackBundle bundle = new IMUCallbackBundle();
            bundle.callback = callback;
            // TODO: fixed size
            bundle.buffer = ByteBuffer.allocateDirect(IMU_DATA_SIZE);

            mIMUDataCallbacks.add(bundle);
            if (mIMUDataCallbacks.size() == 1) {
                try {
                    mAprSenseAIDLService.setIMUDataCallback(mIImuDataCallback);
                } catch (RemoteException e) {
                    throw new VisionServiceException("add IMU Data Callback exception! ", e);
                }
            }
        }
    }

    /**
     * Remove the IMU data callback.
     *
     * @param callback
     */
    void removeIMUDataCallback(IMUDataCallback callback) {
        checkConnected();

        synchronized (mIMUDataCallbacks) {
            IMUCallbackBundle removeBundle = null;
            for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                mIMUDataCallbacks.remove(callback);
                if (bundle.callback == callback) {
                    removeBundle = bundle;
                    break;
                }
            }

            if (removeBundle != null) {
                // TODO: release byte buffer
                mIMUDataCallbacks.remove(removeBundle);
            }

            if (mIMUDataCallbacks.size() == 0) {
                try {
                    mAprSenseAIDLService.removeIMUDataCallback();
                } catch (RemoteException e) {
                    throw new VisionServiceException("An exception occurs in removing the IMU Data callback! ", e);
                }
            }
        }
    }

    byte[] getCalibrationData() {
        checkConnected();

        try {
            return mAprSenseAIDLService.getCalibrationData();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get calibration data failed", e);
        }
    }

    ColorDepthCalibration getColorDepthCalibrationData() {
        checkConnected();

        try {
            return mAprSenseAIDLService.getColorDepthCalibrationData();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get calibration data failed", e);
        }
    }

    MotionModuleCalibration getMotionModuleCalibrationData() {
        checkConnected();

        try {
            return mAprSenseAIDLService.getMotionModuleCalibrationData();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get calibration data failed", e);
        }
    }

    boolean isFishEyeAutoExposureEnabled() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.isFishEyeAutoExposureEnabled();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get fish eye camera auto exposure enable state error", e);
        }
    }

    void enableFishEyeAutoExposure(boolean enable) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.enableFishEyeAutoExposure(enable);
        } catch (RemoteException e) {
            throw new VisionServiceException("Set fish eye camera auto exposure enable/disable state error", e);
        }
    }

    float getFishEyeEvCompensation() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getFishEyeEvCompensation();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get fish eye camera EV compensation error", e);
        }
    }

    void setFishEyeEvCompensation(float compensation) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.setFishEyeEvCompensation(compensation);
        } catch (RemoteException e) {
            throw new VisionServiceException("Set fish eye camera EV compensation error", e);
        }
    }

    float getFishEyeManualExposure() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getFishEyeManualExposure();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get fish eye camera manual exposure error", e);
        }
    }

    void setFishEyeManualExposure(float exposure) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.setFishEyeManualExposure(exposure);
        } catch (RemoteException e) {
            throw new VisionServiceException("Set fish eye camera manual exposure error", e);
        }
    }

    float getFishEyeManualGain() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getFishEyeManualGain();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get fish eye camera manual gain error", e);
        }
    }

    void setFishEyeManualGain(float gain) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.setFishEyeManualGain(gain);
        } catch (RemoteException e) {
            throw new VisionServiceException("Set fish eye camera manual gain error", e);
        }
    }

    void setFishEyeAutoExposureROI(Rect rect) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.setFishEyeAutoExposureROI(rect);
        } catch (RemoteException e) {
            throw new VisionServiceException("Set fish eye camera auto exposure ROI error", e);
        }
    }

    long getNanoDiffFromEpochToMotion() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getNanoDiffFromEpochToMotion();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get nano diff from micro epoch to motion timestamp error", e);
        }
    }

    private void checkConnected() throws VisionServiceException {
        if (mAprSenseAIDLService == null) {
            throw VisionServiceException.getServiceNotConnectedException();
        }
    }

    private void checkAprSenseStarted() throws VisionServiceException {
        try {
            boolean started = mAprSenseAIDLService.isAprSenseRunning();
            if (!started) {
                throw VisionServiceException.getAprSenseNotStaredException();
            }
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }

    }

    private void checkDtsStarted() {
        checkConnected();
        if (!mDTSEnabled.get()) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalStateException(DTSError.DTS_NOT_STARTED);
        }
    }

    protected void preview(int streamType, Surface surface, boolean drawPerson) throws VisionServiceException {
        checkConnected();

        synchronized (mPreviewMap) {
            if (mPreviewMap.get(streamType) != null) {
                throw new VisionServiceException("Preview stream is duplicated.");
            }
            mPreviewMap.put(streamType, mDummy);
        }

        try {
            mAprSenseAIDLService.startPreview(streamType, surface, drawPerson);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void stopPreview(int streamType) {
        checkConnected();

        synchronized (mPreviewMap) {
            if (mPreviewMap.get(streamType) == null) {
                Logger.w(TAG, "stopPreview: stream " + streamType + " did not start preview");
                return;
            }
            mPreviewMap.remove(streamType);
        }

        try {
            mAprSenseAIDLService.stopPreview(streamType);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void enablePersonTrackingAndDetection(Surface mainCameraPreview) {
        checkConnected();
        if (mDTSEnabled.get()) {
            return;
        }

        try {
            mAprSenseAIDLService.enablePersonTrackingAndDetection(mainCameraPreview);
            mDTSEnabled.set(true);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void enablePersonTrackingAndDetection(Surface mainCameraPreview, int videoSource) {
        checkConnected();
        if (mDTSEnabled.get()) {
            return;
        }

        try {
            mAprSenseAIDLService.enablePersonTrackingAndDetection2(mainCameraPreview, videoSource);
            mDTSEnabled.set(true);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void disablePersonTrackingAndDetection() {
        checkConnected();
        if (!mDTSEnabled.get()) {
            return;
        }

        mDTSEnabled.set(false);
        try {
            mAprSenseAIDLService.disablePersonTrackingAndDetection();
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    DTSPerson[] detectPersons(long timeoutMicroSeconds) {
        if (timeoutMicroSeconds <= 0) {
            Logger.e(TAG, "illegalStateException timeoutMicroSeconds <= 0");
            throw new IllegalArgumentException("Timeout is illegal!");
        }
        checkDtsStarted();

        try {
            return mAprSenseAIDLService.detectPersons(timeoutMicroSeconds);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void startPersonTracking(DTSPerson person, PersonTrackingProfile profile, long timeout,
                             PersonTrackingListener listener) {
        if (timeout <= 0) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Timeout is illegal!");
        }

        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("PersonTrackingListener can't be null!");
        }

        checkDtsStarted();

        if (person == null) {
            // if input person is null, try to track the person in the center of the image
            person = new DTSPerson(0, 0, 0, null, 0, 0, 0, new Rect(0, 0, 640, 480), null, 0, 0, 0, 0);
        }

        mPersonTrackingListener = listener;

        try {
            mAprSenseAIDLService.startPersonTrackingWithProfile(person, profile, timeout, mIPersonTrackingListener);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void stopPersonTracking() {
        checkDtsStarted();

        try {
            mAprSenseAIDLService.stopPersonTracking2();
            mPersonTrackingListener = null;
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void startPlannerPersonTracking(DTSPerson person, PersonTrackingProfile profile, long timeout,
                                    PersonTrackingWithPlannerListener listener) {
        if (timeout <= 0) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Timeout is illegal!");
        }

        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("PersonTrackingListener can't be null!");
        }

        checkDtsStarted();

        if (person == null) {
            // if input person is null, try to track the person in the center of the image
            person = new DTSPerson(0, 0, 0, null, 0, 0, 0, new Rect(0, 0, 640, 480), null, 0, 0, 0, 0);
        }
        mPlannerPersonTrackingListener = listener;
        try {
            mAprSenseAIDLService.startPlannerPersonTracking2(person, profile, timeout, mIPlannerPersonTackingListener);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void stopPlannerPersonTracking() {
        checkDtsStarted();
        try {
            mPlannerPersonTrackingListener = null;
            mAprSenseAIDLService.stopPlannerPersonTracking();
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    /**
     * todo
     *
     * @param person
     * @param profile
     * @param timeout
     * @param listener
     */
    void startTrajectoryPersonTracking(DTSPerson person, PersonTrackingProfile profile, long timeout, PersonTrackingWithTrajectoryListener listener) {
        if (timeout <= 0) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Timeout is illegal!");
        }

        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("PersonTrackingListener can't be null!");
        }

        checkDtsStarted();

        if (person == null) {
            // if input person is null, try to track the person in the center of the image
            person = new DTSPerson(0, 0, 0, null, 0, 0, 0, new Rect(0, 0, 640, 480), null, 0, 0, 0, 0);
        }
        mPersonTrackingWithTrajectoryListener = listener;
        try {
            mAprSenseAIDLService.startPersonTrackingWithTrajectory(person, profile, timeout, mIPersonTrackingWithTrajectoryListener);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    /**
     * todo
     */
    void stopTrajectoryPersonTracking() {
        checkDtsStarted();
        try {
            mPersonTrackingWithTrajectoryListener = null;
            mAprSenseAIDLService.stopPersonTrackingWithTrajectory();
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void setPlannerTrackingConfig(PersonTrackingProfile profile) {
        if (profile == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Profile can't be null!");
        }
        checkDtsStarted();
        try {
            mAprSenseAIDLService.setPlannerTrackingConfig(profile);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void startGeneralTracking(DTSObject object, GeneralTrackingProfile profile, long timeout, GeneralTrackingListener listener) {
        if (timeout <= 0) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("Timeout is illegal!");
        }

        if (listener == null) {
            Logger.e(TAG, "illegalStateException");
            throw new IllegalArgumentException("GeneralTrackingListener can't be null!");
        }

        checkDtsStarted();

        if (object == null) {
            // if input person is null, try to track the person in the center of the image
            object = new DTSObject(0, 0, 0, null, 0, 0, 0, new Rect(0, 0, 640, 480), null, 0, 0, 0);
        }
        mGeneralTrackingListener = listener;
        try {
            mAprSenseAIDLService.startGeneralTracking(object, profile, timeout, mIGeneralTrackingListener);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    void stopGeneralTracking() {
        checkDtsStarted();
        try {
            mGeneralTrackingListener = null;
            mAprSenseAIDLService.stopGeneralTracking();
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    float getHWTemperature() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getHWTemperature();
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    IAprSenseServiceCallback.Stub mAprSenseServiceCallback = new IAprSenseServiceCallback.Stub() {
        @Override
        public void onAprStateChanged(int state, int errorCode, String errorMessage) throws RemoteException {
            switch (state) {
                case VisionServiceState.APR_SENSE_STARTED:
                    notifyAprSenseStarted();
                    break;
                case VisionServiceState.APR_SENSE_STOPPED:
                    notifyRealSenseStopped();
                    break;
                case VisionServiceState.APR_SENSE_ERROR:
                    notifyRealSenseError(errorCode, errorMessage);
                    break;
            }
        }
    };

    FindPersonsHandler.Stub mFindPersonHandler = new FindPersonsHandler.Stub() {
        @Override
        public void onFindPersons(List<Person> persons) throws RemoteException {
            for (PersonDetectCallback personDetectCallback : mPersonDetectCallback) {
                personDetectCallback.onPersonDetected(persons);
            }
        }
    };

    IIMUDataCallback.Stub mIImuDataCallback = new IIMUDataCallback.Stub() {
        @Override
        public void onNewData(byte[] buff, int length, int frameCount) throws RemoteException {
            synchronized (mIMUDataCallbacks) {
                for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(buff);
                    ReadableByteChannel readableByteChannel = Channels.newChannel(byteArrayInputStream);
                    bundle.buffer.rewind();
                    try {
                        readableByteChannel.read(bundle.buffer);
                    } catch (IOException e) {
                        // TODO:
                        Logger.e(TAG, "An error occurs in copying the IMU data.", e);
                    }
                    bundle.buffer.rewind();
                    bundle.callback.onNewData(bundle.buffer, length, frameCount);
                }
            }
        }
    };

    private void notifyServiceConnected() {
        synchronized (mVisionServiceCallbacks) {
            for (VisionServiceCallback visionServiceCallback : mVisionServiceCallbacks) {
                visionServiceCallback.onServiceConnected();
            }
        }
    }

    private void notifyServiceDisconnected() {
        synchronized (mVisionServiceCallbacks) {
            for (VisionServiceCallback visionServiceCallback : mVisionServiceCallbacks) {
                visionServiceCallback.onServiceDisconnected();
            }
            mVisionServiceCallbacks.clear();
        }
    }

    private void notifyAprSenseStarted() {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseStarted();
            }
        }
    }

    private void notifyRealSenseStopped() {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseStopped();
            }
        }
    }

    private void notifyRealSenseError(int errorCode, String errorMsg) {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseError(errorCode, errorMsg);
            }
        }
    }

    private class SharedMemoryHolder {
        ParcelFileDescriptor parcelFileDescriptor;
        FileInputStream inputStream;
    }

    Surface getSurface() {
        try {
            return mAprSenseAIDLService.getSurface();
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    void setVideoSource(int videoSource) {
        try {
            mAprSenseAIDLService.setVideoSource(videoSource);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    void setPoseRecognitionEnabled(boolean enabled) {
        try {
            mAprSenseAIDLService.setPoseRecognitionEnabled(enabled);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    boolean isPoseRecognitionEnabled() {
        try {
            return mAprSenseAIDLService.isPoseRecognitionEnabled();
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    public Version getVersion() {
        return new Version(VersionInfo.version_channel,
                VersionInfo.version_name,
                VersionInfo.version_code,
                VersionInfo.version_min);
    }

    /**
     * for test
     */
    protected void setAprSenseAIDLService(IAprSenseService aprSenseAIDLService) {
        mAprSenseAIDLService = aprSenseAIDLService;
    }


    /**
     * start ds5
     */

    public void startExtDepth(int leftConfig, int rightConfig) {
        checkConnected();
        startImageStreamToBufferByMemoryFile(ExtDepthStreamFactory.createExtDepthConfig(leftConfig, StreamType.EXT_DEPTH_L));
        startImageStreamToBufferByMemoryFile(ExtDepthStreamFactory.createExtDepthConfig(rightConfig, StreamType.EXT_DEPTH_R));
    }

    /**
     * stop ds5
     */

    public void stopExtDepth() {
        checkConnected();
        stopImageTransferMemoryFileBuffer(StreamType.EXT_DEPTH_L);
        stopImageTransferMemoryFileBuffer(StreamType.EXT_DEPTH_R);
    }

    RS2Intrinsic getExtDepthIntrinsics(int type) {
        checkConnected();
        try {
            return mAprSenseAIDLService.getExtDepthIntrinsics(type);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    public void setExtDepthEmitterEnabled(boolean enabled) {
        checkConnected();
        try {
            mAprSenseAIDLService.setExtDepthEmitterEnabled(enabled);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    void setAutoExposureEnabled(int type, boolean isEnabled) {
        checkConnected();
        try {
            mAprSenseAIDLService.setAutoExposureEnabled(type, isEnabled);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    int getExposureCondition() {
        checkDtsStarted();
        try {
            return mAprSenseAIDLService.getExposureCondition();
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    void recoverDevices(IAprSenseRecoverCallback callback) {
        checkConnected();
        try {
            mAprSenseAIDLService.recoverDevices(callback);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    public String setOptions(int streamType, int rs2OptionInt, float value) {
        checkConnected();
        String errorMsg;
        try {
            errorMsg = mAprSenseAIDLService.setOptions(streamType, rs2OptionInt, value);
        } catch (RemoteException e) {
            Logger.e(TAG, "setOptions RemoteException -> " + e.getLocalizedMessage());
            errorMsg = "remoteException " + e.getLocalizedMessage();
        }
        return errorMsg;
    }
}
