package com.segway.robot.sdk.vision;

import android.content.Context;
import android.graphics.Rect;
import android.util.SparseArray;
import android.view.Surface;

import com.segway.robot.sdk.base.bind.BindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.vision.calibration.ColorDepthCalibration;
import com.segway.robot.sdk.vision.calibration.MotionModuleCalibration;
import com.segway.robot.sdk.vision.frame.Frame;
import com.segway.robot.sdk.vision.frame.FrameImpl;
import com.segway.robot.sdk.vision.frame.FrameInfo;
import com.segway.robot.sdk.vision.imu.IMUDataCallback;
import com.segway.robot.sdk.vision.stream.StreamInfo;
import com.segway.robot.sdk.vision.stream.StreamType;

import java.nio.ByteBuffer;

/**
 * Vision is management and help class to communicate with VisionService
 */
public class Vision {
    private static Vision mInstance = new Vision();
    private BindController mBindController = new ForegroundBindController();
    private boolean mBindControllerInit;
    private SparseArray<FrameListener> mFrameListenerSparseArray = new SparseArray<>();
    private final VisionServiceManager mVisionServiceManager;
    private final DTS mDTS;

    public static Vision getInstance() {
        return mInstance;
    }

    protected Vision() {
        mVisionServiceManager = VisionServiceManager.getInstance();
        mDTS = new DTS(mVisionServiceManager);
    }

    /**
     * Connect to the vision service.
     * @param context any Android Context.
     * @param listener the connection state callback.
     * @return true if the connection is successful.
     */
    public synchronized boolean bindService(Context context, ServiceBinder.BindStateListener listener) {
        if (!mBindControllerInit) {
            mBindControllerInit = true;
            mBindController.init(context, mVisionServiceManager);
        }

        if (mBindController.bindAcceptable()) {
            return mVisionServiceManager.bindService(context, listener);
        }
        return false;
    }

    /**
     * Disconnect from the vision service.
     */
    public synchronized void unbindService() {
        mVisionServiceManager.unbindService();
    }

    /**
     * Get the StreamInfo of the currently activated stream.
     * @return an array of the StreamInfo.
     */
    public StreamInfo[] getActivatedStreamInfo() {
        return mVisionServiceManager.getActivatedStreamProfiles();
        /*// fish eye stream is not available in this sdk
        int count = 0;
        for (StreamInfo info : mVisionServiceManager.getActivatedStreamProfiles()) {
            if (info.getStreamType() == StreamType.COLOR || info.getStreamType() == StreamType.DEPTH) {
                count ++;
            }
        }

        StreamInfo[] streamInfos = new StreamInfo[count];
        int i = 0;
        for (StreamInfo info : mVisionServiceManager.getActivatedStreamProfiles()) {
            if (info.getStreamType() == StreamType.COLOR || info.getStreamType() == StreamType.DEPTH) {
                streamInfos[i++] = info;
            }
        }

        return streamInfos;*/
    }

    /**
     * Get the StreamInfo of the streamType.
     * @param streamType input stream type
     * @return detail info
     */
    public StreamInfo getStreamInfo(@StreamType.VisionStreamType int streamType) {
        StreamInfo[] infos = mVisionServiceManager.getActivatedStreamProfiles();
        for (StreamInfo info : infos) {
            if (info.getStreamType() == streamType) {
                return info;
            }
        }
        return null;
    }

    /**
     * Fill image to the surface for preview.
     * @param surface the surface to fill image.
     * @param streamType
     */
    public synchronized void startPreview(@StreamType.VisionStreamType int streamType, Surface surface) {
        if (surface == null) {
            throw new IllegalArgumentException("Surface is null");
        }

        if (getStreamInfo(streamType) == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        // TODO: 2016/9/27 support surface array by service
        mVisionServiceManager.preview(streamType, surface, false);
    }

    /**
     * Stop preview.
     * @param streamType
     */
    public synchronized void stopPreview(@StreamType.VisionStreamType int streamType) {
        if (getStreamInfo(streamType) == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        mVisionServiceManager.stopPreview(streamType);
    }

    /**
     * Start listening the frame and set the frame listener for streamType.
     * @param listener
     * @param streamType
     */
    public synchronized void startListenFrame(@StreamType.VisionStreamType final int streamType, final FrameListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("Listener is null");
        }

        StreamInfo streamInfo = getStreamInfo(streamType);
        if (streamInfo == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        synchronized (mFrameListenerSparseArray) {
            if (mFrameListenerSparseArray.get(streamType) != null) {
                throw new IllegalArgumentException("Stream type:" + streamType + " is listening");
            }

            mFrameListenerSparseArray.append(streamType, listener);
        }
        mVisionServiceManager.startImageTransferMemoryFileBuffer(streamInfo, new ImageStreamCallback() {
            @Override
            public void onNewImage(FrameInfo frameInfo, ByteBuffer buffer) {
                Frame frame = new FrameImpl(frameInfo, buffer);
                listener.onNewFrame(streamType, frame);
            }
        });
    }

    /**
     * Stop listening the frame and remove the frame listener.
     * @param streamType
     */
    public synchronized void stopListenFrame(@StreamType.VisionStreamType int streamType) {
        StreamInfo streamInfo = getStreamInfo(streamType);
        if (streamInfo == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        synchronized (mFrameListenerSparseArray) {
            if (mFrameListenerSparseArray.get(streamType) == null) {
                return;
            }
            mFrameListenerSparseArray.remove(streamType);
        }
        mVisionServiceManager.stopImageTransferMemoryFileBuffer(streamInfo.getStreamType());
    }

    /**
     * Get calibration data for color image and depth image
     * @return calibration data
     */
    public ColorDepthCalibration getColorDepthCalibrationData() {
        return mVisionServiceManager.getColorDepthCalibrationData();
    }

    /**
     * Definition of the frame listener.
     */
    public interface FrameListener {
        void onNewFrame(@StreamType.VisionStreamType int streamType, Frame frame);
    }

    // Person Detection and Tracking

    /**
     * Get the DTS instance
     * @return DTS instance
     */
    public DTS getDTS() {
        return mDTS;
    }

    /**
     * Set the IMU data callback
     *
     * @param imuCallback
     */
    public void setIMUCallback(IMUDataCallback imuCallback) {
        mVisionServiceManager.addIMUDataCallback(imuCallback);
    }

    /**
     * Remove IMU data callback
     */
    public void removeIMUCallback(IMUDataCallback imuCallback) {
        mVisionServiceManager.removeIMUDataCallback(imuCallback);
    }

    /**
     * Get motion module calibration data
     *
     * @return calibration data
     */
    public MotionModuleCalibration getMotionModuleCalibrationData() {
        return mVisionServiceManager.getMotionModuleCalibrationData();
    }

    /**
     * Get fish eye camera auto exposure enable state
     *
     * @return auto exposure enable state
     */
    public boolean isFishEyeAutoExposureEnabled() {
        return mVisionServiceManager.isFishEyeAutoExposureEnabled();
    }

    /**
     * Set fish eye camera auto exposure enable state
     *
     * @param enable auto exposure enable state
     */
    public void enableFishEyeAutoExposure(boolean enable) {
        mVisionServiceManager.enableFishEyeAutoExposure(enable);
    }

    /**
     * Get fish eye EV compensation value
     * This api is valid only in auto exposure mode
     *
     * @return fish eye EV compensation value
     */
    public float getFishEyeEvCompensation() {
        return mVisionServiceManager.getFishEyeEvCompensation();
    }

    /**
     * Set fish eye EV compensation value
     * This api is valid only in auto exposure mode
     *
     * @param compensation fish eye EV compensation value
     */
    public void setFishEyeEvCompensation(float compensation) {
        mVisionServiceManager.setFishEyeEvCompensation(compensation);
    }

    /**
     * Get fish eye exposure value
     * This api is valid only in manual exposure mode
     *
     * @return fish eye exposure value
     */
    public float getFishEyeManualExposure() {
        return mVisionServiceManager.getFishEyeManualExposure();
    }

    /**
     * Set fish eye exposure value
     * This api is valid only in manual exposure mode
     *
     * @param exposure fish eye exposure value
     */
    public void setFishEyeManualExposure(float exposure) {
        mVisionServiceManager.setFishEyeManualExposure(exposure);
    }

    /**
     * Get fish eye gain value
     * This api is valid only in manual exposure mode
     *
     * @return fish eye gain value
     */
    public float getFishEyeManualGain() {
        return mVisionServiceManager.getFishEyeManualGain();
    }

    /**
     * Set fish eye gain value
     * This api is valid only in manual exposure mode
     *
     * @param gain fish eye gain value
     */
    public void setFishEyeManualGain(float gain) {
        mVisionServiceManager.setFishEyeManualGain(gain);
    }

    /**
     * Set fish eye auto exposure ROI rect
     *
     * @param rect rect to set exposure
     */
    public void setFishEyeAutoExposureROI(Rect rect) {
        mVisionServiceManager.setFishEyeAutoExposureROI(rect);
    }
}
