package com.segway.robot.sdk.vision;

import android.media.MediaRecorder;
import android.view.Surface;

import com.segway.robot.algo.dts.DTSPerson;
import com.segway.robot.algo.dts.PersonDetectListener;
import com.segway.robot.algo.dts.PersonTrackingListener;

/**
 * Detection-tracking system(DTS) module is a vision-based tracking system.
 * It gives a human's upper body detection as well as tracking implementation.
 */
public class DTS {
    private int mVideoSource = 0;
    private Surface mSurface;
    private VisionServiceManager mVisionServiceManager;
    private boolean isVideoSelected = false;

    DTS(VisionServiceManager visionServiceManager) {
        mVisionServiceManager = visionServiceManager;
    }

    /**
     * Set the video source for DTS.
     * The video source id defined in DTS {@link VideoSource}.
     * DTS requires main camera frame, there are two video source: 1. {@link VideoSource#CAMERA}
     * 2.{@link VideoSource#SURFACE}.
     * If set to {@link VideoSource#CAMERA} mode, Vision Service will open the main camera,
     * you can use setPreviewDisplay to preview the camera stream.
     * If set to {@link VideoSource#SURFACE}, Vision Service will not open the main camera
     * and you should open and manage the main camera life circle.
     * This method must be called before {@link DTS#start()}.
     *
     * @param videoSource the video source of DTS
     */
    public void setVideoSource(int videoSource) {
        if (videoSource != VideoSource.SURFACE && videoSource != VideoSource.CAMERA) {
            throw new IllegalArgumentException("Illegal VideoSource mode!");
        }
        isVideoSelected = true;
        mVideoSource = videoSource;
    }

    /**
     * Set the surface to preview main camera stream
     * This method is only available while choose camera as the video source
     * {@link VideoSource#CAMERA}
     *
     * @param surface the surface to preview main camera stream
     */
    public void setPreviewDisplay(Surface surface) {
        if (mVideoSource == VideoSource.CAMERA) {
            mSurface = surface;
        } else {
            throw new IllegalStateException("Only be supported in VideoSource.CAMERA mode!");
        }
    }

    /**
     * Get the surface for DTS
     * If in {@link VideoSource#CAMERA} mode, return the surface that
     * has been set in {@link DTS#setPreviewDisplay(Surface)}.
     * If in {@link VideoSource#SURFACE} mode, call {@link DTS#start()} first, then return the
     * surface for detection, otherwise will return null.
     *
     * @return the surface for DTS
     */
    public Surface getSurface() {
        switch (mVideoSource) {
            case VideoSource.CAMERA:
                return mSurface;
            case VideoSource.SURFACE:
                return mVisionServiceManager.getSurface();
            default:
                return null;
        }
    }

    /**
     * Start the DTS module
     * Before call this method, call {@link DTS#setVideoSource(int)} first.
     */
    public void start() {
        if (!isVideoSelected) {
            throw new IllegalStateException("Please set video source first!");
        }
        mVisionServiceManager.enablePersonTrackingAndDetection(mSurface, mVideoSource);
    }

    /**
     * Stop the DTS module
     */
    public void stop() {
        mVisionServiceManager.disablePersonTrackingAndDetection();
    }

    /**
     * Start detecting person detection.
     *
     * @param listener the detected person callback PersonTrackingListener which can be null.
     */
    public void startDetectingPerson(PersonDetectListener listener) {
        mVisionServiceManager.startDetectingPerson(listener);
    }

    /**
     * Stop detecting person.
     */
    public void stopDetectingPerson() {
        mVisionServiceManager.stopDetectingPerson();
    }

    /**
     * Detect persons
     * This method return one or more person if detected, otherwise an empty array if timeout.
     *
     * @param timeoutMicroSeconds timeout in micro seconds
     * @return detected persons DTSPerson
     */
    public DTSPerson[] detectPersons(long timeoutMicroSeconds) {
        return mVisionServiceManager.detectPersons(timeoutMicroSeconds);
    }

    /**
     * Start person tracking
     * If the target person is missing for more than a certain time set by timeout,
     * the tracking will stop and the onPersonTrackingError in listener will be called.
     *
     * @param person   the target to track
     * @param timeout  timeout in micro seconds
     * @param listener listener for being notified that a new state of person tracking
     */
    public void startPersonTracking(DTSPerson person, long timeout, PersonTrackingListener listener) {
        mVisionServiceManager.startPersonTracking(person, timeout, listener);
    }

    /**
     * Stop person tracking
     */
    public void stopPersonTracking() {
        mVisionServiceManager.stopPersonTracking();
    }

    /**
     * Defines the video source. These constants are used with
     * {@link MediaRecorder#setVideoSource(int)}.
     */
    public final class VideoSource {
        private VideoSource() {
        }

        /**
         * Camera video source
         * <p>
         * Using the {@link android.hardware.Camera} API as video source.
         * </p>
         */
        public static final int CAMERA = 1;

        /**
         * Surface video source
         * <p>
         * Using a Surface as video source.
         * </p><p>
         * This flag must be used when recording from an
         * {@link android.hardware.camera2} API source.
         * </p><p>
         * When using this video source type, use {@link MediaRecorder#getSurface()}
         * to retrieve the surface created by MediaRecorder.
         */
        public static final int SURFACE = 2;
    }
}
