package com.segway.robot.sdk.vision.internal.socket;

import android.net.LocalSocket;
import android.net.LocalSocketAddress;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;

import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.vision.stream.StreamInfo;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;

/**
 * Created by ark338 on 16/4/7.
 */
public class FrameClientThread{
    private static final String TAG = "FrameClientThread";
    private static final int CONNECT = 0;
    private static final int READ = 1;

    final int MAX_RETRY = 5;
    final long INTERVAL = 500;

    StreamInfo mStreamInfo;
    String mServerAddress;
    boolean isConnected;
    LocalSocket mLocalSocket;
    boolean isConnectFailed;
    boolean isTransferEnded;
    boolean isReadSuccess;
    ReadableByteChannel mReadableByteChannel;
    HandlerThread mHandlerThread;
    Handler mHandler;

    public FrameClientThread(StreamInfo mStreamInfo) {
        this.mStreamInfo = mStreamInfo;
        mHandlerThread = new HandlerThread("Read Frame Thread " + mStreamInfo.getStreamType());
        mHandlerThread.start();
        mHandler = new WorkHandler(mHandlerThread.getLooper());
    }

    public void connectServer(String address) {
        this.mServerAddress = address;
        mHandler.sendEmptyMessage(CONNECT);
    }

    public void disconnect() {
        if (mLocalSocket != null) {
            try {
                mLocalSocket.close();
            } catch (IOException e) {
            }
        }
    }

    public synchronized void readFrame(final ByteBuffer buffer, final int length) {
        isTransferEnded = false;
        isReadSuccess = false;
        Message msg = new Message();
        msg.what = READ;
        msg.obj = buffer;
        msg.arg1 = length;
        mHandler.sendMessage(msg);
    }

    public void interrupt() {
        mHandlerThread.interrupt();
    }

    public boolean isConnectFailed() {
        return isConnectFailed;
    }

    public boolean isConnected() {
        return isConnected;
    }

    public boolean isTransferEnded() {
        return isTransferEnded;
    }

    public boolean isReadSuccess() {
        return isReadSuccess;
    }

    private void doConnect() {
        int count = 0;
        while (count < MAX_RETRY) {
            try {
                Thread.sleep(INTERVAL);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            mLocalSocket = new LocalSocket();
            try {
                mLocalSocket.connect(new LocalSocketAddress(mServerAddress));
                InputStream inputStream = mLocalSocket.getInputStream();
                mReadableByteChannel = Channels.newChannel(inputStream);
                isConnected = true;
                return;
            } catch (IOException e) {
                // TODO:
            }
            count++;
        }
        isConnectFailed = true;
    }

    private void doRead(ByteBuffer buffer, int length) {
        try {
            int count = 0;
            buffer.rewind();
            while (count < length) {
                int read = mReadableByteChannel.read(buffer);
                if (read == -1) {
                    break;
                }
                count += read;
            }
            isReadSuccess = true;
        } catch (IOException e) {
            Logger.e(TAG ,"read frame exception", e);
        }
        isTransferEnded = true;
        synchronized (FrameClientThread.this) {
            FrameClientThread.this.notify();
        }
    }

    private class WorkHandler extends Handler {
        public WorkHandler(Looper looper) {
            super(looper);
        }

        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            switch (msg.what) {
                case CONNECT:
                    doConnect();
                    break;
                case READ:
                    ByteBuffer buffer = (ByteBuffer) msg.obj;
                    int length = msg.arg1;
                    doRead(buffer, length);
                    break;
            }

        }
    }

}
