package com.segway.robot.sdk.vision;

import android.media.MediaRecorder;
import android.view.Surface;

import com.segway.robot.algo.dts.DTSPerson;
import com.segway.robot.algo.dts.PersonTrackingListener;

/**
 * Detection-tracking system(DTS) module is a vision-based tracking system.
 * It gives a human's upper body detection as well as tracking implementation.
 */
public class DTS {
    private int mVideoSource = VideoSource.CAMERA;
    private Surface mSurface;
    private VisionServiceManager mVisionServiceManager;

    DTS(VisionServiceManager visionServiceManager) {
        mVisionServiceManager = visionServiceManager;
    }

    /**
     * Set the video source of DTS
     * The video source id defined in DTS@VideoSource
     * DTS require main camera frame, there is two video source: 1. camera 2. surface
     * If you choose the camera as the video source, Vision Service will open the main camera,
     * you can use setPreviewDisplay to preview the camera stream
     * If you choose the surface as the video source, Vision Service will not open the main camera
     * and you should open and manage the main camera.
     * Camera is the only supported video source for now.
     * @param videoSource the video source of DTS
     */
    public void setVideoSource(int videoSource) {
        if (videoSource != VideoSource.CAMERA) {
            throw new IllegalArgumentException("Only support camera video source now");
        }
        mVideoSource = videoSource;
    }

    /**
     * Set the surface to preview main camera stream
     * This method is only available while choose camera as the video source
     * @param surface the surface to preview main camera stream
     */
    public void setPreviewDisplay(Surface surface) {
        mSurface = surface;
    }

    /**
     * Get the surface for DTS
     * This method is only available while choose surface as the video source
     * @return the surface for DTS
     */
    public Surface getSurface() {
        // TODO: 2017/1/24
        return null;
    }

    /**
     * Start the DTS module
     */
    public void start() {
        mVisionServiceManager.enablePersonTrackingAndDetection(mSurface);
    }

    /**
     * Stop the DTS module
     */
    public void stop() {
        mVisionServiceManager.disablePersonTrackingAndDetection();
    }

    /**
     * Detect persons
     * This method return one or more person if detected, otherwise an empty array if timeout.
     * @param timeoutMicroSeconds timeout in micro seconds
     * @return detected persons
     */
    public DTSPerson[] detectPersons(long timeoutMicroSeconds) {
        return mVisionServiceManager.detectPersons(timeoutMicroSeconds);
    }

    /**
     * Start person tracking
     * If the target person is missing for more than a certain time set by timeout,
     * the tracking will stop and the onPersonTrackingError in listener will be called.
     * @param person the target to track
     * @param timeout timeout in micro seconds
     * @param listener listener for being notified that a new state of person tracking
     */
    public void startPersonTracking(DTSPerson person, long timeout, PersonTrackingListener listener) {
        mVisionServiceManager.startPersonTracking(person, timeout, listener);
    }

    /**
     * Stop person tracking
     */
    public void stopPersonTracking() {
        mVisionServiceManager.stopPersonTracking();
    }

    /**
     * Defines the video source. These constants are used with
     * {@link MediaRecorder#setVideoSource(int)}.
     */
    public final class VideoSource {
        private VideoSource() {}

        /** Camera video source
         * <p>
         * Using the {@link android.hardware.Camera} API as video source.
         * </p>
         */
        public static final int CAMERA = 1;

        /** Surface video source
         * <p>
         * Using a Surface as video source.
         * </p><p>
         * This flag must be used when recording from an
         * {@link android.hardware.camera2} API source.
         * </p><p>
         * When using this video source type, use {@link MediaRecorder#getSurface()}
         * to retrieve the surface created by MediaRecorder.
         */
        public static final int SURFACE = 2;
    }
}
