package com.segway.robot.sdk.vision.person;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * Point3DF holds three float coordinates
 */
public class Point3DF implements Parcelable {
    public float x;
    public float y;
    public float z;

    public Point3DF() {}

    public Point3DF(float x, float y, float z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }

    public Point3DF(Point3DF p) {
        this.x = p.x;
        this.y = p.y;
        this.z = p.z;
    }

    /**
     * Set the point's x, y, and z coordinates.
     */
    public final void set(float x, float y, float z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }

    /**
     * Set the point's x, y, and z coordinates to the coordinates of p.
     */
    public final void set(Point3DF p) {
        this.x = p.x;
        this.y = p.y;
        this.z = p.z;
    }

    public final void negate() {
        x = -x;
        y = -y;
        z = -z;
    }

    public final void offset(float dx, float dy, float dz) {
        x += dx;
        y += dy;
        z += dz;
    }

    /**
     * Returns true if the point's coordinates equal (x,y,z).
     */
    public final boolean equals(float x, float y, float z) {
        return this.x == x && this.y == y && this.z == z;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Point3DF point3DF = (Point3DF) o;

        if (Float.compare(point3DF.x, x) != 0) return false;
        if (Float.compare(point3DF.y, y) != 0) return false;
        if (Float.compare(point3DF.z, z) != 0) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result = (x != +0.0f ? Float.floatToIntBits(x) : 0);
        result = 31 * result + (y != +0.0f ? Float.floatToIntBits(y) : 0);
        result = 31 * result + (z != +0.0f ? Float.floatToIntBits(z) : 0);
        return result;
    }

    @Override
    public String toString() {
        return "Point3DF(" + x + ", " + y + ", " + z + ")";
    }

    /**
     * Return the Euclidean distance from (0,0,0) to the point.
     */
    public final float length() {
        return length(x, y, z);
    }

    /**
     * Returns the Euclidean distance from (0,0,0) to (x,y,z).
     */
    public static float length(float x, float y, float z) {
        return (float) Math.sqrt((x*x) + (y*y) + (z*z));
    }

    /**
     * the Parcelable interface methods.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Write this point to the specified parcel. To restore a point from
     * a parcel, use readFromParcel().
     * @param out the parcel where the point's coordinates are written.
     */
    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeFloat(x);
        out.writeFloat(y);
        out.writeFloat(z);
    }

    public static final Creator<Point3DF> CREATOR = new Creator<Point3DF>() {
        /**
         * Return a new point from the data in the specified parcel.
         */
        public Point3DF createFromParcel(Parcel in) {
            Point3DF r = new Point3DF();
            r.readFromParcel(in);
            return r;
        }

        /**
         * Return an array of rectangles of the specified size.
         */
        public Point3DF[] newArray(int size) {
            return new Point3DF[size];
        }
    };

    /**
     * Set the point's coordinates from the data stored in the specified
     * parcel. To write a point to a parcel, call writeToParcel().
     *
     * @param in the parcel where the point's coordinates are read.
     */
    public void readFromParcel(Parcel in) {
        x = in.readFloat();
        y = in.readFloat();
        z = in.readFloat();
    }
}

