package com.segway.robot.sdk.vision.stream;

import android.os.Parcel;
import android.os.Parcelable;

import com.segway.robot.sdk.vision.stream.PixelFormat.VisionPixelFormat;
import com.segway.robot.sdk.vision.stream.StreamType.VisionStreamType;

/**
 * The stream profile for color and depth streams.
 */
public class StreamInfo implements Parcelable{
    int streamType;
    int resolution;
    int width;
    int height;
    int fps;
    int pixelFormat;
    float pixelBytes;

    /**
     * Constructor of StreamInfo.
     * @param streamType the stream type, see the definition of StreamType.
     * @param width the width of a stream.
     * @param height the height of a stream.
     * @param fps the frame rate of a stream, see the definition of FrameRate.
     * @param pixelFormat the pixel format of a stream, see the definition of PixelFormat.
     */
    public StreamInfo(int streamType, int width, int height, int fps, int pixelFormat) {
        this.streamType = streamType;
        this.width = width;
        this.height = height;
        this.fps = fps;
        this.pixelFormat = pixelFormat;
        this.pixelBytes = PixelFormat.getPixelBytes(pixelFormat);
        this.resolution = Resolution.toVisionServiceResolution(width, height);
    }

    /**
     * Create a StreamInfo instance.
     * @param streamType the stream type, see the definition of StreamType.
     * @param resolution the resolution, see the definition of Resolution.
     * @param fps frame the rate of stream, see definition of FrameRate.
     * @param pixelFormat the pixel format of stream, see the definition of PixelFormat.
     * @return the StreamInfo instance.
     */
    public static StreamInfo create(@VisionStreamType int streamType,
                                    @Resolution.VisionResolution int resolution,
                                    @FrameRate.VisionFrameRate int fps,
                                    @VisionPixelFormat int pixelFormat) {
        return new StreamInfo(streamType,
                Resolution.getWidth(resolution),
                Resolution.getHeight(resolution),
                fps, pixelFormat);
    }

    /**
     * Get the stream type.
     * @return the stream type.
     */
    @VisionStreamType
    public int getStreamType() {
        return streamType;
    }

    /**
     * Get the resolution type.
     * @return the resolution type.
     */
    public int getResolution() {
        return resolution;
    }

    /**
     * Get the width.
     * @return the width.
     */
    public int getWidth() {
        return width;
    }

    /**
     * Get the height.
     * @return the height.
     */
    public int getHeight() {
        return height;
    }

    /**
     * Get the frame rate.
     * @return the frame rate.
     */
    public int getFps() {
        return fps;
    }

    /**
     * Get the pixel format.
     * @return the pixel format.
     */
    @VisionPixelFormat
    public int getPixelFormat() {
        return pixelFormat;
    }

    /**
     * Get the pixel length in bytes.
     * @return the pixel length in bytes.
     */
    public float getPixelBytes() {
        return pixelBytes;
    }

    /**
     * Hash code.
     * @return hash code.
     */
    @Override
    public int hashCode() {
        return (width << 24) + (height << 16) + (fps << 12) + (pixelFormat << 8) + streamType;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Write an instance to Parcel.
     * @param dest the dest Parcel.
     * @param flags the flags.
     */
    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeInt(streamType);
        dest.writeInt(resolution);
        dest.writeInt(width);
        dest.writeInt(height);
        dest.writeInt(fps);
        dest.writeInt(pixelFormat);
    }

    /**
     * Creator instance for StreamInfo
     */
    public static final Creator<StreamInfo> CREATOR = new Creator<StreamInfo>() {
        @Override
        public StreamInfo createFromParcel(Parcel source) {
            int streamType = source.readInt();
            int resolution = source.readInt();
            int width = source.readInt();
            int height = source.readInt();
            int fps = source.readInt();
            int pixelFormat = source.readInt();
            return new StreamInfo(streamType, width, height, fps, pixelFormat);
        }

        @Override
        public StreamInfo[] newArray(int size) {
            return new StreamInfo[size];
        }
    };
}
