package com.segway.robot.sdk.vision;

import android.view.Surface;

import com.segway.robot.algo.dts.DTSPerson;
import com.segway.robot.algo.dts.PersonDetectListener;
import com.segway.robot.algo.dts.PersonTrackingListener;

/**
 * Detection-tracking system (DTS) module is a vision-based tracking system.
 * It provides a person's upper body detection and tracking implementation.
 */
public class DTS {
    private int mVideoSource = 0;
    private Surface mSurface;
    private VisionServiceManager mVisionServiceManager;
    private boolean isVideoSelected = false;

    DTS(VisionServiceManager visionServiceManager) {
        mVisionServiceManager = visionServiceManager;
    }

    /**
     * Set the video source for DTS.
     * The video source id defined in DTS {@link VideoSource}.
     * DTS requires main camera frame, there are two video source: 1. {@link VideoSource#CAMERA}
     * 2.{@link VideoSource#SURFACE}.
     * If set to {@link VideoSource#CAMERA} mode, Vision Service will open the main camera,
     * you can use setPreviewDisplay to preview the camera stream.
     * If set to {@link VideoSource#SURFACE}, Vision Service will not open the main camera
     * and you should open and manage the main camera life circle.
     * This method must be called before {@link DTS#start()}.
     *
     * @param videoSource the video source of DTS
     */
    public void setVideoSource(int videoSource) {
        if (videoSource != VideoSource.SURFACE && videoSource != VideoSource.CAMERA) {
            throw new IllegalArgumentException("Illegal VideoSource mode!");
        }
        isVideoSelected = true;
        mVideoSource = videoSource;
    }

    /**
     * Set the surface to preview main camera stream
     * This method is only available while choose camera as the video source
     * {@link VideoSource#CAMERA}
     *
     * @param surface the surface to preview main camera stream
     */
    public void setPreviewDisplay(Surface surface) {
        if (mVideoSource == VideoSource.CAMERA) {
            mSurface = surface;
        } else {
            throw new IllegalStateException("Only be supported in VideoSource.CAMERA mode!");
        }
    }

    /**
     * Get the surface for DTS
     * If in {@link VideoSource#CAMERA} mode, return the surface that
     * has been set in {@link DTS#setPreviewDisplay(Surface)}.
     * If in {@link VideoSource#SURFACE} mode, call {@link DTS#start()} first, then return the
     * surface for detection, otherwise will return null.
     *
     * @return the surface for DTS
     */
    public Surface getSurface() {
        switch (mVideoSource) {
            case VideoSource.CAMERA:
                return mSurface;
            case VideoSource.SURFACE:
                return mVisionServiceManager.getSurface();
            default:
                return null;
        }
    }

    /**
     * Start the DTS module
     * Before call this method, call {@link DTS#setVideoSource(int)} first.
     */
    public void start() {
        if (!isVideoSelected) {
            throw new IllegalStateException("Please set video source first!");
        }
        mVisionServiceManager.enablePersonTrackingAndDetection(mSurface, mVideoSource);
    }

    /**
     * Stop the DTS module
     */
    public void stop() {
        mVisionServiceManager.disablePersonTrackingAndDetection();
    }

    /**
     * Start detecting person.
     *
     * @param listener the person detection result callback {@link PersonDetectListener} and it
     *                 can't be null.
     */
    public void startDetectingPerson(PersonDetectListener listener) {
        mVisionServiceManager.startDetectingPerson(listener);
    }

    /**
     * Stop detecting person.
     */
    public void stopDetectingPerson() {
        mVisionServiceManager.stopDetectingPerson();
    }

    /**
     * Detect persons.
     * This method return one or more person if any person is detected, otherwise return an empty
     * array if timeout or no person is detected.
     *
     * @param timeoutMicroSeconds timeout in micro seconds.
     * @return the array of detected persons.
     */
    public DTSPerson[] detectPersons(long timeoutMicroSeconds) {
        return mVisionServiceManager.detectPersons(timeoutMicroSeconds);
    }

    /**
     * Start person tracking.
     * If the target person is missing for more than a certain time which is set by timeout,
     * the tracking will be stopped and the onPersonTrackingError interface will be called.
     *
     * @param person   the target person to be tracked.
     * @param timeout  timeout in micro seconds.
     * @param listener the listener {@link PersonTrackingListener} for person tracking result.
     */
    public void startPersonTracking(DTSPerson person, long timeout, PersonTrackingListener listener) {
        mVisionServiceManager.startPersonTracking(person, timeout, listener);
    }

    /**
     * Stop person tracking
     */
    public void stopPersonTracking() {
        mVisionServiceManager.stopPersonTracking();
    }

    /**
     * Enable or disable pose recognition module. If pose recognition is no longer needed, it is recommended that
     * disable it.
     *
     * @param enabled true to enable pose recognition, false to disable it
     */
    public void setPoseRecognitionEnabled(boolean enabled) {
        mVisionServiceManager.setPoseRecognitionEnabled(enabled);
    }

    /**
     * To get whether the pose recognition is enabled.
     *
     * @return true if the pose recognition is enabled, otherwise false.
     */
    public boolean isPoseRecognitionEnabled() {
        return mVisionServiceManager.isPoseRecognitionEnabled();
    }

    /**
     * Defines the video source mode. These constants are used with
     * {@link DTS#setVideoSource(int)}.
     */
    public final class VideoSource {
        private VideoSource() {
        }

        /**
         * Camera video source mode.
         * If set DTS video source in this mode, when start DTS, SDK will
         * use android.hardware.camera2 API to open the camera device automatically.
         */
        public static final int CAMERA = 1;

        /**
         * Surface video source mode.
         * If setDTS video source in this mode, when start DTS, SDK will create a surface to get
         * image buffer.
         */
        public static final int SURFACE = 2;
    }
}
