package com.segway.robot.sdk.vision;

import android.content.Context;
import android.util.SparseArray;
import android.view.Surface;

import com.segway.robot.sdk.base.bind.BindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.vision.frame.Frame;
import com.segway.robot.sdk.vision.frame.FrameImpl;
import com.segway.robot.sdk.vision.frame.FrameInfo;
import com.segway.robot.sdk.vision.stream.StreamInfo;
import com.segway.robot.sdk.vision.stream.StreamType;

import java.nio.ByteBuffer;

/**
 * Vision is management and help class to communicate with VisionService
 */
public class Vision {
    private static Vision mInstance = new Vision();
    private BindController mBindController = new ForegroundBindController();
    private boolean mBindControllerInit;
    private SparseArray<FrameListener> mFrameListenerSparseArray = new SparseArray<>();
    private final VisionServiceManager mVisionServiceManager;

    public static Vision getInstance() {
        return mInstance;
    }

    protected Vision() {
        mVisionServiceManager = VisionServiceManager.getInstance();
    }

    /**
     * Connect to the vision service.
     * @param context any Android Context.
     * @param listener the connection state callback.
     * @return true if the connection is successful.
     */
    public synchronized boolean bindService(Context context, ServiceBinder.BindStateListener listener) {
        if (!mBindControllerInit) {
            mBindControllerInit = true;
            mBindController.init(context, mVisionServiceManager);
        }

        if (mBindController.bindAcceptable()) {
            return mVisionServiceManager.bindService(context, listener);
        }
        return false;
    }

    /**
     * Disconnect from the vision service.
     */
    public synchronized void unbindService() {
        mVisionServiceManager.unbindService();
    }

    /**
     * Get the StreamInfo of the currently activated stream.
     * @return an array of the StreamInfo.
     */
    public StreamInfo[] getActivatedStreamInfo() {
        return mVisionServiceManager.getActivatedStreamProfiles();
    }

    /**
     * Get the StreamInfo of the streamType.
     * @param streamType input stream type
     * @return detail info
     */
    public StreamInfo getStreamInfo(@StreamType.VisionStreamType int streamType) {
        StreamInfo[] infos = mVisionServiceManager.getActivatedStreamProfiles();
        for (StreamInfo info : infos) {
            if (info.getStreamType() == streamType) {
                return info;
            }
        }
        return null;
    }

    /**
     * Fill image to the surface for preview.
     * @param surface the surface to fill image.
     * @param streamType
     */
    public synchronized void startPreview(int streamType, Surface surface) {
        if (surface == null) {
            throw new IllegalArgumentException("Surface is null");
        }

        if (getStreamInfo(streamType) == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        // TODO: 2016/9/27 support surface array by service
        mVisionServiceManager.preview(streamType, surface, false);
    }

    /**
     * Stop preview.
     * @param streamType
     */
    public synchronized void stopPreview(int streamType) {
        if (getStreamInfo(streamType) == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        mVisionServiceManager.stopPreview(streamType);
    }

    /**
     * Start listening the frame and set the frame listener for streamType.
     * @param listener
     * @param streamType
     */
    public synchronized void startListenFrame(final int streamType, final FrameListener listener) {
        if (listener == null) {
            throw new IllegalArgumentException("Listener is null");
        }

        StreamInfo streamInfo = getStreamInfo(streamType);
        if (streamInfo == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        if (mFrameListenerSparseArray.get(streamType) != null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " is listening");
        }

        mFrameListenerSparseArray.append(streamType, listener);
        mVisionServiceManager.startImageStream(streamInfo, new ImageStreamCallback() {
            @Override
            public void onNewImage(FrameInfo frameInfo, ByteBuffer buffer) {
                Frame frame = new FrameImpl(frameInfo, buffer);
                listener.onNewFrame(streamType, frame);
            }
        });
    }

    /**
     * Stop listening the frame and remove the frame listener.
     * @param streamType
     */
    public synchronized void stopListenFrame(int streamType) {
        StreamInfo streamInfo = getStreamInfo(streamType);
        if (streamInfo == null) {
            throw new IllegalArgumentException("Stream type:" + streamType + " illegal or not activated");
        }

        if (mFrameListenerSparseArray.get(streamType) == null) {
            return;
        }

        mFrameListenerSparseArray.remove(streamType);
        mVisionServiceManager.stopImageStream(streamInfo);
    }

    /**
     * Definition of the frame listener.
     */
    public interface FrameListener {
        void onNewFrame(int streamType, Frame frame);
    }

    // coming in next stage
    // Person Detection and Tracking
}
