package com.segway.robot.sdk.vision.stream;

import android.support.annotation.IntDef;

/**
 * the supported resolution.
 */
public abstract class Resolution {
    static final int COLOR_MASK = 0x40000000;
    static final int DEPTH_MASK = 0x20000000;
    static final int FISH_EYE_MASK = 0x10000000;


    /**
     * the supported resolution for a color image.
     */
    public interface Color {
        int R_320_240 = ((320 << 16) | 240) | COLOR_MASK;
        int R_640_480 = ((640 << 16) | 480) | COLOR_MASK;
        int R_1280_720 = ((1280 << 16) | 720) | COLOR_MASK;
        int R_1920_1080 = ((1920 << 16) | 1080) | COLOR_MASK;
    }

    /**
     * the supported resolution for a depth image.
     */
    public interface Depth {
        int R_320_240 = ((320 << 16) | 240) | DEPTH_MASK;
        int R_480_360 = ((480 << 16) | 360) | DEPTH_MASK;
        int R_628_468 = ((628 << 16) | 468) | DEPTH_MASK;
    }

    /**
     * the supported resolution for a fish eye image.
     */
    public interface FishEye {
        int R_640_480 = ((640 << 16) | 480) | FISH_EYE_MASK;
    }

    @IntDef({Color.R_320_240, Color.R_640_480, Color.R_1280_720, Color.R_1920_1080,
            Depth.R_320_240, Depth.R_480_360, Depth.R_628_468, FishEye.R_640_480})
    public @interface VisionResolution{};

    /**
     * Get the width from the resolution.
     * @param resolution the resolution defined for color or depth.
     * @return the width of the input resolution.
     */
    public static int getWidth (@Resolution.VisionResolution int resolution) {
        switch (resolution) {
            case Resolution.Color.R_320_240:
                return 320;
            case Resolution.Color.R_640_480:
                return 640;
            case Resolution.Color.R_1280_720:
                return 1280;
            case Resolution.Color.R_1920_1080:
                return 1920;
            case Resolution.Depth.R_320_240:
                return 320;
            case Resolution.Depth.R_480_360:
                return 480;
            case Resolution.Depth.R_628_468:
                return 628;
            case Resolution.FishEye.R_640_480:
                return 640;
            default:
                return 0;
        }
    }

    /**
     * Get the height from the resolution.
     * @param resolution the resolution defined for color or depth.
     * @return the height of the input resolution.
     */
    public static int getHeight (@Resolution.VisionResolution int resolution) {
        switch (resolution) {
            case Resolution.Color.R_320_240:
                return 240;
            case Resolution.Color.R_640_480:
                return 480;
            case Resolution.Color.R_1280_720:
                return 720;
            case Resolution.Color.R_1920_1080:
                return 1080;
            case Resolution.Depth.R_320_240:
                return 240;
            case Resolution.Depth.R_480_360:
                return 360;
            case Resolution.Depth.R_628_468:
                return 468;
            case Resolution.FishEye.R_640_480:
                return 480;
            default:
                return 0;
        }
    }

    /**
     * Get the resolution from the width and height.
     * @param width the width.
     * @param height the height.
     * @return Resolution.Color or Resolution.Depth defined value, or 0 for an error.
     */
    public static int toVisionServiceResolution(int width, int height) {
        int colorResolution = (width << 16) | height | COLOR_MASK;
        int depthResolution = (width << 16) | height | DEPTH_MASK;
        int fishEyeResolution = (width << 16) | height | FISH_EYE_MASK;

        switch (colorResolution) {
            case Resolution.Color.R_320_240:
                return Resolution.Color.R_320_240;
            case Resolution.Color.R_640_480:
                return Resolution.Color.R_640_480;
            case Resolution.Color.R_1280_720:
                return Resolution.Color.R_1280_720;
            case Resolution.Color.R_1920_1080:
                return Resolution.Color.R_1920_1080;
        }

        switch (depthResolution) {
            case Resolution.Depth.R_320_240:
                return Resolution.Depth.R_320_240;
            case Resolution.Depth.R_480_360:
                return Resolution.Depth.R_480_360;
            case Resolution.Depth.R_628_468:
                return Resolution.Depth.R_628_468;
        }

        switch (fishEyeResolution) {
            case Resolution.FishEye.R_640_480:
                return Resolution.FishEye.R_640_480;
        }

        return 0;
    }
}
