package com.segway.robot.sdk.vision.internal.socket;

import android.net.LocalSocket;
import android.net.LocalSocketAddress;

import com.segway.robot.sdk.vision.ImageStreamCallback;
import com.segway.robot.sdk.vision.internal.frame.BaseFrameInfo;
import com.segway.robot.sdk.vision.internal.frame.FrameBuffer;
import com.segway.robot.sdk.vision.internal.frame.RecyclableFrame;
import com.segway.robot.sdk.vision.stream.StreamProfile;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;

import timber.log.Timber;

/**
 * Created by ark338 on 16/3/15.
 */
public class ImageReaderThreadManager {
    private Map<Integer, Thread> mWorkThreadMap = new HashMap<>();

    private static ImageReaderThreadManager mInstance;
    private ImageReaderThreadManager(){}
    public static synchronized ImageReaderThreadManager getInstance() {
        if (mInstance == null) {
            mInstance = new ImageReaderThreadManager();
        }
        return mInstance;
    }

    public synchronized void addConnection(final StreamProfile streamProfile,
                           String address, final ImageStreamCallback callback) throws IOException {
        final LocalSocket localSocket = new LocalSocket();
        try {
            localSocket.connect(new LocalSocketAddress(address));
            final InputStream inputStream = localSocket.getInputStream();
            Thread workThread = new Thread("Image Transfer Thread, Type:" + streamProfile.getStreamType()){
                @Override
                public void run() {
                    int bufferLength = (int)(streamProfile.getPixelBytes() * streamProfile.getHeight() * streamProfile.getWidth());
                    ByteBuffer frameInfoBuffer = ByteBuffer.allocate(BaseFrameInfo.SIZE);
                    ByteBuffer imageBuffer = ByteBuffer.allocate(bufferLength);
                    try {
                        while (!isInterrupted()) {
                            try {
                                // read frame info
                                int cnt = 0;
                                frameInfoBuffer.rewind();
                                while (cnt < BaseFrameInfo.SIZE && !isInterrupted()) {
                                    int len = inputStream.read(frameInfoBuffer.array(), cnt, BaseFrameInfo.SIZE - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                // read image
                                cnt = 0;
                                imageBuffer.rewind();
                                while (cnt < bufferLength && !isInterrupted()) {
                                    int len = inputStream.read(imageBuffer.array(), cnt, bufferLength - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                BaseFrameInfo baseFrameInfo = BaseFrameInfo.fromByteBuffer(frameInfoBuffer);
                                callback.onNewImage(baseFrameInfo, imageBuffer);
                            } catch (IOException e) {
                                Timber.d("socket exception:" + e.getMessage());
                                break;
                            }
                        }
                    } finally {
                        try {
                            localSocket.close();
                        } catch (IOException e) {
                        }
                    }
                }
            };

            mWorkThreadMap.put(streamProfile.getStreamType(), workThread);

            workThread.start();
        } catch (IOException e) {
            Timber.e(e,"connect socket error!");
            throw e;
        }
    }

    public synchronized void addConnection(final StreamProfile streamProfile,
                                           String address, final FrameBuffer frameBuffer) throws IOException {
        final LocalSocket localSocket = new LocalSocket();
        try {
            localSocket.connect(new LocalSocketAddress(address));
            final InputStream inputStream = localSocket.getInputStream();
            Thread workThread = new Thread("Image Transfer Thread, Type:" + streamProfile.getStreamType()){
                @Override
                public void run() {
                    ByteBuffer frameInfoBuffer = ByteBuffer.allocate(BaseFrameInfo.SIZE);
                    int imageSize = (int) (streamProfile.getPixelBytes() * streamProfile.getWidth() * streamProfile.getHeight());
                    try {
                        while (!isInterrupted()) {
                            try {
                                // read frame info
                                int cnt = 0;
                                frameInfoBuffer.rewind();
                                while (cnt < BaseFrameInfo.SIZE && !isInterrupted()) {
                                    int len = inputStream.read(frameInfoBuffer.array(), cnt, BaseFrameInfo.SIZE - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }

                                BaseFrameInfo baseFrameInfo = BaseFrameInfo.fromByteBuffer(frameInfoBuffer);
                                RecyclableFrame recyclableFrame = RecyclableFrame.create(imageSize, baseFrameInfo);
                                // read image
                                ByteBuffer imageBuffer = recyclableFrame.getByteBuffer();
                                cnt = 0;
                                imageBuffer.rewind();
                                while (cnt < imageSize && !isInterrupted()) {
                                    int len = inputStream.read(imageBuffer.array(), cnt, imageSize - cnt);
                                    if (len == -1) {
                                        break;
                                    }
                                    cnt += len;
                                }
                                imageBuffer.rewind();
                                frameBuffer.add(recyclableFrame);
                            } catch (IOException e) {
                                Timber.d("socket exception:" + e.getMessage());
                                break;
                            }
                        }
                    } finally {
                        try {
                            localSocket.close();
                        } catch (IOException e) {
                        }
                    }
                }
            };

            mWorkThreadMap.put(streamProfile.getStreamType(), workThread);

            workThread.start();
        } catch (IOException e) {
            Timber.e(e,"connect socket error!");
            throw e;
        }
    }

    public synchronized void removeConnection(StreamProfile streamProfile) {
        Thread workThread = mWorkThreadMap.get(streamProfile.getStreamType());
        if (workThread != null) {
            workThread.interrupt();
            mWorkThreadMap.remove(streamProfile.getStreamType());
        }
    }

}
