package com.segway.robot.sdk.vision;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.ParcelFileDescriptor;
import android.os.RemoteException;
import android.util.SparseArray;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.view.SurfaceView;

import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;
import com.segway.robot.sdk.vision.IAprSenseService;
import com.segway.robot.sdk.vision.IAprSenseServiceCallback;
import com.segway.robot.sdk.vision.imu.IIMUDataCallback;
import com.segway.robot.sdk.vision.imu.IMUCallbackBundle;
import com.segway.robot.sdk.vision.imu.IMUDataCallback;
import com.segway.robot.sdk.vision.internal.ImageTransferType;
import com.segway.robot.sdk.vision.internal.VisionServiceState;
import com.segway.robot.sdk.vision.internal.frame.BaseFrameInfo;
import com.segway.robot.sdk.vision.internal.frame.DepthFrameInfo;
import com.segway.robot.sdk.vision.internal.frame.FishEyeFrameInfo;
import com.segway.robot.sdk.vision.internal.frame.FrameBuffer;
import com.segway.robot.sdk.vision.internal.socket.FrameClientThread;
import com.segway.robot.sdk.vision.internal.socket.ImageReaderThreadManager;
import com.segway.robot.sdk.vision.params.VisionParamID;
import com.segway.robot.sdk.vision.person.FindPersonsHandler;
import com.segway.robot.sdk.vision.person.Person;
import com.segway.robot.sdk.vision.person.PersonDetectCallback;
import com.segway.robot.sdk.vision.stream.FrameRate;
import com.segway.robot.sdk.vision.stream.PixelFormat;
import com.segway.robot.sdk.vision.stream.Resolution;
import com.segway.robot.sdk.vision.stream.StreamProfile;
import com.segway.robot.sdk.vision.stream.StreamType.VisionStreamType;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.util.ArrayList;
import java.util.List;

import timber.log.Timber;

/**
 * the entry of the vision service.
 */
public class VisionServiceManager {
    private static final String SERVICE_PACKAGE_NAME = "com.segway.robot.host.coreservice.vision";
    private static final String SERVICE_CLASS_NAME = "com.segway.robot.host.coreservice.vision.AprSenseService";
    private static final int SURFACE_HOLDER_CALLBACK_TAG_KEY = 0x29cad07c;
    private static final int SURFACE_HOLDER_PROFILE_TAG_KEY = 0x29cad07d;
    private static final int IMU_DATA_SIZE = 800;
    private static VisionServiceManager mInstance;
    private static Context mContext;
    private Object mDummy = new Object();
    private IAprSenseService mAprSenseAIDLService;
    private List<VisionServiceCallback> mVisionServiceCallbacks = new ArrayList<>();
    private List<AprSenseStateCallback> mAprSenseStateCallback = new ArrayList<>();
    private List<PersonDetectCallback> mPersonDetectCallback = new ArrayList<>();
    private List<IMUCallbackBundle> mIMUDataCallbacks = new ArrayList<>();
    private SparseArray<Object> mImageCallbackMap= new SparseArray<>();
    private SparseArray<FrameBuffer> mFrameBufferMap = new SparseArray<>();
    private SparseArray<FrameClientThread> mFrameTransferMap= new SparseArray<>();
    private SparseArray<SharedMemoryHolder> mFrameTransferSharedMemoryMap= new SparseArray<>();

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            // get remote service stub
            mAprSenseAIDLService = IAprSenseService.Stub.asInterface(service);
            try {
                // add internal AprSense Service Callback
                mAprSenseAIDLService.addCallback(mAprSenseServiceCallback);

                // add FindPersonHandler
                mAprSenseAIDLService.addFindPersonHandler(mFindPersonHandler);
            } catch (RemoteException e) {
                // notify error
                notifyRealSenseError(VisionServiceError.APR_SENSE_START_ERROR, "Cannot add the internal AprSense callback.");

                // disconnect to remote service
                mContext.unbindService(this);
                Timber.e("Connected to AprSenseService but cannot add the internal callback.", e);
                return;
            }

            // check version
            try {
                Version serviceVersion = mAprSenseAIDLService.getVersion();
                getVersion().check("VersionInfo", serviceVersion);
            } catch (RemoteException e) {
                // TODO: 2016/9/23
            } catch (VersionMismatchException e) {
                Timber.e(e, "Version mismatch");
                // TODO: 2016/9/23  new api can pass reason
//                notifyServiceDisconnected();
            }

            notifyServiceConnected();
            Timber.d("AprSenseService connected.");
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            mAprSenseAIDLService = null;
            notifyServiceDisconnected();
            mContext = null;
            synchronized (mFrameTransferMap) {
                for (int i = 0; i < mFrameTransferMap.size(); i++) {
                    FrameClientThread frameClientThread = mFrameTransferMap.valueAt(i);
                    frameClientThread.disconnect();
                }
                mFrameTransferMap.clear();
            }

            synchronized (mFrameTransferSharedMemoryMap) {
                for (int i = 0; i < mFrameTransferSharedMemoryMap.size(); i++) {
                    SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.valueAt(i);
                    sharedMemoryHolder.parcelFileDescriptor = null;
                    try {
                        sharedMemoryHolder.inputStream.close();
                    } catch (IOException e) {
                    }
                    sharedMemoryHolder.inputStream = null;

                }
                mFrameTransferSharedMemoryMap.clear();
            }

            synchronized (mImageCallbackMap) {
                mImageCallbackMap.clear();
            }

            Timber.d("AprSenseService disconnected.");
        }
    };

    /**
     * Get an instance of VisionServiceManager.
     * @return an instance of VisionServiceManager.
     */
    public static synchronized VisionServiceManager getInstance() {
        if (mInstance == null) {
            mInstance = new VisionServiceManager();
        }
        return mInstance;
    }

    /**
     * Add VisionServiceCallback. This callback receives remote service connection state changes.
     * @param visionServiceCallback a callback instance
     */
    public void addVisionServiceCallback(VisionServiceCallback visionServiceCallback) {
        if (visionServiceCallback == null) {
            return;
        }

        synchronized (mVisionServiceCallbacks) {
            if (!mVisionServiceCallbacks.contains(visionServiceCallback)) {
                mVisionServiceCallbacks.add(visionServiceCallback);
            }
        }
    }

    /**
     * Remove VisionServiceCallback.
     * @param visionServiceCallback  a callback instance.
     */
    public void removeVisionServiceCallback(VisionServiceCallback visionServiceCallback) {
        if (visionServiceCallback == null) {
            return;
        }

        synchronized (mVisionServiceCallbacks) {
                mVisionServiceCallbacks.remove(visionServiceCallback);
        }
    }

    /**
     * Add AprSenseStateCallback. This callback receives the AprSense state changes.
     * @param aprSenseStateCallback a callback instance.
     */
    public void addRealSenseStateCallback(AprSenseStateCallback aprSenseStateCallback) {
        if (aprSenseStateCallback == null) {
            return;
        }

        synchronized (mAprSenseStateCallback) {
            if (!mAprSenseStateCallback.contains(aprSenseStateCallback)) {
                mAprSenseStateCallback.add(aprSenseStateCallback);
            }
        }
    }

    /**
     * Remove AprSenseStateCallback.
     * @param aprSenseStateCallback a callback instance.
     */
    public void removeRealSenseStateCallback(AprSenseStateCallback aprSenseStateCallback) {
        if (aprSenseStateCallback == null) {
            return;
        }

        synchronized (mAprSenseStateCallback) {
            mAprSenseStateCallback.remove(aprSenseStateCallback);
        }
    }

    /**
     * Connect to the vision service.
     * @param context any Android Context.
     * @param visionServiceCallback the connection state callback.
     * @return true if the connection is successful.
     */
    public synchronized boolean connectService(Context context, VisionServiceCallback visionServiceCallback) {
        // check input parameter
        if (context == null) {
            throw new IllegalArgumentException("Context cannot be null!");
        }

        // check service already started
        if (mContext != null) {
            throw new VisionServiceException("The service is already connected.");
        }

        // save application context
        mContext = context.getApplicationContext();

        // save connection state callback
        addVisionServiceCallback(visionServiceCallback);

        // connect service
        Intent startServiceIntent = new Intent();
        startServiceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        startServiceIntent.putExtra("PackageNameFPC", context.getPackageName());
        return mContext.bindService(startServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
    }

    /**
     * Disconnect from the vision service.
     */
    public synchronized void disconnectService() {
        if (mContext!= null) {
            // unbind service
            mContext.unbindService(mServiceConnection);
            mContext = null;
        }
    }

    /**
     * Return the service connection state.
     * @return
     */
    public synchronized boolean isServiceConnected() {
        return mContext != null;
    }

    /**
     * Start AprSense
     * @param aprSenseStateCallback AprSenseStateCallback which can be null.
     */
    public synchronized void startAprSense(AprSenseStateCallback aprSenseStateCallback) {
        checkConnected();

        try {
            if (mAprSenseAIDLService.isAprSenseRunning()) {
                throw new VisionServiceException("AprSense is already started.");
            }
            mAprSenseAIDLService.startAprSense();
            addRealSenseStateCallback(aprSenseStateCallback);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop AprSense.
     */
    public synchronized void stopAprSense() {
        checkConnected();

        try {
            if (!mAprSenseAIDLService.isAprSenseRunning()) {
                throw new VisionServiceException("AprSense is not running");
            }
            mAprSenseAIDLService.stopAprSense();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Return the AprSense state.
     * @return true if AprSense is running.
     */
    public boolean isAprSenseRunning() {
        checkConnected();

        try {
            return mAprSenseAIDLService.isAprSenseRunning();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Set and enable the stream profile for AprSense. This Function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     * For the color camera and the depth camera, each can set only one stream profile at a time.
     * @param streamType the stream type, see the StreamType.
     * @param resolution the image resolution, see the VisionResolution.Color and VisionResolution.Depth.
     * @param fps the image frame rate, see the FrameRate.
     * @param pixelFormat the pixel format, see the PixelFormat.
     */
    public void enableStream(@VisionStreamType int streamType,
                             @Resolution.VisionResolution int resolution,
                             @FrameRate.VisionFrameRate int fps,
                             @PixelFormat.VisionPixelFormat int pixelFormat) throws VisionServiceException {
        checkConnected();

        try {
            mAprSenseAIDLService.enableStream(streamType, resolution, fps, pixelFormat);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }

    }

    /**
     * Set and enable the stream profile for AprSense. This Function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     * For the color camera and the depth camera, each can set only one stream profile at a time.
     * @param streamProfile see the StreamProfile.
     */
    public void enableStream(StreamProfile streamProfile) {
        checkConnected();

        int resolution = Resolution.toVisionServiceResolution(streamProfile.getWidth(), streamProfile.getHeight());
        if (resolution == 0) {
            throw new VisionServiceException("The width or height is not supported.");
        }

        try {
            mAprSenseAIDLService.enableStream(streamProfile.getStreamType(), resolution,
                    streamProfile.getFps(), streamProfile.getPixelFormat());
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Disable all the streams for AprSense. This function must be called before startAprSense.
     * If AprSense is already started, restart it to apply the changes.
     */
    public void cleanStreams() throws VisionServiceException {
        checkConnected();

        try {
            mAprSenseAIDLService.cleanStreams();
        } catch (RemoteException e) {
            throw  new VisionServiceException(e);
        }
    }

    /**
     * Get the activated stream profile for AprSense.
     * @return StreamProfile array.
     */
    public StreamProfile[] getActivatedStreamProfiles() throws VisionServiceException {
        checkConnected();

        try {
            return mAprSenseAIDLService.getActivatedStreamProfile();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start the selected stream preview. The image will be displayed on the input surface view directly.
     * The stream profile must be enabled before AprSense is started.
     * @param surfaceView the SurfaceView to display preview image.
     * @param streamProfile the StreamProfile to select stream that will be displayed.
     * @param showPerson If it is set to be true, the person rectangle will be drawn on the preview.
     */
    public void startPreview(final SurfaceView surfaceView, final StreamProfile streamProfile, final boolean showPerson) throws VisionServiceException {
        checkConnected();
        checkAprSenseStarted();

        SurfaceHolder surfaceHolder = surfaceView.getHolder();
        surfaceHolder.setFixedSize(streamProfile.getWidth(), streamProfile.getHeight());
        SurfaceHolder.Callback callback = new SurfaceHolder.Callback() {
            @Override
            public void surfaceCreated(SurfaceHolder holder) {
                Timber.i("Surface %d. surfaceCreated", surfaceView.getId());
                try {
                    preview(streamProfile.getStreamType(), holder.getSurface(), showPerson);
                } catch (VisionServiceException e) {
                    notifyRealSenseError(VisionServiceError.PREVIEW_ERROR,e.getMessage());
                }
            }

            @Override
            public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
                Timber.i("Surface %d. surfaceChanged format = %d width = %d height = %d",
                        surfaceView.getId(), format, width, height);
            }

            @Override
            public void surfaceDestroyed(SurfaceHolder holder) {
                Timber.i("Surface %d. surfaceDestroyed", surfaceView.getId());
            }
        };
        surfaceView.setTag(SURFACE_HOLDER_CALLBACK_TAG_KEY, callback);
        surfaceView.setTag(SURFACE_HOLDER_PROFILE_TAG_KEY, streamProfile);

        surfaceHolder.addCallback(callback);
    }

    /**
     * Stop the preview.
     * @param surfaceView the SurfaceView to stop the preview.
     */
    public void stopPreview(final SurfaceView surfaceView) {
        checkConnected();
        checkAprSenseStarted();

        SurfaceHolder surfaceHolder = surfaceView.getHolder();
        SurfaceHolder.Callback callback = (SurfaceHolder.Callback) surfaceView.getTag(SURFACE_HOLDER_CALLBACK_TAG_KEY);
        StreamProfile streamProfile = (StreamProfile) surfaceView.getTag(SURFACE_HOLDER_PROFILE_TAG_KEY);
        if (callback != null) {
            surfaceHolder.removeCallback(callback);
        }
        try {
            mAprSenseAIDLService.stopPreview(streamProfile.getStreamType());
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Start the selected image in a stream.
     * @param streamProfile the selected stream.
     * @param callback the image callback.
     */
    public void startImageStream(StreamProfile streamProfile, ImageStreamCallback callback) {
        checkConnected();

        if (streamProfile == null) {
            throw new IllegalArgumentException("The stream profile cannot be null.");
        }

        if (callback == null) {
            throw new IllegalArgumentException("The image stream callback cannot be null.");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamProfile.getStreamType()) != null) {
                throw new VisionServiceException("The stream is duplicated.");
            }
            mImageCallbackMap.put(streamProfile.getStreamType(), mDummy);
        }

        try {
            String address = mAprSenseAIDLService.setupImageTransferSocket(streamProfile.getStreamType(),
                    ImageTransferType.PUSH);
            ImageReaderThreadManager.getInstance().addConnection(streamProfile, address, callback);
        } catch (Exception e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * stream to sdk buffer
     * @param streamProfile
     */
    public void startImageStreamToBuffer(final StreamProfile streamProfile) {
        checkConnected();

        if (streamProfile == null) {
            throw new IllegalArgumentException("stream profile can't be null");
        }

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamProfile.getStreamType()) != null) {
                throw new VisionServiceException("stream duplicated");
            }
            mImageCallbackMap.put(streamProfile.getStreamType(), mDummy);
        }

        FrameBuffer frameBuffer;
        synchronized (mFrameBufferMap) {
            if (mFrameBufferMap.get(streamProfile.getStreamType()) == null) {
                mFrameBufferMap.put(streamProfile.getStreamType(), new FrameBuffer());
            }

            frameBuffer = mFrameBufferMap.get(streamProfile.getStreamType());
        }

        try {
            String address = mAprSenseAIDLService.setupImageTransferSocket(streamProfile.getStreamType(),
                    ImageTransferType.PUSH);
            ImageReaderThreadManager.getInstance().addConnection(streamProfile, address, frameBuffer);
        } catch (Exception e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop the image stream.
     * @param streamProfile the selected stream.
     */
    public void stopImageStream(StreamProfile streamProfile) {
        checkConnected();
        checkAprSenseStarted();

        synchronized (mImageCallbackMap) {
            if (mImageCallbackMap.get(streamProfile.getStreamType()) == null) {
                throw new VisionServiceException("The stream is not started " + streamProfile.getStreamType());
            }
            mImageCallbackMap.remove(streamProfile.getStreamType());
        }

        synchronized (mFrameBufferMap) {
            mFrameBufferMap.delete(streamProfile.getStreamType());
        }

        try {
            ImageReaderThreadManager.getInstance().removeConnection(streamProfile);
            mAprSenseAIDLService.stopImageTransfer(streamProfile.getStreamType(),
                    ImageTransferType.PUSH);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get latest frame with tid larger than previousTid
     * @param streamProfile stream profile
     * @param previousTid tid of the previous frame
     * @return Frame object for success or null for no new frame
     */
    public Frame getLatestFrameForStream(StreamProfile streamProfile, long previousTid) {
        FrameBuffer frameBuffer;
        if (streamProfile == null) {
            throw new IllegalArgumentException("stream profile can't be null");
        }

        synchronized (mFrameBufferMap) {
            frameBuffer = mFrameBufferMap.get(streamProfile.getStreamType());
        }

        if (frameBuffer == null) {
            throw new IllegalArgumentException("stream image transfer not initialized");
        }

        return frameBuffer.getLatest(previousTid);
    }

    /**
     * Return frame got from getLatestFrameForStream to buffer
     * This is very important for the buffer queue is limited
     * @param streamProfile stream profile
     * @param frame frame to return
     */
    public void returnFrameToStream(StreamProfile streamProfile, Frame frame) {
        FrameBuffer frameBuffer;
        if (streamProfile == null) {
            throw new IllegalArgumentException("stream profile can't be null");
        }

        if (frame == null) {
            throw new IllegalArgumentException("can not return null frame");
        }

        synchronized (mFrameBufferMap) {
            frameBuffer = mFrameBufferMap.get(streamProfile.getStreamType());
        }

        if (frameBuffer == null) {
            throw new IllegalArgumentException("stream image transfer not initialized");
        }

        frameBuffer.returnFrame(frame);
    }

    /**
     * Enable the frame transfer by StreamProfile.
     * @param profile the selected stream.
     */
    public void enableFrameTransfer(StreamProfile profile) {
        checkConnected();
        synchronized (mFrameTransferMap) {
            if (mFrameTransferMap.get(profile.getStreamType()) != null) {
                throw new VisionServiceException("The frame profile is duplicated.");
            }
        }

        String address;
        try {
            address = mAprSenseAIDLService.setupImageTransferSocket(profile.getStreamType(), ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException("An error occurs in seting up the image transfer socket.", e);
        }

        FrameClientThread frameClientThread =  new FrameClientThread(profile);
        frameClientThread.connectServer(address);
        // TODO: connect failed
        synchronized (mFrameTransferMap) {
            mFrameTransferMap.put(profile.getStreamType(), frameClientThread);
        }
    }

    /**
     * Disable the frame transfer by StreamProfile.
     * @param profile the selected stream.
     */
    public void disableFrameTransfer(StreamProfile profile) {
        checkConnected();
        synchronized (mFrameTransferMap) {
            FrameClientThread frameClientThread = mFrameTransferMap.get(profile.getStreamType());
            if(frameClientThread != null) {
                frameClientThread.disconnect();
                mFrameTransferMap.remove(profile.getStreamType());
            }
        }

        try {
            mAprSenseAIDLService.stopImageTransfer(profile.getStreamType(),
                    ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Enable frame transfer by StreamProfile
     * @param profile Stream selected
     */
    public void enableFrameTransferMemoryFile(StreamProfile profile) {
        checkConnected();
        synchronized (mFrameTransferSharedMemoryMap) {
            if (mFrameTransferSharedMemoryMap.get(profile.getStreamType()) != null) {
                throw new VisionServiceException("frame profile duplicated");
            }
        }

        try {
            ParcelFileDescriptor parcelFileDescriptor =
                    mAprSenseAIDLService.setupImageTransferSharedMemory(
                            profile.getStreamType(), ImageTransferType.PULL);
            FileInputStream fileInputStream = new FileInputStream(parcelFileDescriptor.getFileDescriptor());
            SharedMemoryHolder sharedMemoryHolder = new SharedMemoryHolder();
            sharedMemoryHolder.parcelFileDescriptor = parcelFileDescriptor;
            sharedMemoryHolder.inputStream = fileInputStream;
            synchronized (mFrameTransferSharedMemoryMap) {
                mFrameTransferSharedMemoryMap.put(profile.getStreamType(), sharedMemoryHolder);
            }
        } catch (RemoteException e) {
            throw new VisionServiceException("setup image transfer socket error", e);
        }
    }

    /**
     * Disable frame transfer by StreamProfile
     * @param profile Stream selected
     */
    public void disableFrameTransferMemoryFile(StreamProfile profile) {
        checkConnected();
        synchronized (mFrameTransferSharedMemoryMap) {
            SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.get(profile.getStreamType());
            if(sharedMemoryHolder != null) {
                sharedMemoryHolder.parcelFileDescriptor = null;
                try {
                    sharedMemoryHolder.inputStream.close();
                } catch (IOException e) {
                }
                sharedMemoryHolder.inputStream = null;
                mFrameTransferSharedMemoryMap.remove(profile.getStreamType());
            }
        }

        // TODO: 16/6/23
        try {
            mAprSenseAIDLService.stopImageTransfer(profile.getStreamType(),
                    ImageTransferType.PULL);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get the fish eye frame.
     * @param streamProfile the selected stream which must be enabled by enableFrameTransfer.
     * @param buffer the image data to fill in the buffer.
     * @return the frame information.
     */
    public FishEyeFrameInfo getFishEyeFrame(StreamProfile streamProfile, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            throw new IllegalArgumentException("The buffer is null.");
        }

        BaseFrameInfo baseFrameInfo = getFrame(streamProfile, buffer);
        if (baseFrameInfo != null) {
            return new FishEyeFrameInfo(baseFrameInfo);
        }

        return null;
    }

    /**
     * Get the depth frame.
     * @param streamProfile the selected frame which must be enabled by enableFrameTransfer.
     * @param buffer the image data to fill in the buffer.
     * @return the frame information.
     */
    public DepthFrameInfo getDepthFrame(StreamProfile streamProfile, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            throw new IllegalArgumentException("The buffer is null.");
        }

        BaseFrameInfo baseFrameInfo = getFrame(streamProfile, buffer);
        if (baseFrameInfo != null) {
            return new DepthFrameInfo(baseFrameInfo);
        }

        return null;
    }

    /**
     * Get the color frame which is temporary solution.
     * @param streamProfile the selected frame which must be enabled by enableFrameTransfer.
     * @param buffer the image data to fill in the buffer.
     * @return the frame information.
     */
    @Deprecated
    public BaseFrameInfo getColorFrame(StreamProfile streamProfile, ByteBuffer buffer) {
        checkConnected();
        checkAprSenseStarted();

        if (buffer == null) {
            throw new IllegalArgumentException("The buffer is null.");
        }

        return getFrame(streamProfile, buffer);
    }

    public BaseFrameInfo getFrameSharedMemory(StreamProfile profile, ByteBuffer buffer) {
        SharedMemoryHolder sharedMemoryHolder = mFrameTransferSharedMemoryMap.get(profile.getStreamType());
        if (sharedMemoryHolder == null) {
            throw new VisionServiceException("stream " + profile.getStreamType() + " not enabled");
        }

        try {
            BaseFrameInfo baseFrameInfo = mAprSenseAIDLService.getFrameMemoryFile(profile.getStreamType(),
                    profile.getResolution(), profile.getPixelFormat());

            if (baseFrameInfo == null) {
                return null;
            }

            try {
                buffer.rewind();
                FileInputStream inputStream = sharedMemoryHolder.inputStream;
                int cnt = inputStream.getChannel().position(0).read(buffer);
            } catch (IOException e) {
                throw new VisionServiceException(e);
            }

            return baseFrameInfo;
        } catch (RemoteException e) {
            // TODO: dell with connect failed
            throw new VisionServiceException("get Frame failed", e);
        }
    }

    private BaseFrameInfo getFrame(StreamProfile profile, ByteBuffer buffer) {
        BaseFrameInfo baseFrameInfo;
        FrameClientThread frameClientThread;
        synchronized (mFrameTransferMap) {
            frameClientThread = mFrameTransferMap.get(profile.getStreamType());
        }

        if (frameClientThread == null) {
            throw new VisionServiceException("frame transfer for stream " + profile.getStreamType()
                    + " not started");
        }

        try {
            baseFrameInfo = mAprSenseAIDLService.getFrame(profile.getStreamType(),
                    profile.getResolution(), profile.getPixelFormat());
        } catch (RemoteException e) {
            // TODO: dell with connect failed
            throw new VisionServiceException("get Frame failed", e);
        }

        if (baseFrameInfo == null) {
            return null;
        }

        int length = (int)(PixelFormat.getPixelBytes(profile.getPixelFormat())
                * profile.getHeight() * profile.getWidth());
        frameClientThread.readFrame(buffer, length);

        // wait for transfer complete
        synchronized (frameClientThread) {
            while (!frameClientThread.isTransferEnded()) {
                try {
                    frameClientThread.wait();
                } catch (InterruptedException e) {
                }
            }
        }

        if (!frameClientThread.isReadSuccess()) {
            Timber.e("read frame not success, check log for details");
            throw new VisionServiceException("read frame failed");
        }
        return baseFrameInfo;
    }

    /**
     * Start person detection.
     * @param callback the detected person callback which can be null.
     */
    public void startPersonDetection(PersonDetectCallback callback) {
        checkConnected();

        try {
            mAprSenseAIDLService.startPersonDetection();
            if (callback != null) {
                mPersonDetectCallback.add(callback);
            }
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Stop person detection and clean all the callbacks.
     */
    public void stopPersonDetection() {
        checkConnected();
        checkAprSenseStarted();

        mPersonDetectCallback.clear();
        try {
            mAprSenseAIDLService.stopPersonDetection();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }


    /**
     * Start tracking on a selected person.
     * @param personId the person ID which is got from the person detection callback.
     */
    public void startPersonTracking(int personId) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.startPersonTracking(personId);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }


    /**
     * Stop tracking person.
     * @param personId the person ID which is got from the person detection callback.
     */
    public void stopPersonTracking(int personId) {
        checkConnected();
        checkAprSenseStarted();

        try {
            mAprSenseAIDLService.stopPersonTracking(personId);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Return person find now.
     * @return
     */
    public List<Person> findPersons() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.findPersons();
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Load the person database.
     * @param databasePath the person database path.
     */
    // TODO: not tested
    public void loadPersonDatabase(String databasePath) {
        checkConnected();

        try {
            mAprSenseAIDLService.loadRecognitionDatabase(databasePath);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Save the person database.
     * @param databasePath the person database path.
     */
    // TODO: not tested
    public void savePersonDatabase(String databasePath) {
        checkConnected();

        try {
            mAprSenseAIDLService.saveRecognitionDatabase(databasePath);
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }
    }

    /**
     * Get the AprSense timestamp.
     * @return the 64bit timestamp in ms.
     */
    public long getAprSenseTimeStamp() {
        checkConnected();
        checkAprSenseStarted();

        try {
            return mAprSenseAIDLService.getAprSenseTimeStamp();
        } catch (RemoteException e) {
            throw new VisionServiceException("An exception occurs in getting the AprSense timestamp! ", e);
        }
    }

    /**
     * Get Depth Focal Length X
     * @return float parameter
     */
    public float getDepthFocalLengthX() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_FOCAL_LENGTH_X);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Get Depth Focal Length Y
     * @return float parameter
     */
    public float getDepthFocalLengthY() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_FOCAL_LENGTH_Y);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }


    /**
     * Get Depth Principal Point X
     * @return float parameter
     */
    public float getDepthPrincipalPointX() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_PRINCIPAL_POINT_X);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Get Depth Principal Point Y
     * @return float parameter
     */
    public float getDepthPrincipalPointY() {
        try {
            return mAprSenseAIDLService.getFloatParam(VisionParamID.DEPTH_PRINCIPAL_POINT_Y);
        } catch (RemoteException e) {
            throw new VisionServiceException(e.getMessage(), e);
        }
    }

    /**
     * Add the IMU data callback.
     * @param callback
     */
    public void addIMUDataCallback(IMUDataCallback callback) {
        checkConnected();

        synchronized (mIMUDataCallbacks) {
            for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                if (bundle.callback == callback) {
                    return;
                }
            }

            IMUCallbackBundle bundle = new IMUCallbackBundle();
            bundle.callback = callback;
            // TODO: fixed size
            bundle.buffer = ByteBuffer.allocateDirect(IMU_DATA_SIZE);

            mIMUDataCallbacks.add(bundle);
            if (mIMUDataCallbacks.size() == 1) {
                try {
                    mAprSenseAIDLService.setIMUDataCallback(mIImuDataCallback);
                } catch (RemoteException e) {
                    throw new VisionServiceException("add IMU Data Callback exception! ", e);
                }
            }
        }
    }

    /**
     * Remove the IMU data callback.
     * @param callback
     */
    public void removeIMUDataCallback(IMUDataCallback callback) {
        checkConnected();

        synchronized (mIMUDataCallbacks) {
            IMUCallbackBundle removeBundle = null;
            for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                mIMUDataCallbacks.remove(callback);
                if (bundle.callback == callback) {
                    removeBundle = bundle;
                    break;
                }
            }

            if (removeBundle != null) {
                // TODO: release byte buffer
                mIMUDataCallbacks.remove(removeBundle);
            }

            if (mIMUDataCallbacks.size() == 0) {
                try {
                    mAprSenseAIDLService.removeIMUDataCallback();
                } catch (RemoteException e) {
                    throw new VisionServiceException("An exception occurs in removing the IMU Data callback! ", e);
                }
            }
        }
    }

    public byte[] getCalibrationData() {
        checkConnected();

        try {
            return mAprSenseAIDLService.getCalibrationData();
        } catch (RemoteException e) {
            throw new VisionServiceException("Get calibration data failed");
        }
    }

    private void checkConnected() throws VisionServiceException {
        if (mAprSenseAIDLService == null) {
            throw VisionServiceException.getServiceNotConnectedException();
        }
    }

    private void checkAprSenseStarted() throws VisionServiceException {
        try {
            boolean started = mAprSenseAIDLService.isAprSenseRunning();
            if (!started) {
                throw VisionServiceException.getAprSenseNotStaredException();
            }
        } catch (RemoteException e) {
            throw new VisionServiceException(e);
        }

    }

    private void preview(int streamType, Surface surface, boolean drawPerson) throws VisionServiceException{
        try {
            mAprSenseAIDLService.startPreview(streamType, surface, drawPerson);
        } catch (RemoteException e) {
            throw VisionServiceException.toVisionServiceException(e);
        }
    }

    IAprSenseServiceCallback.Stub mAprSenseServiceCallback = new IAprSenseServiceCallback.Stub() {
        @Override
        public void onAprStateChanged(int state, int errorCode, String errorMessage) throws RemoteException {
            switch (state) {
                case VisionServiceState.APR_SENSE_STARTED:
                    notifyAprSenseStarted();
                    break;
                case VisionServiceState.APR_SENSE_STOPPED:
                    notifyRealSenseStopped();
                    break;
                case VisionServiceState.APR_SENSE_ERROR:
                    notifyRealSenseError(errorCode,errorMessage);
                    break;
            }
        }
    };

    FindPersonsHandler.Stub mFindPersonHandler = new FindPersonsHandler.Stub() {
        @Override
        public void onFindPersons(List<Person> persons) throws RemoteException {
            for (PersonDetectCallback personDetectCallback : mPersonDetectCallback) {
                personDetectCallback.onPersonDetected(persons);
            }
        }
    };

    IIMUDataCallback.Stub mIImuDataCallback = new IIMUDataCallback.Stub() {
        @Override
        public void onNewData(byte[] buff, int length, int frameCount) throws RemoteException {
            synchronized (mIMUDataCallbacks) {
                for (IMUCallbackBundle bundle : mIMUDataCallbacks) {
                    ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(buff);
                    ReadableByteChannel readableByteChannel = Channels.newChannel(byteArrayInputStream);
                    bundle.buffer.rewind();
                    try {
                        readableByteChannel.read(bundle.buffer);
                    } catch (IOException e) {
                        // TODO:
                        Timber.e(e,"An error occurs in copying the IMU data.");
                    }
                    bundle.callback.onNewData(bundle.buffer, length, frameCount);
                }
            }
        }
    };

    private void notifyServiceConnected() {
        synchronized (mVisionServiceCallbacks) {
            for (VisionServiceCallback visionServiceCallback : mVisionServiceCallbacks) {
                visionServiceCallback.onServiceConnected();
            }
        }
    }

    private void notifyServiceDisconnected() {
        synchronized (mVisionServiceCallbacks) {
            for (VisionServiceCallback visionServiceCallback : mVisionServiceCallbacks) {
                visionServiceCallback.onServiceDisconnected();
            }
            mVisionServiceCallbacks.clear();
        }
    }

    private void notifyAprSenseStarted() {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseStarted();
            }
        }
    }

    private void notifyRealSenseStopped() {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseStopped();
            }
        }
    }

    private void notifyRealSenseError(int errorCode, String errorMsg) {
        synchronized (mAprSenseStateCallback) {
            for (AprSenseStateCallback aprSenseStateCallback : mAprSenseStateCallback) {
                aprSenseStateCallback.onAprSenseError(errorCode, errorMsg);
            }
        }
    }

    private class SharedMemoryHolder {
        ParcelFileDescriptor parcelFileDescriptor;
        FileInputStream inputStream;
    }

    public Version getVersion() {
        return new Version(VersionInfo.version_channel,
                VersionInfo.version_name,
                VersionInfo.version_code,
                VersionInfo.version_min);
    }
}
