package com.segway.robot.sdk.voice;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;

import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;
import com.segway.robot.sdk.voice.tts.ITtsListener;
import com.segway.robot.sdk.voice.tts.TtsListener;

import java.util.concurrent.atomic.AtomicBoolean;

class SpeakerManager implements ServiceBinder {
    private static final String TAG = SpeakerManager.class.getSimpleName();
    private static final String SPEAKER_SERVICE_PACKAGE_NAME = "com.segway.robot.coreservice.voiceservice";
    private static final String SPEAKER_SERVICE_CLASS_NAME = "com.segway.robot.coreservice.voiceservice.TextToSpeechService";

    private BindStateListener mListener;
    ISpeaker mISpeaker;
    private Context mContext;
    private TtsListener mTtsListener;
    private boolean isBind = false;

    private boolean isSpeaking;
    private final AtomicBoolean mSpeakFinishLock = new AtomicBoolean(true);

    SpeakerManager() {

    }

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            Logger.d(TAG, "The service connected");
            mISpeaker = ISpeaker.Stub.asInterface(service);
            if (mISpeaker != null) {
                mListener.onBind();
            }

            // check version
            try {
                Version serviceVersion = mISpeaker.getVersion();
                getVersion().check("VersionInfo", serviceVersion);
            } catch (RemoteException e) {
                String error = "Cannot get Speaker Service version, err = " + e.getMessage();

                // disconnect to remote service
                mListener.onUnbind(error);
                mISpeaker = null;
                Logger.e(TAG, error, e);
            } catch (VersionMismatchException e) {
                String error = "Version mismatch: " + e.getMessage();

                // disconnect to remote service
                mListener.onUnbind(error);
                mISpeaker = null;
                Logger.e(TAG, "Version mismatch", e);

                throw e;
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            Logger.d(TAG, "The service disconnected");
            isBind = false;
            mISpeaker = null;
            mListener.onUnbind("The service disconnected.");
        }
    };

    private ITtsListener.Stub mITtsListener = new ITtsListener.Stub() {
        @Override
        public void onSpeechStarted(String word) throws RemoteException {
            mTtsListener.onSpeechStarted(word);
        }

        @Override
        public void onSpeechFinished(String word) throws RemoteException {
            isSpeaking = false;
            mTtsListener.onSpeechFinished(word);
            synchronized (mSpeakFinishLock) {
                mSpeakFinishLock.set(true);
                mSpeakFinishLock.notify();
            }
        }

        @Override
        public void onSpeechError(String word, String reason) throws RemoteException {
            isSpeaking = false;
            mTtsListener.onSpeechError(word, reason);
            synchronized (mSpeakFinishLock) {
                mSpeakFinishLock.set(true);
                mSpeakFinishLock.notify();
            }
        }
    };

    @Override
    public synchronized boolean bindService(Context context, BindStateListener listener) {
        if (isBind) {
            return true;
        }
        if (context == null) {
            throw new IllegalStateException("The context can not be null!");
        }
        if (listener == null) {
            throw new IllegalStateException("The listener can not be null!");
        }
        mContext = context.getApplicationContext();
        mListener = listener;
        Intent voiceServiceIntent = new Intent();
        voiceServiceIntent.setClassName(SPEAKER_SERVICE_PACKAGE_NAME, SPEAKER_SERVICE_CLASS_NAME);
        mContext.startService(voiceServiceIntent);
        Logger.d(TAG, "try to connect to the speaker service");
        isBind = mContext.bindService(voiceServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
        return isBind;
    }

    @Override
    public synchronized void unbindService() {
        Logger.d(TAG, "unbindService");
        if(!isBind) {
            return;
        }
        if (mContext != null) {
            mContext.unbindService(mServiceConnection);
        }
        isBind = false;
        mISpeaker = null;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mListener;
    }

    /**
     * Get current using language
     */
    @Languages.Language
    int getLanguage() throws RemoteException, VoiceException {
        if (mISpeaker == null) {
            throw new VoiceException("The tts service disconnected!");
        }
        int language = mISpeaker.getLanguage();
        if (language == 0) {
            return Languages.EN_US;
        } else {
            return Languages.ZH_CN;
        }
    }

    /**
     * Make the text to speech.
     * @param text the speech content.
     * @param ttsListener A {@link TtsListener} which provides callbacks for certain text-to-speech (TTS) generation events.
     * @throws VoiceException
     * @throws RemoteException
     */
    public synchronized void speak(String text, TtsListener ttsListener) throws VoiceException, RemoteException {
        if (mISpeaker == null) {
            throw new VoiceException("The speaker service is disconnected!");
        }
        if (ttsListener == null) {
            throw new IllegalArgumentException("The TtsListener cannot be null!");
        }
        if (isSpeaking) {
            throw new VoiceException("The speaker is speaking!");
        }
        isSpeaking = true;
        mTtsListener = ttsListener;
        mISpeaker.speak(text, mITtsListener);
        synchronized (mSpeakFinishLock) {
            mSpeakFinishLock.set(false);
        }
    }

    /**
     * Stop the speech.
     * @throws RemoteException
     * @throws VoiceException
     */
    public void stopSpeak() throws RemoteException, VoiceException {
        if (mISpeaker == null) {
            throw new VoiceException("The speaker service disconnected!");
        }
        mISpeaker.stopSpeak();
    }

    /**
     * Wait until the speech finishes.
     * @param timeout the timeout in millisecond.
     * @return true for timeout, false for else.
     * @throws RemoteException
     */
    public boolean waitForSpeakFinish(long timeout) throws RemoteException {
        synchronized (mSpeakFinishLock) {
            long time = System.currentTimeMillis();
            while (!mSpeakFinishLock.get()) {
                try {
                    mSpeakFinishLock.wait(timeout);
                } catch (InterruptedException e) {
                }
            }

            // TODO: 2016/9/7 check time out return value
            return System.currentTimeMillis() - time > timeout;
        }
    }

    public Version getVersion() {
        return new Version(VersionInfo.version_channel,
                VersionInfo.version_name,
                VersionInfo.version_code,
                VersionInfo.version_min);
    }
}
