package com.segway.robot.sdk.voice;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.Binder;
import android.os.IBinder;
import android.os.RemoteException;

import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.voice.audiodata.IRawDataListener;
import com.segway.robot.sdk.voice.audiodata.RawDataListener;
import com.segway.robot.sdk.voice.grammar.GrammarCheckUtil;
import com.segway.robot.sdk.voice.grammar.GrammarConstraint;
import com.segway.robot.sdk.voice.grammar.Slot;
import com.segway.robot.sdk.voice.recognition.IRecognitionListener;
import com.segway.robot.sdk.voice.recognition.IWakeupListener;
import com.segway.robot.sdk.voice.recognition.RecognitionListener;
import com.segway.robot.sdk.voice.recognition.RecognitionResult;
import com.segway.robot.sdk.voice.recognition.WakeupListener;
import com.segway.robot.sdk.voice.recognition.WakeupResult;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by ark338 on 16/9/1.
 */
class RecognizerManager implements ServiceBinder {
    private static final String TAG = RecognizerManager.class.getSimpleName();
    private static final String RECOGNIZER_SERVICE_PACKAGE_NAME = "com.segway.robot.coreservice.voiceservice";
    private static final String RECOGNIZER_SERVICE_CLASS_NAME = "com.segway.robot.coreservice.voiceservice.RecognitionService";
    /**
     * Record mode parameters
     */
    final int BIT_RATE = 16;
    final int SAMPLE_RATE = 16000;
    IRecognizer mIRecognizer;
    private Context mContext;
    private BindStateListener mListener;
    private boolean isBind = false;
    private WakeupListener mWakeupListener;
    private RecognitionListener mRecognitionListener;
    private RawDataListener mRawDataListener;
    private final Object mLockWakeupListener;
    private final Object mLockRecognitionListener;

    // TODO: 16/9/1 singleton
    RecognizerManager() {
        mLockWakeupListener = new Object();
        mLockRecognitionListener = new Object();
    }

    private IWakeupListener.Stub mIWakeupListener = new IWakeupListener.Stub() {
        @Override
        public void onStandby() {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onStandby();
                }
            }
        }

        @Override
        public void onWakeupResult(WakeupResult wakeupResult) {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onWakeupResult(wakeupResult);
                }
            }
        }

        @Override
        public void onWakeupError(String error) {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onWakeupError(error);
                }
            }
        }
    };

    private IRecognitionListener.Stub mIRecognitionListener = new IRecognitionListener.Stub() {
        @Override
        public void onRecognitionStart() {
            synchronized (mLockRecognitionListener) {
                if (mRecognitionListener != null) {
                    mRecognitionListener.onRecognitionStart();
                }
            }
        }

        @Override
        public boolean onRecognitionResult(RecognitionResult recognitionResult) {
            boolean doRecognition;
            synchronized (mLockRecognitionListener) {
                doRecognition = mRecognitionListener != null && mRecognitionListener.onRecognitionResult(recognitionResult);
            }
            return doRecognition;
        }

        @Override
        public boolean onRecognitionError(String error) {
            boolean doRecognition;
            synchronized (mLockRecognitionListener) {
                doRecognition = mRecognitionListener != null && mRecognitionListener.onRecognitionError(error);
            }
            return doRecognition;
        }
    };

    private IRawDataListener.Stub mIRawDataListener = new IRawDataListener.Stub() {
        @Override
        public void onRawData(byte[] data, int dataLength) {
            mRawDataListener.onRawData(data, dataLength);
        }
    };

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            Logger.i(TAG, "service connected");
            mIRecognizer = IRecognizer.Stub.asInterface(service);

            afterOnServiceConnected();
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            Logger.i(TAG, "service disconnected");
            isBind = false;
            mListener.onUnbind("The recognition service disconnected!");
            mIRecognizer = null;
        }
    };

    protected void afterOnServiceConnected() {
        // check version
//        try {
//            Version serviceVersion = mIRecognizer.getVersion();
//            getVersion().check("VersionInfo", serviceVersion);
//        } catch (RemoteException e) {
//            String error = "Cannot get Recognition Service version, err = " + e.getMessage();
//            // disconnect to remote service
//            mListener.onUnbind(error);
//            mIRecognizer = null;
//            unbindService();
//            Logger.e(TAG, error, e);
//            return;
//        } catch (VersionMismatchException e) {
//            String error = "Version mismatch: " + e.getMessage();
//
//            // disconnect to remote service
//            mListener.onUnbind(error);
//            mIRecognizer = null;
//            unbindService();
//            Logger.e(TAG, "Version mismatch", e);
//            return;
//        }

        if (mIRecognizer != null) {
            try {
                mIRecognizer.registerWatcher(new Binder(), mContext.getPackageName());
            } catch (RemoteException e) {
                Logger.e(TAG, "onServiceConnected: ", e);
                mListener.onUnbind(e.getMessage());
                unbindService();
                return;
            }
            mListener.onBind();
        }
    }

    @Override
    public synchronized boolean bindService(Context context, BindStateListener listener) {
        Logger.i(TAG, "RecognizerManager bindService");
        if (isBind) {
            return true;
        }
        if (context == null) {
            throw new IllegalArgumentException("The context cannot be null!");
        }
        if (listener == null) {
            throw new IllegalArgumentException("The BindStateListener cannot be null!");
        }
        mContext = context.getApplicationContext();
        mListener = listener;
        Intent voiceServiceIntent = new Intent();
        voiceServiceIntent.setClassName(RECOGNIZER_SERVICE_PACKAGE_NAME, RECOGNIZER_SERVICE_CLASS_NAME);
        mContext.startService(voiceServiceIntent);
        Logger.i(TAG, "try to connect to the recognition service");
        isBind = mContext.bindService(voiceServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
        return isBind;
    }

    @Override
    public synchronized void unbindService() {
        Logger.i(TAG, "unbindService");
        Logger.i(TAG, "isBind: " + isBind);
        if (!isBind) {
            return;
        }
        if (mContext != null) {
            if (mIRecognizer != null) {
                try {
                    mIRecognizer.unregisterWatcher(mContext.getPackageName());
                } catch (RemoteException e) {
                    Logger.e(TAG, "RemoteException: ", e);
                }
            }
            mContext.unbindService(mServiceConnection);
        }
        isBind = false;
        mIRecognizer = null;
    }

    @Override
    public boolean isBind() {
        return isBind;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mListener;
    }

    /**
     * Get current using language
     */
    @Languages.Language
    int getLanguage() throws VoiceException {
        checkConnectionStatus();
        int language = 0;
        try {
            language = mIRecognizer.getLanguage();
        } catch (RemoteException e) {
            throw new VoiceException("getLanguage", e);
        }
        if (language == 0) {
            return Languages.EN_US;
        } else {
            return Languages.ZH_CN;
        }
    }

    /**
     * Set to enable/disable beam forming
     * In current version, beam forming direction is fixed to the front of robot head
     * Beam forming is not effective in wakeup stage
     *
     * @param enable
     */
    void beamForming(boolean enable) throws VoiceException {
        checkConnectionStatus();
        try {
            mIRecognizer.beamForming(enable);
        } catch (RemoteException e) {
            throw new VoiceException("beamForming", e);
        }
    }

    /**
     * Create GrammarConstraint from json file
     * <p>
     * <p>
     * sample
     {
         "name": "play_music",
         "slotList": [
             {
                 "name": "play_cmd",
                 "isOptional": false,
                 "word": [
                     "play",
                     "play music"
                 ]
             },
             {
                 "name": "song",
                 "isOptional": false,
                 "word": [
                     "happy birthday",
                     "jingle bells"
                 ]
             }
         ]
     }
     *
     * @param json file describe GrammarConstraint
     * @return
     */
    GrammarConstraint createGrammarConstraint(String json) throws VoiceException {
        // TODO: 2016/9/30 verify
        GrammarConstraint grammarFromJson;
        try {
            JSONObject grammarConstraintObject = new JSONObject(json);
            List<Slot> slotList = new ArrayList<>();
            String name = grammarConstraintObject.getString("name");
            JSONArray slotListFromJson = grammarConstraintObject.getJSONArray("slotList");
            int length = slotListFromJson.length();
            for (int i = 0; i < length; i++) {
                JSONObject slotObject = slotListFromJson.getJSONObject(i);
                Slot slot = new Slot(slotObject.getString("name"));
                slot.setOptional(slotObject.getBoolean("isOptional"));
                JSONArray wordListFromJson = slotObject.getJSONArray("word");
                int wordLength = wordListFromJson.length();
                String[] words = new String[wordLength];
                for (int j = 0; j < wordLength; j++) {
                    words[j] = wordListFromJson.getString(j);
                }
                slot.addWordsArray(words);
                slotList.add(slot);
            }
            grammarFromJson = new GrammarConstraint(name, slotList);
        } catch (JSONException e) {
            throw new VoiceException("Json illegal, exception: ", e);
        }
        if (GrammarCheckUtil.containSystemGrammar(grammarFromJson)) {
            throw new VoiceException("This GrammarConstraint(" + grammarFromJson.getName() + ") contains one or more system grammars.");
        }
        return grammarFromJson;
    }

    /**
     * Add GrammarConstraint
     *
     * @param grammarConstraint
     */
    void addGrammarConstraint(GrammarConstraint grammarConstraint) throws VoiceException {
        checkConnectionStatus();
        if (grammarConstraint.getName() == null || grammarConstraint.getName().equals("")) {
            throw new IllegalArgumentException("Your grammar constraint has no name!");
        }
        int language = getLanguage();
        Logger.i(TAG, "RecognitionLanguage: " + language);
        List<Slot> slotList = grammarConstraint.getSlotList();
        int size = slotList.size();
        if (size == 0) {
            throw new IllegalArgumentException("Your grammar constraint has no slot!");
        }
        boolean b = true;
        for (Slot slot : slotList) {
            if (language == Languages.EN_US) {
                b = CheckUtil.isEnSlot(slot);
            } else {
                b = CheckUtil.isCnSlot(slot);
            }
            if (!b) {
                Logger.e(TAG, "\"" + slot.getName() + "\" is an illegal slot");
                throw new VoiceException("\"" + slot.getName() + "\" is an illegal slot");
            }
        }
        /*if (!b) {
            throw new IllegalStateException("The GrammarConstraint contains one or more illegal slots");
        }*/
        if (GrammarCheckUtil.containSystemGrammar(grammarConstraint)) {
            throw new VoiceException("This GrammarConstraint(" + grammarConstraint.getName() + ") contains one or more system grammars.");
        }
        try {
            mIRecognizer.addGrammarConstraint(grammarConstraint);
        } catch (RemoteException e) {
            throw new VoiceException("addGrammarConstraint", e);
        }
    }

    /**
     * Remove GrammarConstraint
     *
     * @param grammarConstraint
     */
    void removeGrammarConstraint(GrammarConstraint grammarConstraint) throws VoiceException {
        checkConnectionStatus();
        try {
            mIRecognizer.removeGrammarConstraint(grammarConstraint);
        } catch (RemoteException e) {
            throw new VoiceException("removeGrammarConstraint", e);
        }
    }

    /**
     * Start Wakeup and Recognition
     *
     * @param wakeUpListener      listener for wakeup state and result
     * @param recognitionListener listener for recognition state and result
     */
    synchronized void startRecognition(final WakeupListener wakeUpListener, final RecognitionListener recognitionListener) throws VoiceException {
        checkConnectionStatus();
        if (wakeUpListener == null) {
            throw new IllegalArgumentException("The WakeupListener cannot be null!");
        }
        if (recognitionListener == null) {
            throw new IllegalArgumentException("The RecognitionListener cannot be null!");
        }
        mWakeupListener = wakeUpListener;
        mRecognitionListener = recognitionListener;
        try {
            mIRecognizer.startRecognition(mIWakeupListener, mIRecognitionListener);
        } catch (RemoteException e) {
            throw new VoiceException("startRecognition", e);
        }
    }

    /**
     * Stop Recognition
     */
    void stopRecognition() throws VoiceException {
        checkConnectionStatus();
        mWakeupListener = null;
        mRecognitionListener = null;
        try {
            mIRecognizer.stopRecognition();
        } catch (RemoteException e) {
            throw new VoiceException("stopRecognition", e);
        }
    }

    /**
     * Force enable beam forming and start listen
     * Wakeup and Recognition will be blocked
     *
     * @param listener raw data listener
     */
    void startBeamFormingListen(RawDataListener listener) throws VoiceException {
        checkConnectionStatus();
        mRawDataListener = listener;
        try {
            mIRecognizer.startBeamFormingListen(mIRawDataListener);
        } catch (RemoteException e) {
            throw new VoiceException("startBeamFormingListen", e);
        }
    }

    /**
     * Stop listener
     */
    void stopBeamFormingListen() throws VoiceException {
        checkConnectionStatus();
        try {
            mIRecognizer.stopBeamFormingListen();
        } catch (RemoteException e) {
            throw new VoiceException("stopBeamFormingListen", e);
        }
    }

    void setSoundEnabled(boolean enabled) throws VoiceException {
        checkConnectionStatus();
        try {
            mIRecognizer.setSoundEnabled(enabled);
        } catch (RemoteException e) {
            throw new VoiceException("setSoundEnabled", e);
        }
    }

    void disconnectService() throws VoiceException {
        checkConnectionStatus();
        try {
            mIRecognizer.disconnectService();
        } catch (RemoteException e) {
            throw new VoiceException("disconnectService", e);
        }
    }

    public Version getVersion() {
        return new Version(VersionInfo.version_channel,
                VersionInfo.version_name,
                VersionInfo.version_code,
                VersionInfo.version_min);
    }

    synchronized void startWakeupMode(WakeupListener wakeUpListener) throws VoiceException {
        checkConnectionStatus();
        if (wakeUpListener == null) {
            throw new IllegalArgumentException("The WakeupListener cannot be null!");
        }
        mWakeupListener = wakeUpListener;
        try {
            mIRecognizer.startWakeupMode(mIWakeupListener);
        } catch (RemoteException e) {
            throw new VoiceException("startWakeupMode", e);
        }
    }

    synchronized void startRecognitionMode(RecognitionListener recognitionListener) throws VoiceException {
        checkConnectionStatus();
        if (recognitionListener == null) {
            throw new IllegalArgumentException("The RecognitionListener cannot be null!");
        }
        mRecognitionListener = recognitionListener;
        try {
            mIRecognizer.startRecognitionMode(mIRecognitionListener);
        } catch (RemoteException e) {
            throw new VoiceException("startRecognitionMode", e);
        }
    }

    synchronized void startRecognitionAndWakeup(RecognitionListener recognitionListener, WakeupListener wakeUpListener) throws VoiceException {
        checkConnectionStatus();
        if (wakeUpListener == null) {
            throw new IllegalArgumentException("The WakeupListener cannot be null!");
        }
        if (recognitionListener == null) {
            throw new IllegalArgumentException("The RecognitionListener cannot be null!");
        }
        mWakeupListener = wakeUpListener;
        mRecognitionListener = recognitionListener;
        try {
            mIRecognizer.startNewRecognition(mIWakeupListener, mIRecognitionListener);
        } catch (RemoteException e) {
            throw new VoiceException("startRecognitionAndWakeup", e);
        }
    }

    void checkConnectionStatus() throws VoiceException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
    }

    //for test
    protected void setIRecognizer(IRecognizer recognizer) {
        mIRecognizer = recognizer;
    }
}
