package com.segway.robot.sdk.voice.grammar;

import android.os.Parcel;
import android.os.Parcelable;

import java.util.ArrayList;
import java.util.List;

public class Slot implements Parcelable {

    private String mName;
    private boolean isOptional = false;
    private List<String> mWordList = new ArrayList<>();

    /**
     * the constructor of the Slot class.
     *
     * @param name the name of the slot.
     */
    public Slot(String name) {
        this.mName = name;
    }

    /**
     * the constructor of the Slot class.
     *
     * @param name     the name of the slot.
     * @param optional the option of the slot.
     */
    public Slot(String name, boolean optional) {
        this.mName = name;
        this.isOptional = optional;
    }

    /**
     * the constructor of the Slot class.
     *
     * @param name     the name of the slot.
     * @param optional the option of the slot.
     * @param wordList the word list of the slot.
     */
    public Slot(String name, boolean optional, List<String> wordList) {
        this.mName = name;
        this.isOptional = optional;
        this.mWordList = wordList;
    }

    /**
     * the parcel constructor of the Slot class.
     *
     * @param in the parcel where to read the name, the option and the word list of the slot.
     */
    protected Slot(Parcel in) {
        mName = in.readString();
        isOptional = in.readByte() != 0;
        mWordList = in.createStringArrayList();
    }

    public static final Creator<Slot> CREATOR = new Creator<Slot>() {

        /**
         * @return a new slot from the data in the specified parcel.
         */
        @Override
        public Slot createFromParcel(Parcel in) {
            return new Slot(in);
        }

        /**
         * @return an array of slots of the specified size.
         */
        @Override
        public Slot[] newArray(int size) {
            return new Slot[size];
        }
    };

    /**
     * Set the name of the slot.
     *
     * @param name the new name of slot.
     */
    public void setName(String name) {
        this.mName = name;
    }

    /**
     * Get the name of the slot.
     *
     * @return the name of the slot.
     */
    public String getName() {
        return this.mName;
    }

    /**
     * Set the option of the slot.
     *
     * @param optional the option of the slot.
     */
    public void setOptional(boolean optional) {
        this.isOptional = optional;
    }

    /**
     * Get the option of the slot.
     *
     * @return the option of the slot.
     */
    public boolean getOptional() {
        return this.isOptional;
    }

    /**
     * Add a new slot word.
     *
     * @param Word the new slot word.
     */
    public void addWord(String Word) {
        mWordList.add(Word);
    }

    /**
     * Add an array of the slot words.
     *
     * @param Words an array of the slot words.
     */
    public void addWordsArray(String[] Words) {
        for (String word : Words) {
            mWordList.add(word);
        }
    }

    /**
     * Get the word list of the slot.
     *
     * @return a list of words in the slot.
     */
    public List<String> getWordList() {
        return this.mWordList;
    }

    /**
     * Delete the word in the slot.
     *
     * @param deleteSlotWord the word to be deleted.
     * @return true if the word is deleted, return false if the word is not deleted.
     */
    public boolean deleteWord(String deleteSlotWord) {
        return mWordList.remove(deleteSlotWord);
    }

    /**
     * It is suitable for Chinese grammar slot only.
     *
     * @return converted to a string description of the slot with the option state.
     */
    public String toOptionalString() {
        if (isOptional) {
            return "[<" + this.mName + ">]";
        } else {
            return "<" + this.mName + ">";
        }
    }

    @Override
    public String toString() {
        StringBuilder slot = new StringBuilder();
        slot.append("<").append(this.mName).append(">:");
        int slotLength = mWordList.size();
        if (slotLength == 0) {
            return this.mName;
        }
        slot.append(mWordList.get(0));
        if (slotLength > 1) {
            for (int i = 1; i < slotLength; i++) {
                slot.append("|").append(mWordList.get(i));
            }
        }
        slot.append(";\n");
        return slot.toString();
    }

    public final String getSlotInfo() {
        int slotLength = mWordList.size();
        if (slotLength == 0) {
            return this.mName;
        }
        StringBuilder slot = new StringBuilder();
        slot.append("<").append(this.mName).append(">:");
        slot.append("\"").append(mWordList.get(0)).append("\"");
        if (slotLength > 1) {
            for (int i = 1; i < slotLength; i++) {
                slot.append("|\"").append(mWordList.get(i)).append("\"");
            }
        }
        slot.append(";\n");
        return slot.toString();
    }

    /**
     * Parcelable interface methods.
     *
     * @return a bitmask indicating the set of special object types marshaled by this Parcelable object instance.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mName);
        dest.writeByte((byte) (isOptional ? 1 : 0));
        dest.writeStringList(mWordList);
    }
}
