package com.segway.robot.sdk.voice;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;

import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.voice.audiodata.IRawDataListener;
import com.segway.robot.sdk.voice.audiodata.RawDataListener;
import com.segway.robot.sdk.voice.grammar.GrammarConstraint;
import com.segway.robot.sdk.voice.grammar.IRebuildGrammarListener;
import com.segway.robot.sdk.voice.grammar.RebuildGrammarListener;
import com.segway.robot.sdk.voice.recognition.IRecognitionListener;
import com.segway.robot.sdk.voice.recognition.IWakeupListener;
import com.segway.robot.sdk.voice.recognition.RecognitionListener;
import com.segway.robot.sdk.voice.recognition.RecognitionResult;
import com.segway.robot.sdk.voice.recognition.WakeupListener;
import com.segway.robot.sdk.voice.recognition.WakeupResult;
import com.segway.robot.sdk.voice.tts.ITtsListener;
import com.segway.robot.sdk.voice.tts.TtsListener;

public class VoiceManager {
    private static final String TAG = VoiceManager.class.getSimpleName();
    private static final String SERVICE_PACKAGE_NAME = "com.segway.robot.coreservice.voiceservice";
    private static final String SERVICE_CLASS_NAME = "com.segway.robot.coreservice.voiceservice.VoiceAssistantService";
    private static final String SERVICE_ACTION = "com.segway.robot.coreservice.voiceservice.VOICE_SERVICE";
    private Context mContext;
    private IVoiceService mVoiceService;
    private ServiceStateListener mServiceStateListener;
    private static VoiceManager mVoiceManager;

    private VoiceManager() {
    }

    /**
     * Create a VoiceManager instance.
     *
     * @return an instance of VoiceManager.
     */
    public static synchronized VoiceManager getInstance() {
        if (mVoiceManager == null) {
            mVoiceManager = new VoiceManager();
        }
        return mVoiceManager;
    }

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            Logger.d(TAG, "service connected");
            mVoiceService = IVoiceService.Stub.asInterface(service);
            if (mVoiceService != null) {
                mServiceStateListener.onServiceConnected();
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            Logger.d(TAG, "service disconnected");
            mServiceStateListener.onServiceDisconnected();
            mVoiceService = null;
        }
    };

    /**
     * Connect to the voice service.
     *
     * @param context  The context connects to the voice service.
     * @param callback A {@link ServiceStateListener} that provides the connection state.
     * @return true if the connection succeeds, false if the connection fails.
     */
    public boolean bind(Context context, ServiceStateListener callback) {
        if (context == null) {
            throw new IllegalStateException("The context cannot be null!");
        }
        if (callback == null) {
            throw new IllegalStateException("The ServiceStateListener cannot be null!");
        }
        mContext = context.getApplicationContext();
        mServiceStateListener = callback;
        Intent voiceServiceIntent = new Intent();
        voiceServiceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        mContext.startService(voiceServiceIntent);
        Logger.d(TAG, "try to connect to the voice service");
        return mContext.bindService(voiceServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
    }

    /**
     * Disconnect from the voice service.
     */
    public void unbind() {
        Logger.d(TAG, "unbindService");
        if (mContext != null) {
            mContext.unbindService(mServiceConnection);
            Intent voiceServiceIntent = new Intent();
            voiceServiceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        }
    }

    /**
     * Acquire permission of voice control.
     *
     * @param voicePermissionListener a {@link VoicePermissionListener} that provides the acquisition result.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void acquireVoiceControlPermission(final VoicePermissionListener voicePermissionListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (voicePermissionListener == null) {
            throw new IllegalStateException("The VoicePermissionListener cannot be null!");
        }
        mVoiceService.acquireVoiceControlPermission(new VoiceControlPermissionListener.Stub() {

            @Override
            public void onPermit() throws RemoteException {
                voicePermissionListener.onPermit();
            }

            @Override
            public void onDeprive() throws RemoteException {
                voicePermissionListener.onDeprive();
            }
        });
    }

    /**
     * Release permission of voice control.
     *
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void releaseVoiceControlPermission() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mVoiceService.releaseVoiceControlPermission();
    }

    /**
     * whether the voice service supports cloud or not.
     *
     * @return true if it supports the cloud service.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean isCloudSupported() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        return mVoiceService.isCloudSupported();
    }

    // TODO: 2016/8/10 hide

    /**
     * Register a callback to get the audio raw data when this recognizer is initiated and starts to work.
     *
     * @param rawDataListener the {@link RawDataListener} that gives the audio raw data.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void setRawDataListener(final RawDataListener rawDataListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (rawDataListener == null) {
            throw new IllegalStateException("The RawDataListener cannot be null!");
        }
        mVoiceService.setRawDataListener(new IRawDataListener.Stub() {

            @Override
            public void onRawData(byte[] data, int dataLength) throws RemoteException {
                rawDataListener.onRawData(data, dataLength);
            }
        });
    }

    /**
     * Initiate a wake-up recognition.
     *
     * @param wakeupListener A {@link WakeupListener} to be notified when a wake-up phrase is detected,
     *                       or the error description if an error occurs.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void startWakeup(final WakeupListener wakeupListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (wakeupListener == null) {
            throw new IllegalStateException("The WakeupListener cannot be null!");
        }
        mVoiceService.startWakeup(new IWakeupListener.Stub() {
            @Override
            public void onStandby() throws RemoteException {
                wakeupListener.onStandby();
            }

            @Override
            public void onWakeupResult(WakeupResult wakeupResult) throws RemoteException {
                wakeupListener.onWakeupResult(wakeupResult);
            }

            @Override
            public void onWakeupError(String error) throws RemoteException {
                wakeupListener.onWakeupError(error);
            }
        });
    }

    /**
     * Initiate a recognition.
     *
     * @param recognitionListener A {@link RecognitionListener} that provides the results,
     *                            or the error description if an error occurred.
     * @param isLocal             Whether the recognition needs cloud support or not.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void startRecognition(final RecognitionListener recognitionListener, boolean isLocal) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (recognitionListener == null) {
            throw new IllegalStateException("The RecognitionListener cannot be null!");
        }
        mVoiceService.startRecognition(new IRecognitionListener.Stub() {
            @Override
            public void onRecognitionStart() throws RemoteException {
                recognitionListener.onRecognitionStart();
            }

            @Override
            public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
                return recognitionListener.onRecognitionResult(recognitionResult);
            }

            @Override
            public boolean onRecognitionError(String error) throws RemoteException {
                return recognitionListener.onRecognitionError(error);
            }
        }, isLocal);
    }

    /**
     * Terminate the current recognition.
     *
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void stopRecognition() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        try {
            mVoiceService.stopRecognition();
        } catch (RemoteException e) {
            // TODO: 2016/8/19
            e.printStackTrace();
        }
    }

    /**
     * Speech to text.
     *
     * @param recognitionListener A {@link RecognitionListener} that provides the results,
     *                            or the error description if an error occurs.
     * @throws RemoteException an exception that occurs in the remote voice server connection..
     */
    public void speechToText(final RecognitionListener recognitionListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (recognitionListener == null) {
            throw new IllegalStateException("The RecognitionListener cannot be null!");
        }
        mVoiceService.speechToText(new IRecognitionListener.Stub() {
            @Override
            public void onRecognitionStart() throws RemoteException {
                recognitionListener.onRecognitionStart();
            }

            @Override
            public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
                return recognitionListener.onRecognitionResult(recognitionResult);
            }

            @Override
            public boolean onRecognitionError(String error) throws RemoteException {
                return recognitionListener.onRecognitionError(error);
            }
        });
    }

    /**
     * Start the simple voice control mode where the wake-up and the recognition take turns to execute.
     *
     * @param wakeupListener      A {@link WakeupListener} to be notified when a wake-up phrase is detected,
     *                            or the error description if an error occurs.
     * @param recognitionListener A {@link RecognitionListener} that provides the results,
     *                            or the error description if an error occurs.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void startSimpleVoiceControlMode(final WakeupListener wakeupListener, final RecognitionListener recognitionListener)
            throws RemoteException, VoiceException {
        Logger.d(TAG, "start simple voice control mode");
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (wakeupListener == null) {
            throw new IllegalStateException("The WakeupListener cannot be null!");
        }
        if (recognitionListener == null) {
            throw new IllegalStateException("The RecognitionListener cannot be null!");
        }
        mVoiceService.startSimpleVoiceControlMode(new IWakeupListener.Stub() {
            @Override
            public void onStandby() throws RemoteException {
                wakeupListener.onStandby();
            }

            @Override
            public void onWakeupResult(WakeupResult wakeupResult) throws RemoteException {
                wakeupListener.onWakeupResult(wakeupResult);
            }

            @Override
            public void onWakeupError(String error) throws RemoteException {
                wakeupListener.onWakeupError(error);
            }

        }, new IRecognitionListener.Stub() {
            @Override
            public void onRecognitionStart() throws RemoteException {
                recognitionListener.onRecognitionStart();
            }

            @Override
            public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
                return recognitionListener.onRecognitionResult(recognitionResult);
            }

            @Override
            public boolean onRecognitionError(String error) throws RemoteException {
                return recognitionListener.onRecognitionError(error);
            }
        });
    }

    /**
     * Text to speech.
     *
     * @param text        the speech content.
     * @param ttsListener a {@link TtsListener} which provides callbacks for certain text-to-speech (TTS) generation events.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void textToSpeech(String text, final TtsListener ttsListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (ttsListener == null) {
            throw new IllegalStateException("The TtsListener cannot be null!");
        }
        mVoiceService.textToSpeech(text, new ITtsListener.Stub() {
            @Override
            public void onSpeechStarted(String word) throws RemoteException {
                ttsListener.onSpeechStarted(word);
            }

            @Override
            public void onSpeechFinished(String word) throws RemoteException {
                ttsListener.onSpeechFinished(word);
            }

            @Override
            public void onSpeechError(String word, String reason) throws RemoteException {
                ttsListener.onSpeechError(word, reason);
            }
        });
    }

    /**
     * Stop any TTS that is in progress.
     *
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void stopSpeech() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mVoiceService.stopSpeech();
    }

    /**
     * Get the raw data when beamformer works.
     *
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void getBeamFormerData() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mVoiceService.getBeamFormerData();
    }

    /**
     * Disable beamformer.
     *
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void stopBeamFormer() throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mVoiceService.stopBeamFormer();
    }

    /**
     * Add a grammar constraint into the recognition engine.
     *
     * @param grammarConstraint the {@link GrammarConstraint} to be added to the recognition grammar.
     * @return true if the grammar constraint addition is successful.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean addGrammarConstraint(GrammarConstraint grammarConstraint) throws RemoteException, VoiceException {
        Logger.d(TAG, "add a grammar constraint, it is " + grammarConstraint.getName());
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (grammarConstraint == null || grammarConstraint.getSlotList().isEmpty()) {
            throw new IllegalStateException("grammar constraint is illegal");
        }
        return mVoiceService.addGrammarConstraint(grammarConstraint);
    }

    /**
     * Rebuild the grammar if the grammar is modified.
     *
     * @param rebuildGrammarListener a {@link RebuildGrammarListener} that provides the status of the grammar rebuilding.
     * @return true if the rebuilding is successful.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean rebuildGrammar(final RebuildGrammarListener rebuildGrammarListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (rebuildGrammarListener == null) {
            throw new IllegalStateException("The RebuildGrammarListener cannot be null!");
        }
        return mVoiceService.rebuildGrammar(new IRebuildGrammarListener.Stub() {
            @Override
            public void onComplete() throws RemoteException {
                rebuildGrammarListener.onComplete();
            }

            @Override
            public void onError(String error) throws RemoteException {
                rebuildGrammarListener.onError(error);
            }
        });
    }

    /**
     * Reset the application grammar to the original state.
     *
     * @param rebuildGrammarListener a {@link RebuildGrammarListener} that provides the status of application grammar reset.
     * @return true if the reset is successful.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean resetAppGrammar(final RebuildGrammarListener rebuildGrammarListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (rebuildGrammarListener == null) {
            throw new IllegalStateException("The RebuildGrammarListener cannot be null!");
        }
        return mVoiceService.resetAppGrammar(new IRebuildGrammarListener.Stub() {
            @Override
            public void onComplete() throws RemoteException {
                rebuildGrammarListener.onComplete();
            }

            @Override
            public void onError(String error) throws RemoteException {
                rebuildGrammarListener.onError(error);
            }
        });
    }

    /**
     * Reset the system grammar to the original state.
     *
     * @param rebuildGrammarListener A {@link RebuildGrammarListener} that provides status of system grammar reset.
     * @return true if the reset is successful.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean resetSystemGrammar(final RebuildGrammarListener rebuildGrammarListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (rebuildGrammarListener == null) {
            throw new IllegalStateException("The RebuildGrammarListener cannot be null!");
        }
        return mVoiceService.resetSystemGrammar(new IRebuildGrammarListener.Stub() {
            @Override
            public void onComplete() throws RemoteException {
                rebuildGrammarListener.onComplete();
            }

            @Override
            public void onError(String error) throws RemoteException {
                rebuildGrammarListener.onError(error);
            }
        });
    }

    /**
     * Add a system grammar constraint.
     *
     * @param grammarConstraint the {@link GrammarConstraint} to be added to the system grammar.
     * @return true if the grammar constraint addition is successful.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public boolean addSystemGrammarConstraint(GrammarConstraint grammarConstraint) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (grammarConstraint == null || grammarConstraint.getSlotList().isEmpty()) {
            throw new IllegalStateException("the grammar constraint is illegal");
        }
        return mVoiceService.addSystemGrammarConstraint(grammarConstraint);
    }

    /**
     * Register a callback to get the state of system recognition.
     *
     * @param systemRecognitionListener A {@link RecognitionListener} that provides the results,
     *                                  or the error description if an error occurred.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void setSystemRecognitionListener(final RecognitionListener systemRecognitionListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (systemRecognitionListener == null) {
            throw new IllegalStateException("The RecognitionListener cannot be null!");
        }
        mVoiceService.setSystemRecognitionListener(new IRecognitionListener.Stub() {
            @Override
            public void onRecognitionStart() throws RemoteException {
                systemRecognitionListener.onRecognitionStart();
            }

            @Override
            public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
                return systemRecognitionListener.onRecognitionResult(recognitionResult);
            }

            @Override
            public boolean onRecognitionError(String error) throws RemoteException {
                return systemRecognitionListener.onRecognitionError(error);
            }
        });
    }

    /**
     * Register a callback to get the state of application recognition.
     *
     * @param applicationRecognitionListener A {@link RecognitionListener} that provides the results,
     *                                       or the error description if an error occurred.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void setApplicationRecognitionListener(final RecognitionListener applicationRecognitionListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (applicationRecognitionListener == null) {
            throw new IllegalStateException("The RecognitionListener cannot be null!");
        }
        mVoiceService.setApplicationRecognitionListener(new IRecognitionListener.Stub() {
            @Override
            public void onRecognitionStart() throws RemoteException {
                applicationRecognitionListener.onRecognitionStart();
            }

            @Override
            public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
                return applicationRecognitionListener.onRecognitionResult(recognitionResult);
            }

            @Override
            public boolean onRecognitionError(String error) throws RemoteException {
                return applicationRecognitionListener.onRecognitionError(error);
            }
        });
    }

    /**
     * Set a application wake-up listener.
     *
     * @param applicationWakeupListener A {@link WakeupListener} to be notified when a wake-up phrase is detected,
     *                                  or the error description if an error occurs.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void setApplicationWakeupListener(final WakeupListener applicationWakeupListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (applicationWakeupListener == null) {
            throw new IllegalStateException("The WakeupListener cannot be null!");
        }
        mVoiceService.setApplicationWakeupListener(new IWakeupListener.Stub() {
            @Override
            public void onStandby() throws RemoteException {
                applicationWakeupListener.onStandby();
            }

            @Override
            public void onWakeupResult(WakeupResult wakeupResult) throws RemoteException {
                applicationWakeupListener.onWakeupResult(wakeupResult);
            }

            @Override
            public void onWakeupError(String error) throws RemoteException {
                applicationWakeupListener.onWakeupError(error);
            }
        });
    }

    /**
     * Set a system wake-up listener.
     *
     * @param systemWakeupListener A {@link WakeupListener} to be notified when a wake-up phrase is detected,
     *                             or the error description if an error occurs.
     * @throws RemoteException an exception that occurs in the remote voice server connection.
     */
    public void setSystemWakeupListener(final WakeupListener systemWakeupListener) throws RemoteException, VoiceException {
        if (mVoiceService == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (systemWakeupListener == null) {
            throw new IllegalStateException("The WakeupListener cannot be null!");
        }
        mVoiceService.setSystemWakeupListener(new IWakeupListener.Stub() {
            @Override
            public void onStandby() throws RemoteException {
                systemWakeupListener.onStandby();
            }

            @Override
            public void onWakeupResult(WakeupResult wakeupResult) throws RemoteException {
                systemWakeupListener.onWakeupResult(wakeupResult);
            }

            @Override
            public void onWakeupError(String error) throws RemoteException {
                systemWakeupListener.onWakeupError(error);
            }
        });
    }

    public void enableBeamForming() throws RemoteException {
        mVoiceService.enableBeamForming();
    }

    public void disableBeamForming() throws RemoteException {
        mVoiceService.disableBeamForming();
    }
}
