package com.segway.robot.sdk.voice;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;

import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;
import com.segway.robot.sdk.voice.audiodata.IRawDataListener;
import com.segway.robot.sdk.voice.audiodata.RawDataListener;
import com.segway.robot.sdk.voice.grammar.GrammarConstraint;
import com.segway.robot.sdk.voice.grammar.Slot;
import com.segway.robot.sdk.voice.recognition.IRecognitionListener;
import com.segway.robot.sdk.voice.recognition.IWakeupListener;
import com.segway.robot.sdk.voice.recognition.RecognitionListener;
import com.segway.robot.sdk.voice.recognition.RecognitionResult;
import com.segway.robot.sdk.voice.recognition.WakeupListener;
import com.segway.robot.sdk.voice.recognition.WakeupResult;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by ark338 on 16/9/1.
 */
class RecognizerManager implements ServiceBinder {
    private static final String TAG = RecognizerManager.class.getSimpleName();
    private static final String RECOGNIZER_SERVICE_PACKAGE_NAME = "com.segway.robot.coreservice.voiceservice";
    private static final String RECOGNIZER_SERVICE_CLASS_NAME = "com.segway.robot.coreservice.voiceservice.RecognitionService";
    /**
     * Record mode parameters
     */
    final int BIT_RATE = 16;
    final int SAMPLE_RATE = 16000;
    IRecognizer mIRecognizer;
    private Context mContext;
    private BindStateListener mListener;
    private boolean isBind = false;
    private WakeupListener mWakeupListener;
    private RecognitionListener mRecognitionListener;
    private RawDataListener mRawDataListener;
    private final Object mLockWakeupListener;
    private final Object mLockRecognitionListener;
    private int mRecognitionLanguage;

    // TODO: 16/9/1 singleton
    RecognizerManager() {
        mLockWakeupListener = new Object();
        mLockRecognitionListener = new Object();
    }

    private IWakeupListener.Stub mIWakeupListener = new IWakeupListener.Stub() {
        @Override
        public void onStandby() throws RemoteException {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onStandby();
                }
            }
        }

        @Override
        public void onWakeupResult(WakeupResult wakeupResult) throws RemoteException {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onWakeupResult(wakeupResult);
                }
            }
        }

        @Override
        public void onWakeupError(String error) throws RemoteException {
            synchronized (mLockWakeupListener) {
                if (mWakeupListener != null) {
                    mWakeupListener.onWakeupError(error);
                }
            }
        }
    };

    private IRecognitionListener.Stub mIRecognitionListener = new IRecognitionListener.Stub() {
        @Override
        public void onRecognitionStart() throws RemoteException {
            synchronized (mLockRecognitionListener) {
                if (mRecognitionListener != null) {
                    mRecognitionListener.onRecognitionStart();
                }
            }
        }

        @Override
        public boolean onRecognitionResult(RecognitionResult recognitionResult) throws RemoteException {
            boolean doRecognition;
            synchronized (mLockRecognitionListener) {
                doRecognition = mRecognitionListener != null && mRecognitionListener.onRecognitionResult(recognitionResult);
            }
            return doRecognition;
        }

        @Override
        public boolean onRecognitionError(String error) throws RemoteException {
            boolean doRecognition;
            synchronized (mLockRecognitionListener) {
                doRecognition = mRecognitionListener != null && mRecognitionListener.onRecognitionError(error);
            }
            return doRecognition;
        }
    };

    private IRawDataListener.Stub mIRawDataListener = new IRawDataListener.Stub() {
        @Override
        public void onRawData(byte[] data, int dataLength) throws RemoteException {
            mRawDataListener.onRawData(data, dataLength);
        }
    };

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            Logger.d(TAG, "service connected");
            mIRecognizer = IRecognizer.Stub.asInterface(service);

            if (mIRecognizer != null) {
                mListener.onBind();
            }
            try {
                mRecognitionLanguage = getLanguage();
                Logger.d(TAG, "RecognitionLanguage: " + mRecognitionLanguage);
            } catch (RemoteException | VoiceException e) {
                Logger.e(TAG, "Exception :", e);
            }

            // check version
            try {
                Version serviceVersion = mIRecognizer.getVersion();
                getVersion().check("VersionInfo", serviceVersion);
            } catch (RemoteException e) {
                String error = "Cannot get Recognition Service version, err = " + e.getMessage();

                // disconnect to remote service
                mListener.onUnbind(error);
                mIRecognizer = null;
                Logger.e(TAG, error, e);
            } catch (VersionMismatchException e) {
                String error = "Version mismatch: " + e.getMessage();

                // disconnect to remote service
                mListener.onUnbind(error);
                mIRecognizer = null;
                Logger.e(TAG, "Version mismatch", e);

                throw e;
            }
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            Logger.d(TAG, "service disconnected");
            isBind = false;
            mListener.onUnbind("The recognition service disconnected!");
            mIRecognizer = null;
        }
    };

    @Override
    public synchronized boolean bindService(Context context, BindStateListener listener) {
        Logger.d(TAG, "RecognizerManager bindService");
        if (isBind) {
            return true;
        }
        if (context == null) {
            throw new IllegalArgumentException("The context cannot be null!");
        }
        if (listener == null) {
            throw new IllegalArgumentException("The BindStateListener cannot be null!");
        }
        mContext = context.getApplicationContext();
        mListener = listener;
        Intent voiceServiceIntent = new Intent();
        voiceServiceIntent.setClassName(RECOGNIZER_SERVICE_PACKAGE_NAME, RECOGNIZER_SERVICE_CLASS_NAME);
        mContext.startService(voiceServiceIntent);
        Logger.d(TAG, "try to connect to the recognition service");
        isBind = mContext.bindService(voiceServiceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
        return isBind;
    }

    @Override
    public synchronized void unbindService() {
        Logger.d(TAG, "unbindService");
        if (!isBind) {
            return;
        }
        if (mContext != null) {
            mContext.unbindService(mServiceConnection);
        }
        isBind = false;
        mIRecognizer = null;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mListener;
    }

    /**
     * Get current using language
     */
    @Languages.Language
    int getLanguage() throws RemoteException, VoiceException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        int language = mIRecognizer.getLanguage();
        if (language == 0) {
            return Languages.EN_US;
        } else {
            return Languages.ZH_CN;
        }
    }

    /**
     * Set to enable/disable beam forming
     * In current version, beam forming direction is fixed to the front of robot head
     * Beam forming is not effective in wakeup stage
     *
     * @param enable
     */
    void beamForming(boolean enable) throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mIRecognizer.beamForming(enable);
    }

    /**
     * Create GrammarConstraint from json file
     * <p>
     * <p>
     * sample
     {
         "name": "play_music",
         "slotList": [
             {
                 "name": "play_cmd",
                 "isOptional": false,
                 "word": [
                     "play",
                     "play music"
                 ]
             },
             {
                 "name": "song",
                 "isOptional": false,
                 "word": [
                     "happy birthday",
                     "jingle bells"
                 ]
             }
         ]
     }
     *
     * @param json file describe GrammarConstraint
     * @return
     */
    GrammarConstraint createGrammarConstraint(String json) throws VoiceException {
        // TODO: 2016/9/30 verify
        GrammarConstraint grammarFromJson;
        try {
            JSONObject grammarConstraintObject = new JSONObject(json);
            List<Slot> slotList = new ArrayList<>();
            String name = grammarConstraintObject.getString("name");
            JSONArray slotListFromJson = grammarConstraintObject.getJSONArray("slotList");
            int length = slotListFromJson.length();
            for (int i = 0; i < length; i++) {
                JSONObject slotObject = slotListFromJson.getJSONObject(i);
                Slot slot = new Slot(slotObject.getString("name"));
                slot.setOptional(slotObject.getBoolean("isOptional"));
                JSONArray wordListFromJson = slotObject.getJSONArray("word");
                int wordLength = wordListFromJson.length();
                String[] words = new String[wordLength];
                for (int j = 0; j < wordLength; j++) {
                    words[j] = wordListFromJson.getString(j);
                    boolean b = mRecognitionLanguage == Languages.EN_US ? CheckUtil.isEnChar(words[j]) :
                            CheckUtil.isCnChar(words[j]);
                    if (!b) {
                        throw new IllegalStateException("\"" + words[j] + "\" is an illegal word");
                    }
                }
                slot.addWordsArray(words);
                slotList.add(slot);
            }
            grammarFromJson = new GrammarConstraint(name, slotList);
        } catch (JSONException e) {
            throw new VoiceException("Json illegal, exception: ", e);
        }
        return grammarFromJson;
    }

    /**
     * Add GrammarConstraint
     *
     * @param grammarConstraint
     */
    void addGrammarConstraint(GrammarConstraint grammarConstraint) throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        List<Slot> slotList = grammarConstraint.getSlotList();
        int size = slotList.size();
        if (size == 0) {
            throw new IllegalArgumentException("Your grammar constraint has no slot!");
        }
        for (Slot slot : slotList) {
            boolean b = mRecognitionLanguage == Languages.EN_US ? CheckUtil.isEnSlot(slot) :
                    CheckUtil.isCnSlot(slot);
            Logger.d(TAG, slot.getName() + "is a legal slot?:" + b);
            if (!b) {
                throw new IllegalStateException("\"" + slot.getName() + "\" is an illegal slot");
            }
        }
        mIRecognizer.addGrammarConstraint(grammarConstraint);
    }

    /**
     * Remove GrammarConstraint
     *
     * @param grammarConstraint
     */
    void removeGrammarConstraint(GrammarConstraint grammarConstraint) throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mIRecognizer.removeGrammarConstraint(grammarConstraint);
    }

    /**
     * Start Wakeup and Recognition
     *
     * @param wakeUpListener      listener for wakeup state and result
     * @param recognitionListener listener for recognition state and result
     */
    synchronized void startRecognition(final WakeupListener wakeUpListener, final RecognitionListener recognitionListener) throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        if (wakeUpListener == null) {
            throw new IllegalArgumentException("The WakeupListener cannot be null!");
        }
        if (recognitionListener == null) {
            throw new IllegalArgumentException("The RecognitionListener cannot be null!");
        }
        mWakeupListener = wakeUpListener;
        mRecognitionListener = recognitionListener;
        mIRecognizer.startRecognition(mIWakeupListener, mIRecognitionListener);
    }

    /**
     * Stop Recognition
     */
    void stopRecognition() throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mWakeupListener = null;
        mRecognitionListener = null;
        mIRecognizer.stopRecognition();
    }

    /**
     * Force enable beam forming and start listen
     * Wakeup and Recognition will be blocked
     *
     * @param listener raw data listener
     */
    void startBeamFormingListen(RawDataListener listener) throws VoiceException, RemoteException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mRawDataListener = listener;
        mIRecognizer.startBeamFormingListen(mIRawDataListener);
    }

    /**
     * Stop listener
     */
    void stopBeamFormingListen() throws RemoteException, VoiceException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mIRecognizer.stopBeamFormingListen();
    }

    void disconnectService() throws RemoteException, VoiceException {
        if (mIRecognizer == null) {
            throw new VoiceException("The voice service disconnected!");
        }
        mIRecognizer.disconnectService();
    }

    public Version getVersion() {
        return new Version(VersionInfo.version_channel,
                VersionInfo.version_name,
                VersionInfo.version_code,
                VersionInfo.version_min);
    }
}
