package com.segway.robot.sdk.voice.grammar;

import android.os.Parcel;
import android.os.Parcelable;

import java.util.ArrayList;
import java.util.List;

/**
 * A grammar constraint that specifies a single grammar,
 * # which consists of transitions between word slots. Grammar constraints may be
 * # simple or complex, with optional word slots and branching. Constraints
 * # may also reference other constraints.
 */
public class GrammarConstraint implements Parcelable {

    /**
     * the grammar constraint name.
     */
    private String mName;

    /**
     * the list of slots.
     */
    private List<Slot> mSlotList = new ArrayList<>();

    /**
     * the constructor of the GrammarConstraint class.
     */
    public GrammarConstraint() {
    }

    /**
     * the constructor of the GrammarConstraint class.
     *
     * @param name     the grammar constraint name.
     * @param slotList the list of {@link Slot}.
     */
    public GrammarConstraint(String name, List<Slot> slotList) {
        this.mName = name;
        this.mSlotList = slotList;
    }

    /**
     * the parcel constructor of the GrammarConstraint class.
     *
     * @param in the parcel where to read the grammar constraint name and {@link Slot} list.
     */
    protected GrammarConstraint(Parcel in) {
        mName = in.readString();
        mSlotList = in.createTypedArrayList(Slot.CREATOR);
    }

    public static final Creator<GrammarConstraint> CREATOR = new Creator<GrammarConstraint>() {

        /**
         * @return a new GrammarConstraint from the data in the specified parcel.
         */
        @Override
        public GrammarConstraint createFromParcel(Parcel in) {
            return new GrammarConstraint(in);
        }

        /**
         * @return an array of grammar constraints of the specified size.
         */
        @Override
        public GrammarConstraint[] newArray(int size) {
            return new GrammarConstraint[size];
        }
    };

    /**
     * Set the grammar constraint name.
     *
     * @param name the grammar constraint name.
     */
    public void setName(String name) {
        this.mName = name;
    }

    /**
     * Set the slots list.
     *
     * @param slotList a new list of {@link Slot}.
     */
    public void setSlotList(List<Slot> slotList) {
        this.mSlotList = slotList;
    }

    /**
     * Add a slot.
     *
     * @param slot the newly added {@link Slot}.
     */
    public void addSlot(Slot slot) {
        mSlotList.add(slot);
    }

    /**
     * Add a slot.
     *
     * @param location the location of the list where the slot will be added to.
     * @param slot     the newly added {@link Slot}.
     */
    public void addSlot(int location, Slot slot) {
        mSlotList.add(location, slot);
    }

    /**
     * Delete a slot.
     *
     * @param slot the {@link Slot} to be deleted.
     */
    public void deleteSlot(Slot slot) {
        mSlotList.remove(slot);
    }

    /**
     * Delete a slot.
     *
     * @param location the location of deleted slot.
     */
    public void deleteSlot(int location) {
        mSlotList.remove(location);
    }

    /**
     * Get the grammar constraint name.
     *
     * @return the name of the grammar constraint.
     */
    public String getName() {
        return this.mName;
    }

    /**
     * Get the list of the grammar constraint slots.
     *
     * @return the list of the grammar constraint {@link Slot} .
     */
    public List<Slot> getSlotList() {
        return this.mSlotList;
    }

    /**
     * Grammar to String.
     *
     * @return the grammar string.
     */
    public String toGrammarString() {
        if (mSlotList.isEmpty()) {
            throw new IllegalStateException("The slot list cannot be empty!");
        }
        StringBuilder grammarConstraintBuilder = new StringBuilder();
        for (Slot slot : mSlotList) {
            grammarConstraintBuilder.append(slot.toOptionalString());
        }
        return grammarConstraintBuilder.toString();
    }

    /**
     * Read data from the specified parcel.
     *
     * @param in the specified parcel data.
     */
    public void readFromParcel(Parcel in) {
        mName = in.readString();
        mSlotList = in.createTypedArrayList(Slot.CREATOR);
    }

    /**
     * Parcelable interface methods.
     *
     * @return a bitmask indicating the set of special object types marshaled by this Parcelable object instance.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    /**
     * Write the grammar constraint to the specified parcel. To restore a grammar constraint from
     * a parcel, use readFromParcel().
     *
     * @param dest the specified parcel to be written to.
     */
    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mName);
        dest.writeTypedList(mSlotList);
    }

    @Override
    public String toString() {
        return "GrammarConstraint{" +
                "mName='" + mName + '\'' +
                ", mSlotList=" + mSlotList +
                '}';
    }
}
