package com.segway.robot.sdk.voice;

import android.content.Context;
import android.support.annotation.NonNull;

import com.segway.robot.sdk.base.bind.BindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.voice.tts.TtsListener;

public class Speaker {
    private static final String TAG = Speaker.class.getSimpleName();
    private static Speaker mSpeaker;
    private SpeakerManager mSpeakerManager;
    private boolean mBindControllerInit;
    private BindController mBindController = new ForegroundBindController();

    private Speaker() {
        mSpeakerManager = new SpeakerManager();
    }

    /**
     * Create a Speaker instance.
     * @return an instance of the Speaker.
     */
    public static synchronized Speaker getInstance() {
        if (mSpeaker == null) {
            mSpeaker = new Speaker();
        }
        return mSpeaker;
    }

    /**
     * Bind to the TextToSpeechService.
     * @param context the context that connects to the TextToSpeechService.
     * @param listener a listener that provides the connection state.
     * @return true if the connection succeeds, false if the connection fails.
     */
    public synchronized boolean bindService(@NonNull Context context, @NonNull ServiceBinder.BindStateListener listener) {
        if (!mBindControllerInit) {
            mBindControllerInit = true;
            mBindController.init(context, mSpeakerManager);
        }
        if (mBindController.bindAcceptable()) {
            return mSpeakerManager.bindService(context, listener);
        }
        return false;
    }

    /**
     * Unbind from the TextToSpeechService.
     */
    public synchronized void unbindService() {
        mSpeakerManager.unbindService();
    }

    /**
     * Get current using language
     */
    @Languages.Language
    public int getLanguage() throws  VoiceException {
        return mSpeakerManager.getLanguage();
    }

    /**
     * Make the text to speech.
     * @param text the speech content.
     * @param ttsListener A {@link TtsListener} which provides callbacks for certain text-to-speech (TTS) generation events.
     * @throws VoiceException
     */
    public synchronized void speak(@NonNull String text, @NonNull TtsListener ttsListener) throws VoiceException {
        mSpeakerManager.speak(text, ttsListener);
    }

    /**
     * Stop the speech.
     * @throws VoiceException
     */
    public void stopSpeak() throws VoiceException {
        mSpeakerManager.stopSpeak();
    }

    /**
     * Wait until the speech finishes.
     * @param timeout the timeout in millisecond.
     * @return true for timeout, false for else.
     */
    public boolean waitForSpeakFinish(long timeout) throws VoiceException {
        return mSpeakerManager.waitForSpeakFinish(timeout);
    }
}
