package com.segway.robot.sdk.voice;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.util.Log;

import com.segway.robot.sdk.base.bind.BindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.voice.tts.ITtsListener;
import com.segway.robot.sdk.voice.tts.TtsListener;

import java.util.concurrent.atomic.AtomicBoolean;

public class Speaker {
    private static final String TAG = Speaker.class.getSimpleName();
    private static Speaker mSpeaker;
    private SpeakerManager mSpeakerManager;
    private boolean mBindControllerInit;
    private BindController mBindController = new ForegroundBindController();

    private Speaker() {
        mSpeakerManager = new SpeakerManager();
    }

    /**
     * Create a Speaker instance.
     * @return an instance of the Speaker.
     */
    public static synchronized Speaker getInstance() {
        if (mSpeaker == null) {
            mSpeaker = new Speaker();
        }
        return mSpeaker;
    }

    /**
     * Bind to the TextToSpeechService.
     * @param context the context that connects to the TextToSpeechService.
     * @param listener a listener that provides the connection state.
     * @return true if the connection succeeds, false if the connection fails.
     */
    public synchronized boolean bindService(@NonNull Context context, @NonNull ServiceBinder.BindStateListener listener) {
        if (!mBindControllerInit) {
            mBindControllerInit = true;
            mBindController.init(context, mSpeakerManager);
        }
        if (mBindController.bindAcceptable()) {
            return mSpeakerManager.bindService(context, listener);
        }
        return false;
    }

    /**
     * Unbind from the TextToSpeechService.
     */
    public synchronized void unbindService() {
        mSpeakerManager.unbindService();
    }

    /**
     * Get current using language
     */
    @Languages.Language
    public int getLanguage() throws RemoteException, VoiceException {
        return mSpeakerManager.getLanguage();
    }

    /**
     * Make the text to speech.
     * @param text the speech content.
     * @param ttsListener A {@link TtsListener} which provides callbacks for certain text-to-speech (TTS) generation events.
     * @throws VoiceException
     * @throws RemoteException
     */
    public synchronized void speak(@NonNull String text, @NonNull TtsListener ttsListener) throws VoiceException, RemoteException {
        mSpeakerManager.speak(text, ttsListener);
    }

    /**
     * Stop the speech.
     * @throws RemoteException
     * @throws VoiceException
     */
    public void stopSpeak() throws RemoteException, VoiceException {
        mSpeakerManager.stopSpeak();
    }

    /**
     * Wait until the speech finishes.
     * @param timeout the timeout in millisecond.
     * @return true for timeout, false for else.
     * @throws RemoteException
     */
    public boolean waitForSpeakFinish(long timeout) throws RemoteException {
        return mSpeakerManager.waitForSpeakFinish(timeout);
    }
}
