package com.segway.robot.sdk.voice;

import android.content.Context;
import android.os.RemoteException;
import android.support.annotation.NonNull;

import com.segway.robot.sdk.base.bind.BindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.voice.audiodata.RawDataListener;
import com.segway.robot.sdk.voice.grammar.GrammarConstraint;
import com.segway.robot.sdk.voice.recognition.RecognitionListener;
import com.segway.robot.sdk.voice.recognition.WakeupListener;

public class Recognizer {
    private static Recognizer mRecognizer;
    private RecognizerManager mRecognizerManager;
    private boolean mBindControllerInit;
    private BindController mBindController = new ForegroundBindController();

    private Recognizer() {
        mRecognizerManager = new RecognizerManager();
    }

    /**
     * Create a Recognizer instance.
     * @return an instance of the Recognizer.
     */
    public synchronized static Recognizer getInstance() {
        if (mRecognizer == null) {
            mRecognizer = new Recognizer();
        }
        return mRecognizer;
    }

    /**
     * Bind to the RecognitionService.
     * @param context the context that connects to the RecognitionService.
     * @param listener a listener that provides the connection state.
     * @return true if the connection succeeds, false if the connection fails.
     */
    public synchronized boolean bindService(@NonNull Context context, @NonNull ServiceBinder.BindStateListener listener){
        if (!mBindControllerInit) {
            mBindControllerInit = true;
            mBindController.init(context, mRecognizerManager);
        }
        if (mBindController.bindAcceptable()) {
            return mRecognizerManager.bindService(context, listener);
        }
        return false;
    }

    /**
     * Unbind from the RecognitionService.
     */
    public synchronized void unbindService() {
        mRecognizerManager.unbindService();
    }

    /**
     * Get the current language.
     * @return current language.
     */
    @Languages.Language
    public int getLanguage() throws RemoteException, VoiceException {
        return mRecognizerManager.getLanguage();
    }

    /**
     * Enable or disable the beamforming.
     * @param enable True means enable, false means disable.
     */
    public void beamForming(boolean enable) throws VoiceException, RemoteException {
        mRecognizerManager.beamForming(enable);
    }

    /**
     *
     * sample format
     {
        "name": "play_music",
        "slotList": [
            {
                "name": "play_cmd",
                "isOptional": false,
                "word": [
                    "play",
                    "play music"
                    ]
            },
            {
                "name": "song",
                "isOptional": false,
                "word": [
                    "happy birthday",
                    "jingle bells"
                    ]
            }
        ]
     }
     * @param json grammar constraint a JSON string with the legal JSON format.
     * @return a new grammar constraint which is converted from the JSON string.
     * @throws VoiceException
     */
    public GrammarConstraint createGrammarConstraint(@NonNull String json) throws VoiceException {
        if (json.isEmpty() || json.equals("")) {
            throw new IllegalArgumentException("The input parameter cannot be empty!");
        }
        // TODO: 2016/9/30 verify
        GrammarJsonValidator grammarJsonValidator = new GrammarJsonValidator();
        boolean legal = grammarJsonValidator.validate(json);
        if (!legal) {
            throw new VoiceException("The input JSON character string is illegal!");
        }
        return mRecognizerManager.createGrammarConstraint(json);
    }

    /**
     * Add a grammar constraint to the recognizer.
     * @param grammarConstraint the {@link GrammarConstraint} to be added to the recognition grammar.
     */
    public void addGrammarConstraint(@NonNull GrammarConstraint grammarConstraint) throws VoiceException, RemoteException {
        mRecognizerManager.addGrammarConstraint(grammarConstraint);
    }

    /**
     * Remove a grammar constraint from the recognizer.
     * @param grammarConstraint the {@link GrammarConstraint} will be removed to the recognition grammar.
     */
    public void removeGrammarConstraint(@NonNull GrammarConstraint grammarConstraint) throws VoiceException, RemoteException {
        mRecognizerManager.removeGrammarConstraint(grammarConstraint);
    }

    /**
     * Start to recognize.
     * @param wakeupListener A {@link WakeupListener} to be notified when a wake-up phrase is detected,
     *                       or the error description if an error occurs.
     * @param recognitionListener A {@link RecognitionListener} that provides the results,
     *                            or the error description if an error occurs.
     * @throws VoiceException
     * @throws RemoteException
     */
    public void startRecognition(@NonNull WakeupListener wakeupListener, @NonNull RecognitionListener recognitionListener) throws VoiceException, RemoteException {
        mRecognizerManager.startRecognition(wakeupListener, recognitionListener);
    }

    /**
     * Terminate the current recognition.
     * @throws VoiceException
     * @throws RemoteException
     */
    public void stopRecognition() throws VoiceException, RemoteException {
        mRecognizerManager.stopRecognition();
    }

    /**
     * Make beamformer work and get the beamforming raw data at the same time.
     * @param listener the {@link RawDataListener} that gives the beamforming audio raw data.
     */
    public void startBeamFormingListen(@NonNull RawDataListener listener) throws VoiceException, RemoteException {
        mRecognizerManager.startBeamFormingListen(listener);
    }

    /**
     * Stop beamforming.
     */
    public void stopBeamFormingListen() throws RemoteException, VoiceException {
        mRecognizerManager.stopBeamFormingListen();
    }
}
