package com.segway.robot.sdk.perception.sensor;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;

import com.segway.robot.sdk.base.bind.BlankBindController;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;

/**
 * Created by sunguangshan on 2016/9/8.
 */

public class Sensor implements ServiceBinder {
    private static final String TAG = "Sensor";
    private static final String SERVICE_CLASS_NAME = "com.segway.robot.host.coreservice.locomotionservice.RobotSensorService";
    private static final String SERVICE_PACKAGE_NAME = "com.segway.robot.host.coreservice.locomotionservice";
    private static Sensor mSensor;

    private IRobotSensorService mIRobotSensorService;
    private BindStateListener mBindStateListener;

    public static final int UWB_NAVIGATION_MODE = 0;

    public static final int UWB_FOLLOW_MODE = 1;

    ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            mIRobotSensorService = IRobotSensorService.Stub.asInterface(service);
            // check version
            Version version = VersionInfo.getVersion();
            try {
                Version serviceVersion = mIRobotSensorService.getVersion();
                version.check("VersionInfo", serviceVersion);
            } catch (RemoteException e) {
                String error = "Cannot get Service version, err = " + e.getMessage();

                bindError(e, error);
                return;
            } catch (VersionMismatchException e) {
                String error = "Version mismatch: " + e.getMessage();

                bindError(e, error);
                return;
            }
            mBindStateListener.onBind();
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            mIRobotSensorService = null;
            mBindStateListener.onUnbind("Service Disconnected");
        }
    };
    private Context mContext;
    private boolean isBind = false;
    private boolean isControllerSet = false;

    private Sensor() {
    }

    /**
     * Gets an instance of the RobotInfraredManager.
     *
     * @return the RobotInfraredManager instance.
     */
    public static synchronized Sensor getInstance() {
        if (mSensor == null) {
            mSensor = new Sensor();
        }
        return mSensor;
    }

    /**
     * Connect to the robot head service. This method must be called before using the infrared control APIs.
     *
     * @param context
     * @return true if the connection succeeds.
     */
    public synchronized boolean connectService(Context context) {
        Intent serviceIntent = new Intent();
        serviceIntent.setAction(SERVICE_CLASS_NAME);
        serviceIntent.setPackage(SERVICE_PACKAGE_NAME);
        serviceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        return context.bindService(serviceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
    }


    @Override
    public synchronized boolean bindService(Context context, BindStateListener listener) {
        if (context == null || listener == null) {
            throw new IllegalArgumentException("context and listener can't null");
        }
        if (isBind) {
            return true;
        }
        mBindStateListener = listener;
        if (isControllerSet == false) {
            isControllerSet = true;
            new BlankBindController().init(context, this);
        }
        mContext = context.getApplicationContext();
        return isBind = connectService(mContext);
    }

    @Override
    public synchronized void unbindService() {
        if (!isBind) {
            return;
        }
        isBind = false;
        mContext.unbindService(mServiceConnection);
    }

    @Override
    public boolean isBind() {
        return isBind;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mBindStateListener;
    }

    public UltrasonicData getUltrasonicDistance() {
        try {
            return mIRobotSensorService.getUltrasonicDistance();
        } catch (RemoteException e) {
            e.printStackTrace();
            throw new RobotSensorException("getUltrasonicDistance error", e);
        }
    }

    public InfraredData getInfraredDistance() {
        try {
            return mIRobotSensorService.getInfraredDistance();
        } catch (RemoteException e) {
            e.printStackTrace();
            throw new RobotSensorException("getInfraredDistance error", e);
        }
    }

    public RobotTotalInfo getRobotTotalInfo() {
        try {
            return mIRobotSensorService.getRobotTotalInfo();
        } catch (RemoteException e) {
            throw new RobotSensorException("getRobotTotalInfo error", e);
        }
    }

    public UwbEvent getUwbEvent() {
        try {
            return mIRobotSensorService.getUwbEvent();
        } catch (RemoteException e) {
            throw new RobotSensorException("getRobotTotalInfo error", e);
        }
    }

    private void bindError(Exception e, String error) {
        mBindStateListener.onUnbind(error);
        Logger.e(TAG, error, e);
        isBind = false;
        mContext.unbindService(mServiceConnection);
    }

    public void setUwbMode(int mode) {
        try {
            mIRobotSensorService.setUwbMode(mode);
        } catch (RemoteException e) {
            throw new RobotSensorException("setUwbMode error", e);
        }
    }

    public int getUwbMode() {
        try {
            return mIRobotSensorService.getUwbMode();
        } catch (RemoteException e) {
            throw new RobotSensorException("getUwbMode error", e);
        }
    }

    public UwbEvent[] getUwbEventArray() {
        try {
            return mIRobotSensorService.getUwbEventArray();
        } catch (RemoteException e) {
            throw new RobotSensorException("getUwbEventArray error", e);
        }
    }
}
