package com.segway.robot.sdk.perception.sensor;

import android.os.Parcel;
import android.os.Parcelable;

import com.segway.robot.algo.Pose2D;
import com.segway.robot.sdk.locomotion.head.Angle;
import com.segway.robot.sdk.locomotion.sbv.BasePose;
import com.segway.robot.sdk.locomotion.sbv.BaseTicks;
import com.segway.robot.sdk.locomotion.sbv.BaseWheelInfo;

/**
 * This class include robot all sensor value and some robot (head and base) position info.
 */
public class RobotAllSensors implements Parcelable {
    private final Angle mHeadWorldYaw;
    private final Angle mHeadWorldPitch;
    private final Angle mHeadWorldRoll;
    private final Angle mHeadJointYaw;
    private final Angle mHeadJointPitch;
    private final Angle mHeadJointRoll;
    private final BaseTicks mBaseTicks;
    private final UltrasonicData mUltrasonicData;
    private final InfraredData mInfraredData;
    private final BasePose mBasePose;
    private final Pose2D mPose2D;
    private final BaseWheelInfo mBaseWheelInfo;

    public RobotAllSensors(Angle headWorldYaw, Angle headWorldPitch, Angle headWorldRoll,
                           Angle headJointYaw, Angle headJointPitch, Angle headJointRoll,
                           BaseTicks baseTicks, UltrasonicData ultrasonicData, InfraredData infraredData,
                           BasePose basePose, Pose2D pose2D, BaseWheelInfo baseWheelInfo) {
        mHeadWorldYaw = headWorldYaw;
        mHeadWorldPitch = headWorldPitch;
        mHeadWorldRoll = headWorldRoll;
        mHeadJointYaw = headJointYaw;
        mHeadJointPitch = headJointPitch;
        mHeadJointRoll = headJointRoll;
        mBaseTicks = baseTicks;
        mUltrasonicData = ultrasonicData;
        mInfraredData = infraredData;
        mBasePose = basePose;
        mPose2D = pose2D;
        mBaseWheelInfo = baseWheelInfo;
    }

    protected RobotAllSensors(Parcel in) {
        mHeadWorldYaw = in.readParcelable(Angle.class.getClassLoader());
        mHeadWorldPitch = in.readParcelable(Angle.class.getClassLoader());
        mHeadWorldRoll = in.readParcelable(Angle.class.getClassLoader());
        mHeadJointYaw = in.readParcelable(Angle.class.getClassLoader());
        mHeadJointPitch = in.readParcelable(Angle.class.getClassLoader());
        mHeadJointRoll = in.readParcelable(Angle.class.getClassLoader());
        mBaseTicks = in.readParcelable(BaseTicks.class.getClassLoader());
        mUltrasonicData = in.readParcelable(UltrasonicData.class.getClassLoader());
        mInfraredData = in.readParcelable(InfraredData.class.getClassLoader());
        mBasePose = in.readParcelable(BasePose.class.getClassLoader());
        mPose2D = in.readParcelable(Pose2D.class.getClassLoader());
        mBaseWheelInfo = in.readParcelable(BaseWheelInfo.class.getClassLoader());
    }

    public static final Creator<RobotAllSensors> CREATOR = new Creator<RobotAllSensors>() {
        @Override
        public RobotAllSensors createFromParcel(Parcel in) {
            return new RobotAllSensors(in);
        }

        @Override
        public RobotAllSensors[] newArray(int size) {
            return new RobotAllSensors[size];
        }
    };

    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getWorldYaw()
     * @return angle
     */
    public Angle getHeadWorldYaw() {
        return mHeadWorldYaw;
    }

    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getWorldPitch()
     * @return angle
     */
    public Angle getHeadWorldPitch() {
        return mHeadWorldPitch;
    }

    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getWorldRoll()
     * @return angle
     */
    public Angle getHeadWorldRoll() {
        return mHeadWorldRoll;
    }

    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getHeadJointYaw()
     * @return angle
     */
    public Angle getHeadJointYaw() {
        return mHeadJointYaw;
    }


    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getHeadJointPitch()
     * @return angle
     */
    public Angle getHeadJointPitch() {
        return mHeadJointPitch;
    }

    /**
     * see com.segway.robot.sdk.locomotion.head.Head#getHeadJointRoll()
     * @return angle
     */
    public Angle getHeadJointRoll() {
        return mHeadJointRoll;
    }

    /**
     * Get the robot base ticks.
     * 90 ticks stand for 1 round of robot wheel.
     * see BaseTicks
     * @return the current BaseTicks.
     */
    public BaseTicks getBaseTicks() {
        return mBaseTicks;
    }

    /**
     * @see Sensor#getUltrasonicDistance()
     * @see UltrasonicData
     * @return {@link UltrasonicData}
     */
    public UltrasonicData getUltrasonicData() {
        return mUltrasonicData;
    }

    /**
     * @see Sensor#getInfraredDistance()
     * @see InfraredData
     * @return {@link InfraredData}
     */
    public InfraredData getInfraredData() {
        return mInfraredData;
    }

    /**
     * see com.segway.robot.sdk.locomotion.sbv.BasePose
     * @return base pose
     */
    public BasePose getBasePose() {
        return mBasePose;
    }

    /**
     * see Pose2D
     * @return pose2D
     */
    public Pose2D getPose2D() {
        return mPose2D;
    }

    /**
     * see BaseWheelInfo
     * @return baseWheelInfo
     */
    public BaseWheelInfo getBaseWheelInfo(){ return mBaseWheelInfo; }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeParcelable(mHeadWorldYaw, flags);
        dest.writeParcelable(mHeadWorldPitch, flags);
        dest.writeParcelable(mHeadWorldRoll, flags);
        dest.writeParcelable(mHeadJointYaw, flags);
        dest.writeParcelable(mHeadJointPitch, flags);
        dest.writeParcelable(mHeadJointRoll, flags);
        dest.writeParcelable(mBaseTicks, flags);
        dest.writeParcelable(mUltrasonicData, flags);
        dest.writeParcelable(mInfraredData, flags);
        dest.writeParcelable(mBasePose, flags);
        dest.writeParcelable(mPose2D, flags);
        dest.writeParcelable(mBaseWheelInfo, flags);
    }

    @Override
    public String toString() {
        return "RobotTotalInfo:" +
                " HeadWorldYaw=" + mHeadWorldYaw.toString() +
                ", HeadWorldPitch=" + mHeadWorldPitch.toString() +
                ", HeadWorldRoll=" + mHeadWorldRoll.toString() +
                ", HeadJointYaw=" + mHeadJointYaw.toString() +
                ", HeadJointPitch=" + mHeadJointPitch.toString() +
                ", HeadJointRoll=" + mHeadJointRoll.toString() +
                ", BaseTicks=" + mBaseTicks.toString() +
                ", UltrasonicData=" + mUltrasonicData.toString() +
                ", InfraredData=" + mInfraredData.toString() +
                ", BasePose=" + mBasePose.toString() +
                ", Pose2D=" + mPose2D.toString() +
                ", BaseWheelInfo=" + mBaseWheelInfo.toString() +
                ", time diff=" + (System.currentTimeMillis() * 1000 - mHeadWorldYaw.getTimestamp());
    }
}
