package com.segway.robot.sdk.perception.sensor;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.os.IBinder;
import android.os.RemoteException;

import com.segway.robot.algo.Pose2D;
import com.segway.robot.algo.tf.AlgoTfData;
import com.segway.robot.algo.tf.AlgoTfRequest;
import com.segway.robot.sdk.base.bind.ForegroundBindController;
import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.base.version.Version;
import com.segway.robot.sdk.base.version.VersionMismatchException;

import java.util.List;

/**
 * Created by sgs on 2017/10/23.
 */

class SensorManager extends ISensor implements ServiceBinder {

    private static final String TAG = "SensorManager";
    private static final String SERVICE_CLASS_NAME = "com.segway.robot.host.coreservice.locomotionservice.RobotSensorService";
    private static final String SERVICE_PACKAGE_NAME = "com.segway.robot.host.coreservice.locomotionservice";

    private IRobotSensorService mIRobotSensorService;
    private BindStateListener mBindStateListener;
    private Context mContext;
    private boolean isBind = false;

    SensorManager() {
    }

    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder service) {
            mIRobotSensorService = IRobotSensorService.Stub.asInterface(service);
//            // check version
//            Version version = VersionInfo.getVersion();
//            try {
//                Version serviceVersion = mIRobotSensorService.getVersion();
//                version.check("VersionInfo", serviceVersion);
//            } catch (RemoteException e) {
//                String error = "Cannot get Service version, err = " + e.getMessage();
//
//                bindError(e, error);
//                return;
//            } catch (VersionMismatchException e) {
//                String error = "Version mismatch: " + e.getMessage();
//
//                bindError(e, error);
//                return;
//            }
            mBindStateListener.onBind();
            isBind = true;
        }

        @Override
        public void onServiceDisconnected(ComponentName name) {
            mIRobotSensorService = null;
            mBindStateListener.onUnbind("Service Disconnected");
            isBind = false;
        }
    };

    /**
     * Connect to the robot head service. This method must be called before using the infrared control APIs.
     *
     * @param context
     * @return true if the connection succeeds.
     */
    synchronized boolean connectService(Context context) {
        Intent serviceIntent = new Intent();
        serviceIntent.setAction(SERVICE_CLASS_NAME);
        serviceIntent.setPackage(SERVICE_PACKAGE_NAME);
        serviceIntent.setClassName(SERVICE_PACKAGE_NAME, SERVICE_CLASS_NAME);
        return context.bindService(serviceIntent, mServiceConnection, Context.BIND_AUTO_CREATE);
    }


    private void bindError(Exception e, String error) {
        mBindStateListener.onUnbind(error);
        Logger.e(TAG, error, e);
        isBind = false;
        mContext.unbindService(mServiceConnection);
    }

    @Override
    public boolean bindService(Context context, BindStateListener listener) {
        if (context == null || listener == null) {
            throw new IllegalArgumentException("context and listener can't null");
        }
        if (isBind) {
            return true;
        }
        mBindStateListener = listener;
        mContext = context.getApplicationContext();
        return connectService(mContext);
    }

    @Override
    public void unbindService() {
        if (!isBind) {
            return;
        }
        isBind = false;
        mContext.unbindService(mServiceConnection);
    }

    @Override
    public boolean isBind() {
        return isBind;
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mBindStateListener;
    }

    @Override
    UltrasonicData getUltrasonicDistance() {
        try {
            return mIRobotSensorService.getUltrasonicDistance();
        } catch (RemoteException e) {
            e.printStackTrace();
            throw new RobotSensorException("getUltrasonicDistance error", e);
        }
    }

    @Override
    InfraredData getInfraredDistance() {
        try {
            return mIRobotSensorService.getInfraredDistance();
        } catch (RemoteException e) {
            throw new RobotSensorException("getInfraredDistance error", e);
        }
    }

    @Override
    RobotTotalInfo getRobotTotalInfo() {
        try {
            return mIRobotSensorService.getRobotTotalInfo();
        } catch (RemoteException e) {
            throw new RobotSensorException("getRobotTotalInfo error", e);
        }
    }

    @Override
    AlgoTfData getTfData(String tgt, String src, long lookupTimestamp, int timeTrhMs) {
        try {
            return mIRobotSensorService.getTfData(tgt, src, lookupTimestamp, timeTrhMs);
        } catch (RemoteException e) {
            throw new RobotSensorException("getTfData error", e);
        }
    }

    @Override
    AlgoTfData getTfData(AlgoTfRequest req) {
        try {
            return mIRobotSensorService.getTfData2(req);
        } catch (RemoteException e) {
            throw new RobotSensorException("get TfData2 error", e);
        }
    }

    @Override
    List<AlgoTfData> getMassiveTfData(List<AlgoTfRequest> requestList) {
        try {
            return mIRobotSensorService.getMassiveTfData(requestList);
        } catch (RemoteException e) {
            throw new RobotSensorException("get massive TfData error", e);
        }
    }


    @Override
    boolean setCustomizationTF(AlgoTfData tfData) {
        try {
            Logger.v(TAG, "setCustomizationTF: " + tfData);
            return mIRobotSensorService.setCustmizitionTF(tfData);
        } catch (RemoteException e) {
            throw new RobotSensorException("set Customization tf error", e);
        }
    }

    @Override
    public int resetFrameData(String frameName) {
        try {
            return mIRobotSensorService.resetFrameData(frameName);
        } catch (RemoteException e) {
            throw new RobotSensorException("set Customization tf error", e);
        }
    }


    @Override
    RobotAllSensors getRobotAllSensors() {
        try {
            return mIRobotSensorService.getRobotAllSensors();
        } catch (RemoteException e) {
            throw new RobotSensorException("getRobotAllSensors error", e);
        }
    }

    @Override
    SensorData[] getSensorData(int type) {
        try {
            return mIRobotSensorService.getSensorData(type);
        } catch (RemoteException e) {
            throw new RobotSensorException("getSensorData error", e);
        }
    }

    @Override
    List<SensorData> querySensorData(List queryList) {
        try {
            return mIRobotSensorService.querySensorData(queryList);
        } catch (RemoteException e) {
            throw new RobotSensorException("querySensorData error", e);
        }
    }

    @Override
    Pose2D sensorDataToPose2D(SensorData sensorData) {
        if (sensorData == null) {
            throw new NullPointerException("sensor data is null");
        }
        if (sensorData.floatData == null || sensorData.floatData.length != 5) {
            throw new IllegalStateException("sensor data illegal ");
        }
        float data[] = sensorData.floatData;
        Pose2D pose2D = new Pose2D(data[0], data[1], data[2], data[3], data[4], sensorData.getTimestamp());
        return pose2D;
    }
}
