package com.segway.robot.mobile.sdk.connectivity;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.text.TextUtils;

import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;

import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class provides methods for the mobile applications to register/unregister and bind to/unbind from the
 * connection service in robot.
 */
public class MobileMessageRouter extends MessageRouter {
    private static final String TAG = "MobileMessageRouter";
    private static MobileMessageRouter mMobileMessageRouter = null;
    private Context mContext = null;
    private MobileMessageSocket mMobileMessageSocket = null;
    private MessageConnectionListener mMessageConnectionListener = null;
    private String mIp = "";

    private MobileMessageRouter() {

    }

    /**
     * Initialize a singleton {@code MobileMessageRouter} before invoking all the methods in MobileMessageRouter.
     *
     * @return a MobileMessageRouter instance.
     */
    public static synchronized MobileMessageRouter getInstance() {
        if (mMobileMessageRouter == null) {
            mMobileMessageRouter = new MobileMessageRouter();
        }
        return mMobileMessageRouter;
    }

    /**
     * Register a MessageConnectionListener in the message handler. When the robot application connects to
     * the specific mobile application, the MessageConnectionListener will invoke the onConnectionCreated() interface
     * to create a MessageConnection instance to get the {@link MobileMessageConnection} instance.
     *
     * @param messageConnectionListener the MessageConnectionListener to be registered.
     * @throws Exception If the input argument is illegal, throw IllegalArgumentException,
     *                   otherwise display the other detailed Exception.
     */
    @Override
    public synchronized void register(MessageConnectionListener messageConnectionListener) throws Exception {
        if (messageConnectionListener == null) {
            throw new IllegalArgumentException("MessageConnectionListener can't be null!");
        }

        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }
        mMessageConnectionListener = messageConnectionListener;
        mMobileMessageSocket.setConnectionListener(mMessageConnectionListener);
    }

    /**
     * Set the connection IP. If the input argument IP is illegal, then throw IllegalArgumentException.
     *
     * @param ip the IP of the connection to be set.
     */
    public synchronized void setConnectionIp(String ip) {
        if (TextUtils.isEmpty(ip)) {
            throw new IllegalArgumentException("IP can't be null!");
        }
        if (!isIpIllegal(ip)) {
            throw new IllegalArgumentException("IP isn't illegal");
        }
        mIp = ip;
    }

    /**
     * Unregister to remove the MessageConnectionListener from the MobileMessageRouter instance.
     */
    @Override
    public synchronized void unregister() {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }
        if (mMessageConnectionListener == null) {
            throw new IllegalStateException("MessageConnectionListener isn't registered");
        }
        mMobileMessageSocket.removeConnectionListener();
        mMessageConnectionListener = null;
    }

    /**
     * Bind to the robot connection service. If the input argument is illegal, throw IllegalArgumentException.
     * It isn't recommended to call this synchronous method in the main thread.
     *
     * @param context  any Android context.
     * @param listener the BindStateLister to be used to listen to the bind state of the connection service.
     * @return false.
     */
    @Override
    public boolean bindService(Context context, BindStateListener listener) {
        if (context == null) {
            throw new IllegalArgumentException("Context can't be null!");
        }
        if (listener == null) {
            throw new IllegalArgumentException("BindStateListener can't be null!");
        }
        if (mMobileMessageSocket == null) {
            mMobileMessageSocket = new MobileMessageSocket();
        }
        mContext = context.getApplicationContext();
        try {
            setMetaData(mContext.getPackageName(), getMetaData(mContext));
        } catch (MobileException e) {
            e.printStackTrace();
        }
        mMobileMessageSocket.initSocketClient(mContext.getPackageName(), mIp, listener);
        return mMobileMessageSocket.connect();
    }

    /**
     * Unbind from the robot connection service.
     */
    @Override
    public void unbindService() {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }

        mMobileMessageSocket.disconnect();
        mMobileMessageSocket = null;
    }


    protected void sendBufferMessage(String to, BufferMessage message) throws MobileException {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }
        mMobileMessageSocket.sendMessage(to, message);
    }

    protected void sendStringMessage(String to, StringMessage message) throws MobileException {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }
        mMobileMessageSocket.sendMessage(to, message);
    }

    protected void setListenersReady(String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) throws MobileException {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind");
        }
        mMobileMessageSocket.setListenersReady(mContext.getPackageName(), to, connectionStateListener, messageListener);
    }

    protected synchronized Set getMetaData(Context context) throws MobileException {
        if (context == null) {
            throw new IllegalArgumentException("The context can't be null!");
        }

        PackageManager packageManager = context.getPackageManager();
        if (packageManager == null) {
            throw new IllegalArgumentException("Cannot get PackageManager.");
        }

        ApplicationInfo applicationInfo;
        try {
            applicationInfo = packageManager.getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            throw MobileException.getPackageNameNotFound(e.getMessage(), e);
        }
        if (applicationInfo == null) {
            throw new IllegalArgumentException("Cannot get ApplicationInfo.");
        }

        Bundle data = applicationInfo.metaData;
        if (data == null) {
            throw new IllegalArgumentException("The application metadata is illegal!");
        }

        Set<String> metaData = new HashSet<>();
        for (String key : data.keySet()) {
            String value = data.getString(key);
            if (!TextUtils.isEmpty(value)) {
                metaData.add(value);
            }
        }
        return metaData;
    }

    private void setMetaData(String packageName, Set set) {
        if (mMobileMessageSocket == null) {
            throw new IllegalStateException("MobileMessageRouter isn't bind!");
        }
        if (set == null) {
            throw new IllegalArgumentException("Get metadata failed!");
        }
        mMobileMessageSocket.setMetaData(packageName, set);
    }

    private boolean isIpIllegal(String ip) {
        String ipCheck = "(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])\\.(\\d{1,2}|1\\d\\d|2[0-4]\\d|25[0-5])";
        Pattern pattern = Pattern.compile(ipCheck);
        Matcher matcher = pattern.matcher(ip);
        return matcher.matches();
    }
}