package com.segway.robot.mobile.sdk.connectivity;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Message;
import android.util.Log;

import com.segway.robot.sdk.base.bind.ServiceBinder;
import com.segway.robot.sdk.baseconnectivity.ByteMessage;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

import org.apache.mina.core.future.ConnectFuture;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.filter.codec.serialization.ObjectSerializationCodecFactory;
import org.apache.mina.filter.keepalive.KeepAliveFilter;
import org.apache.mina.filter.keepalive.KeepAliveMessageFactory;
import org.apache.mina.transport.socket.nio.NioSocketConnector;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.InetSocketAddress;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Created by gaofeng on 2016/4/20.
 */
public class MobileMessageSocket implements Handler.Callback {

    private static final String TAG = "MobileMessageSocket";
    private MobileMessageHandler mMobileMessageHandler = null;
    private NioSocketConnector mConnector = null;
    private IoSession mIoSession = null;
    private String mPackageName = null;
    private String mIp = null;
    private HandlerThread mHandlerThread = null;
    private Handler mHandler = null;
    private boolean isSessionOpened = false;
    private final AtomicBoolean isConnectionFinished = new AtomicBoolean(false);
    private ServiceBinder.BindStateListener mBindStateListener = null;
    private boolean isRepeatedConnect = false;

    private static final int PORT = 7798;
    private static final int CONNECTION_TIMEOUT = 5 * 1000;//5s
    private static final int CONNECT = 1;
    private static final int DISCONNECT = 2;

    protected MobileMessageSocket() {
        Log.d(TAG, "MobileMessageSocket: init");
        mMobileMessageHandler = new MobileMessageHandler();
        mHandlerThread = new HandlerThread("ConnectThread");
        mHandlerThread.start();
        mHandler = new Handler(mHandlerThread.getLooper(), this);
        // Create ClientIo
        mConnector = new NioSocketConnector();
        ObjectSerializationCodecFactory codecFactory = new ObjectSerializationCodecFactory();
        codecFactory.setDecoderMaxObjectSize(MessageUtil.MAX_CODER_LENGTH);
        codecFactory.setEncoderMaxObjectSize(MessageUtil.MAX_CODER_LENGTH);
        mConnector.getFilterChain().addLast("codec", new ProtocolCodecFilter(codecFactory));
        KeepAliveMessageFactory heartBeatFactory = new MobileKeepAliveImpl();
        KeepAliveFilter heartBeatFilter = new KeepAliveFilter(heartBeatFactory, IdleStatus.BOTH_IDLE);
        heartBeatFilter.setForwardEvent(true);
        heartBeatFilter.setRequestInterval(MessageUtil.HEART_BEAT_INTERVAL);//mina default 60s
        heartBeatFilter.setRequestTimeout(MessageUtil.HEART_BEAT_TIMEOUT);//mina default 30s
        mConnector.getFilterChain().addLast("heart", heartBeatFilter);
        // set time out
        mConnector.setConnectTimeoutMillis(CONNECTION_TIMEOUT);
        mConnector.getSessionConfig().setWriteTimeout(MessageUtil.WRITE_TIMEOUT);
        mConnector.setHandler(mMobileMessageHandler);

        setSessionStateChange(sessionStateCallback);
    }

    private SessionStateCallback sessionStateCallback = new SessionStateCallback() {
        @Override
        public void onSessionClosed() {
            synchronized (isConnectionFinished) {
                mIoSession = null;
                isSessionOpened = false;
                isConnectionFinished.set(true);
                isConnectionFinished.notifyAll();
            }
            if (isRepeatedConnect) {
                mBindStateListener.onUnbind(MessageError.REPEATED_CONNECTION);
            } else {
                mBindStateListener.onUnbind(MessageError.SESSION_CLOSED);
            }
        }

        @Override
        public void onSessionOpened(IoSession ioSession) {
            synchronized (isConnectionFinished) {
                isSessionOpened = true;
                isRepeatedConnect = false;
                mBindStateListener.onBind();
                mHandler.removeMessages(CONNECT, mIp);
                mIoSession = ioSession;
                isConnectionFinished.set(true);
                isConnectionFinished.notifyAll();
            }
        }

        @Override
        public void onSessionRepeated(IoSession ioSession) {
            isRepeatedConnect = true;
            ioSession.closeNow();
        }
    };

    protected synchronized void disconnect() {
        mHandler.removeMessages(CONNECT, mIp);
        mHandler.sendMessage(mHandler.obtainMessage(DISCONNECT));
    }

    @Override
    public boolean handleMessage(Message msg) {
        Log.d(TAG, "handleMessage() called with: msg = [" + msg + "]");
        switch (msg.what) {
            case CONNECT:
                synchronized (isConnectionFinished) {
                    if (isSessionOpened || mIoSession != null) {
                        mHandler.removeMessages(CONNECT, mIp);
                        return true;
                    }
                    ConnectFuture future = mConnector.connect(new InetSocketAddress(msg.obj.toString(), PORT));
                    future.awaitUninterruptibly();

                    if (future.getException() != null) {
                        Log.e(TAG, "handleMessage, connect exception: ", future.getException());
                        isConnectionFinished.set(true);
                        isConnectionFinished.notifyAll();
                    }
                }
                break;
            case DISCONNECT:
                synchronized (isConnectionFinished) {
                    mHandler.removeMessages(CONNECT, mIp);
                    if (!mConnector.isDisposed() && mIoSession != null) {
                        mIoSession.getCloseFuture().awaitUninterruptibly(100);
                        isSessionOpened = false;
                        try {
                            mMobileMessageHandler.inputClosed(mIoSession);
                        } catch (Exception e) {
                            Log.e(TAG, "handleMessage: Get Exception: ", e);
                        }
                        mIoSession = null;
                        mConnector.dispose();
                    }
                    mMobileMessageHandler = null;
                    mHandler = null;
                    mHandlerThread.quit();
                    mHandlerThread = null;
                    isConnectionFinished.set(true);
                    isConnectionFinished.notifyAll();
                }
        }
        return true;
    }

    protected synchronized void initSocketClient(String packageName, String ip, ServiceBinder.BindStateListener listener) {
        if (mConnector == null) {
            throw new NullPointerException("Connector is null pointer!");
        }

        mIp = ip;
        mPackageName = packageName;
        mBindStateListener = listener;
    }

    protected synchronized boolean connect() {
        if (isSessionOpened || mIoSession != null) {
            return true;
        }
        mHandler.sendMessage(mHandler.obtainMessage(CONNECT, mIp));

        synchronized (isConnectionFinished) {
            while (!isConnectionFinished.get()) {
                try {
                    isConnectionFinished.wait(500);
                } catch (InterruptedException e) {
                    Log.e(TAG, "connect: Get InterruptedException: ", e);
                }
            }
            isConnectionFinished.set(false);
        }
        return isSessionOpened;
    }

    protected boolean isConnected() {
        return mMobileMessageHandler != null && mMobileMessageHandler.isConnected();
    }

    protected void sendMessage(String to, StringMessage body) throws MobileException {
        JSONObject jsonMessage = new JSONObject();
        try {
            jsonMessage.put(MessageUtil.MSG_ID, body.getId());
            jsonMessage.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
            jsonMessage.put(MessageUtil.MSG_FROM, mPackageName);
            jsonMessage.put(MessageUtil.MSG_TO, to);
            jsonMessage.put(MessageUtil.MSG_BODY, body.getContent());
        } catch (JSONException e) {
            throw MobileException.getJsonException(e.getMessage(), e);
        }

        if (mMobileMessageHandler != null && mConnector != null) {
            mMobileMessageHandler.sendMessage(jsonMessage.toString(), to);
        }
    }

    protected void sendMessage(String to, BufferMessage body) throws MobileException {
        Log.d(TAG, "sendMessage() called with: to = [" + to + "], body = [" + "id=" + body.getId() + "]");
        ByteMessage byteMessage = new ByteMessage(body.getId(), MessageUtil.getTimestamp(), mPackageName, to, body.getContent());
        byte[] bufferMessage = byteMessage.packByteMessage(byteMessage);

        if (mMobileMessageHandler != null && mConnector != null) {
            mMobileMessageHandler.sendMessage(bufferMessage, to);
        }
    }

    private void setSessionStateChange(SessionStateCallback callback) {
        mMobileMessageHandler.setSessionChangedCallback(callback);
    }

    protected void setConnectionListener(MessageRouter.MessageConnectionListener listener) {
        mMobileMessageHandler.setMessageConnectionListener(listener);
    }

    protected void removeConnectionListener() {
        mMobileMessageHandler.removeMessageConnectionListener();
    }

    protected void setMetaData(String packageName, Set set) {
        mMobileMessageHandler.setMetaData(packageName, set);
    }

    protected void setListenersReady(String from, String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) {
        mMobileMessageHandler.setListenersReady(from, to, connectionStateListener, messageListener);
    }
}