package com.segway.robot.mobile.sdk.connectivity;

import android.util.Log;

import com.segway.robot.sdk.baseconnectivity.ByteMessage;
import com.segway.robot.sdk.baseconnectivity.ConnectionInfo;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Created by gaofeng on 2016/5/9.
 */
public class MobileMessageHandler extends IoHandlerAdapter {

    private static final String TAG = "MobileMessageHandler";
    private static final String MOBILE_HEARTBEAT = "mobile";

    private MessageRouter.MessageConnectionListener mMessageConnectionListener = null;
    private Map<String, String> mRobotMetaDataMap = new HashMap<>();//<hosts,list<data>>
    private IoSession mIoSession = null;
    private SessionStateCallback mSessionStateCallback = null;
    private ConnectionInfo mConnectionInfo = new ConnectionInfo();
    private final Map<String, MobileMessageConnection> mMessageConnectionMap = new HashMap<>();//<robot name , info>
    private final List<ConnectionReadyInfo> mConnectionsReadyInfo = new ArrayList<>();//<robot name, info>

    @Override
    public void messageReceived(IoSession session, Object message) throws Exception {
        super.messageReceived(session, message);

        if (message instanceof byte[]) {
            ByteMessage byteMessage = new ByteMessage();
            byteMessage.unpackMessage((byte[]) message, byteMessage);
            String from = byteMessage.mFrom;
            String to = byteMessage.mTo;
            int id = byteMessage.mId;
            long timestamp = byteMessage.mTimestamp;
            byte[] messageByte = byteMessage.mMessage;
            Log.d(TAG, "messageReceived >>> BufferMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to + ";m-length=" + messageByte.length);
            synchronized (mConnectionsReadyInfo) {
                for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                    if (info.mPackage.equals(to) && info.mPackageTo.equals(from) && info.mMessageListener != null) {
                        BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                        info.mMessageListener.onMessageReceived(bufferMessage);
                    }
                }
            }
        } else {
            JSONObject jsonObject = new JSONObject(message.toString());
            if (jsonObject.has(MessageUtil.ADD_NEW)) {//add new host
                String data = jsonObject.getString(MessageUtil.ADD_NEW);
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                Log.d(TAG, MessageUtil.ADD_NEW + " >>>>> " + data);
                mRobotMetaDataMap.put(from, data);
                if (mConnectionInfo.getMetadata().contains(from)) {
                    MobileMessageConnection messageConnection = new MobileMessageConnection(from);
                    mMessageConnectionMap.put(from, messageConnection);
                    if (mMessageConnectionListener != null) {
                        mMessageConnectionListener.onConnectionCreated(messageConnection);
                    }
                }

            } else if (jsonObject.has(MessageUtil.REMOVE_HOST)) {//remove host
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                Log.d(TAG, MessageUtil.REMOVE_HOST + " >>>>> " + from);
                mRobotMetaDataMap.remove(from);
                ConnectionStateListener listener = mMessageConnectionMap.get(from).getConnectionStateListener();
                if (listener != null) {
                    listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
                }
                mMessageConnectionMap.remove(from);

                synchronized (mConnectionsReadyInfo) {
                    Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
                    while (iterator.hasNext()) {
                        ConnectionReadyInfo info = iterator.next();
                        if (from.equals(info.mPackageTo)) {
                            iterator.remove();
                        }
                    }
                }

            } else if (jsonObject.has(MessageUtil.UPDATE_LIST)) {//update host app list
                JSONObject object = new JSONObject(jsonObject.getString(MessageUtil.UPDATE_LIST));
                mRobotMetaDataMap = MessageUtil.packJsonToMap(object);
                Log.d(TAG, MessageUtil.UPDATE_LIST + " >>>>> " + mRobotMetaDataMap.toString());
                for (String robot : mConnectionInfo.getMetadata()) {
                    if (mRobotMetaDataMap.containsKey(robot)
                            && mRobotMetaDataMap.get(robot).contains(mConnectionInfo.getPackageName())) {
                        MobileMessageConnection messageConnection = new MobileMessageConnection(robot);
                        mMessageConnectionMap.put(robot, messageConnection);
                        if (mMessageConnectionListener != null) {
                            mMessageConnectionListener.onConnectionCreated(messageConnection);
                        }
                    }
                }

            } else if (jsonObject.has(MessageUtil.REMOVE_DEAD)) {
                String name = jsonObject.getString(MessageUtil.REMOVE_DEAD);
                String hostMeta = mRobotMetaDataMap.get(name);
                Log.d(TAG, MessageUtil.REMOVE_DEAD + " >>>>> " + name + ";" + hostMeta);
                if (hostMeta != null && hostMeta.contains(mConnectionInfo.getPackageName())
                        && mConnectionInfo.getMetadata().contains(name)) {
                    ConnectionStateListener listener = mMessageConnectionMap.get(name).getConnectionStateListener();
                    if (listener != null) {
                        listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
                    }
                    mMessageConnectionMap.remove(name);
                }

                synchronized (mConnectionsReadyInfo) {
                    Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
                    while (iterator.hasNext()) {
                        ConnectionReadyInfo info = iterator.next();
                        if (name.equals(info.mPackageTo)) {
                            iterator.remove();
                        }
                    }
                }
                mRobotMetaDataMap.remove(name);

            } else if (jsonObject.has(MessageUtil.MESSAGE_LISTENER_READY)) {
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                String to = jsonObject.getString(MessageUtil.MESSAGE_LISTENER_READY);
                Log.d(TAG, MessageUtil.MESSAGE_LISTENER_READY + " >>>>> " + "from= " + from + "; to=" + to);
                synchronized (mConnectionsReadyInfo) {
                    for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                        if (info.mPackage.equals(to) && info.mPackageTo.equals(from) && info.mConnectionStateListener != null) {
                            info.mConnectionStateListener.onOpened();
                            Log.d(TAG, "on opened = " + from);
                        }
                    }
                }

            } else if (jsonObject.has(MessageUtil.REPEATED_CONNECTION)) {
                Log.w(TAG, MessageUtil.REPEATED_CONNECTION + " >>>>> " + "robot has been connected");
                mSessionStateCallback.onSessionRepeated(session);

            } else {
                String from = jsonObject.getString(MessageUtil.MSG_FROM);
                String to = jsonObject.getString(MessageUtil.MSG_TO);
                String body = jsonObject.getString(MessageUtil.MSG_BODY);
                long time = jsonObject.getLong(MessageUtil.TIMESTAMP);
                int id = jsonObject.getInt(MessageUtil.MSG_ID);
                Log.d(TAG, "messageReceived >>> StringMessage >>> " + "id=" + id + ";timestamp=" + time + ";from=" + from + ";to=" + to);
                synchronized (mConnectionsReadyInfo) {
                    for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                        if (from.equals(info.mPackageTo) && to.equals(info.mPackage) && info.mMessageListener != null) {
                            StringMessage stringMessage = new StringMessage(time, body, id);
                            info.mMessageListener.onMessageReceived(stringMessage);
                        }
                    }
                }
            }

        }
    }

    @Override
    public void messageSent(IoSession session, Object message) throws Exception {
        super.messageSent(session, message);

        if (message instanceof byte[]) {
            ByteMessage byteMessage = new ByteMessage();
            byteMessage.unpackMessage((byte[]) message, byteMessage);
            String from = byteMessage.mFrom;
            String to = byteMessage.mTo;
            int id = byteMessage.mId;
            long timestamp = byteMessage.mTimestamp;
            byte[] messageByte = byteMessage.mMessage;
            Log.d(TAG, "messageSent >>> BufferMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
            for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
                if (to.equals(connection.getName()) && connection.getMessageListener() != null) {
                    BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                    connection.getMessageListener().onMessageSent(bufferMessage);
                }
            }
        } else {
            JSONObject json = new JSONObject(message.toString());
            if (checkMessageSent(json)) {    //to check message sent
                String to = json.getString(MessageUtil.MSG_TO);
                String from = json.getString(MessageUtil.MSG_FROM);
                String body = json.getString(MessageUtil.MSG_BODY);
                long timestamp = json.getLong(MessageUtil.TIMESTAMP);
                int id = json.getInt(MessageUtil.MSG_ID);
                Log.d(TAG, "messageSent >>> StringMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
                for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
                    if (to.equals(connection.getName()) && connection.getMessageListener() != null) {
                        StringMessage stringMessage = new StringMessage(timestamp, body, id);
                        connection.getMessageListener().onMessageSent(stringMessage);
                    }
                }
            }
        }
    }

    @Override
    public void sessionOpened(IoSession session) throws Exception {
        super.sessionOpened(session);
        Log.d(TAG, "Mobile sessionOpened:" + session);
        mIoSession = session;
        mSessionStateCallback.onSessionOpened(session);
        mConnectionInfo.setIoSession(session);
        session.write(setSessionOpenMessage());
    }

    @Override
    public void sessionClosed(IoSession session) throws Exception {
        Log.w(TAG, "Mobile sessionClosed: " + session);
        super.sessionClosed(session);
        mIoSession = null;
        mSessionStateCallback.onSessionClosed();
        for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
            ConnectionStateListener listener = connection.getConnectionStateListener();
            if (listener != null) {
                listener.onClosed(MessageError.SESSION_CLOSED);
            }
        }
        mMessageConnectionMap.clear();
        synchronized (mConnectionsReadyInfo) {
            mConnectionsReadyInfo.clear();
        }
    }

    @Override
    public void sessionIdle(IoSession session, IdleStatus status) throws Exception {
        super.sessionIdle(session, status);
    }

    @Override
    public void exceptionCaught(IoSession session, Throwable cause) throws Exception {
        super.exceptionCaught(session, cause);
        Log.e(TAG, "Mobile exceptionCaught: ", cause);
        throw MobileException.getIoSessionException("IoSession exception!", cause);
    }

    protected boolean isConnected() {
        return mIoSession != null;
    }

    protected void sendMessage(Object message, String to) throws MobileException {
        if (mIoSession == null) {
            throw MobileException.getNotConnectedException("Can't send messages to the disconnected device");
        }

        if (!mMessageConnectionMap.containsKey(to)) {
            throw MobileException.getMessageConnectionClosedException("Can't send messages to closed MessageConnection: " + to);
        }
        mIoSession.write(message);
    }

    protected void setSessionChangedCallback(SessionStateCallback callback) {
        mSessionStateCallback = callback;
    }

    protected void setMessageConnectionListener(MessageRouter.MessageConnectionListener listener) {
        mMessageConnectionListener = listener;
    }

    protected void removeMessageConnectionListener() {
        mMessageConnectionListener = null;
    }

    protected void setMetaData(String packageName, Set set) {
        Log.d(TAG, "setMetaData: name=" + packageName + "; set =" + set.toString());
        mConnectionInfo.setPackageName(packageName);
        mConnectionInfo.setMetadata(set);
    }

    private String setSessionOpenMessage() {
        JSONObject json = new JSONObject();
        try {
            json.put(MessageUtil.MSG_FROM, mConnectionInfo.getPackageName());
            json.put(MessageUtil.SESSION_OPEN, MessageUtil.packSetToString(mConnectionInfo.getMetadata()));
        } catch (JSONException e) {
            Log.e(TAG, "JSONException: ", e);
        }
        return json.toString();
    }

    protected void setListenersReady(String from, String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) {
        ConnectionReadyInfo info = new ConnectionReadyInfo(from, to, connectionStateListener, messageListener);
        mConnectionsReadyInfo.add(info);
        sendListenerReadyInfo(from, to);
    }

    private void sendListenerReadyInfo(String packageName, String to) {
        if (isConnected()) {
            JSONObject object = new JSONObject();
            try {
                object.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
                object.put(MessageUtil.MSG_FROM, packageName);
                object.put(MessageUtil.MESSAGE_LISTENER_READY, to);
            } catch (JSONException e) {
                Log.e(TAG, "Message is not json", e);
            }

            mConnectionInfo.getIoSession().write(object.toString());
        }
    }

    private boolean checkMessageSent(JSONObject json) {
        return !json.has(MOBILE_HEARTBEAT) && !json.has(MessageUtil.ADD_NEW)
                && !json.has(MessageUtil.UPDATE_LIST) && !json.has(MessageUtil.REMOVE_HOST)
                && !json.has(MessageUtil.REMOVE_DEAD) && !json.has(MessageUtil.MESSAGE_LISTENER_READY)
                && !json.has(MessageUtil.SESSION_OPEN);
    }

    protected final class ConnectionReadyInfo {
        protected String mPackage;
        protected String mPackageTo;
        protected MessageListener mMessageListener = null;
        protected ConnectionStateListener mConnectionStateListener = null;

        private ConnectionReadyInfo(String packageName, String to, ConnectionStateListener listener, MessageListener iMessageListener) {
            mPackage = packageName;
            mPackageTo = to;
            mConnectionStateListener = listener;
            mMessageListener = iMessageListener;
        }
    }
}