package com.segway.robot.mobile.sdk.connectivity;


import com.segway.robot.sdk.base.log.Logger;
import com.segway.robot.sdk.baseconnectivity.ByteMessage;
import com.segway.robot.sdk.baseconnectivity.ConnectionInfo;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.ConnectionStateListener;
import com.segway.robot.sdk.baseconnectivity.MessageConnection.MessageListener;
import com.segway.robot.sdk.baseconnectivity.MessageRouter;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.session.IoSession;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by gaofeng on 2016/5/9.
 */
public class MobileMessageHandler extends IoHandlerAdapter {

    private static final String TAG = "MobileMessageHandler";
    private static final String MOBILE_HEARTBEAT = "mobile";

    private MessageRouter.MessageConnectionListener mMessageConnectionListener = null;
    private Map<String, String> mRobotMetaDataMap = new ConcurrentHashMap<>();//<hosts,list<data>>
    private IoSession mIoSession = null;
    private SessionStateCallback mSessionStateCallback = null;
    private ConnectionInfo mConnectionInfo = new ConnectionInfo();
    private final Map<String, MobileMessageConnection> mMessageConnectionMap = new ConcurrentHashMap<>();//<robot name , info>
    private final List<ConnectionReadyInfo> mConnectionsReadyInfo = new ArrayList<>();//<robot name, info>

    @Override
    public void messageReceived(IoSession session, Object message) throws Exception {
        super.messageReceived(session, message);

        if (message instanceof byte[]) {
            receiveByteMessage(message);
        } else {
            JSONObject jsonObject = new JSONObject(message.toString());
            if (jsonObject.has(MessageUtil.ADD_NEW)) {//add new host
                receiveAddNew(jsonObject);
            } else if (jsonObject.has(MessageUtil.REMOVE_HOST)) {//remove host
                receiveRemoveHost(jsonObject);
            } else if (jsonObject.has(MessageUtil.UPDATE_LIST)) {//update host app list
                receiveUpdateList(jsonObject);
            } else if (jsonObject.has(MessageUtil.REMOVE_DEAD)) {
                receiveRemoveDeadHost(jsonObject);
            } else if (jsonObject.has(MessageUtil.MESSAGE_LISTENER_READY)) {
                receiveListenerReady(jsonObject);
            } else if (jsonObject.has(MessageUtil.REPEATED_CONNECTION)) {
                Logger.w(TAG, MessageUtil.REPEATED_CONNECTION + " >>>>> ip="
                        + jsonObject.getString(MessageUtil.REPEATED_CONNECTION) + ", robot has been connected");
                mSessionStateCallback.onSessionRepeated(session);
            } else {
                receiveStringMessage(jsonObject);
            }
        }
    }

    @Override
    public void messageSent(IoSession session, Object message) throws Exception {
        super.messageSent(session, message);

        if (message instanceof byte[]) {
            onByteMessageSent(message);
        } else {
            onStringMessageSent(message);
        }
    }

    @Override
    public void sessionOpened(IoSession session) throws Exception {
        super.sessionOpened(session);
        Logger.d(TAG, "Mobile sessionOpened:" + session);
        mIoSession = session;
        mSessionStateCallback.onSessionOpened(session);
        mConnectionInfo.setIoSession(session);
        session.write(setSessionOpenMessage());
    }

    @Override
    public void sessionClosed(IoSession session) throws Exception {
        Logger.w(TAG, "Mobile sessionClosed: " + session);
        super.sessionClosed(session);
        doIoSessionClosed(null);
    }

    @Override
    public void sessionIdle(IoSession session, IdleStatus status) throws Exception {
        super.sessionIdle(session, status);
    }

    @Override
    public void exceptionCaught(IoSession session, Throwable cause) throws Exception {
        super.exceptionCaught(session, cause);
        Logger.e(TAG, "Mobile exceptionCaught: ", cause);
        if (cause instanceof IOException) {
            doIoSessionClosed(cause);
        }
        throw MobileException.getIoSessionException("IoSession exception!", cause);
    }

    private void receiveByteMessage(Object message) throws Exception {
        ByteMessage byteMessage = new ByteMessage();
        byteMessage.unpackMessage((byte[]) message, byteMessage);
        String from = byteMessage.mFrom;
        String to = byteMessage.mTo;
        int id = byteMessage.mId;
        long timestamp = byteMessage.mTimestamp;
        byte[] messageByte = byteMessage.mMessage;
        Logger.v(TAG, "messageReceived >>> BufferMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to + ";m-length=" + messageByte.length);
        synchronized (mConnectionsReadyInfo) {
            for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                if (info.mPackage.equals(to) && info.mPackageTo.equals(from) && info.mMessageListener != null) {
                    BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                    info.mMessageListener.onMessageReceived(bufferMessage);
                }
            }
        }
    }

    private void receiveStringMessage(JSONObject jsonObject) throws Exception {
        String from = jsonObject.getString(MessageUtil.MSG_FROM);
        String to = jsonObject.getString(MessageUtil.MSG_TO);
        String body = jsonObject.getString(MessageUtil.MSG_BODY);
        long time = jsonObject.getLong(MessageUtil.TIMESTAMP);
        int id = jsonObject.getInt(MessageUtil.MSG_ID);
        Logger.v(TAG, "messageReceived >>> StringMessage >>> " + "id=" + id + ";timestamp=" + time + ";from=" + from + ";to=" + to);
        synchronized (mConnectionsReadyInfo) {
            for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                if (from.equals(info.mPackageTo) && to.equals(info.mPackage) && info.mMessageListener != null) {
                    StringMessage stringMessage = new StringMessage(time, body, id);
                    info.mMessageListener.onMessageReceived(stringMessage);
                }
            }
        }
    }

    private void receiveAddNew(JSONObject jsonObject) throws Exception {
        String data = jsonObject.getString(MessageUtil.ADD_NEW);
        String from = jsonObject.getString(MessageUtil.MSG_FROM);
        Logger.d(TAG, MessageUtil.ADD_NEW + " >>>>> " + data);
        mRobotMetaDataMap.put(from, data);
        if (mConnectionInfo.getMetadata().contains(from)) {
            MobileMessageConnection messageConnection = new MobileMessageConnection(from);
            mMessageConnectionMap.put(from, messageConnection);
            if (mMessageConnectionListener != null) {
                mMessageConnectionListener.onConnectionCreated(messageConnection);
            }
        }
    }

    private void receiveRemoveHost(JSONObject jsonObject) throws Exception {
        String from = jsonObject.getString(MessageUtil.MSG_FROM);
        Logger.d(TAG, MessageUtil.REMOVE_HOST + " >>>>> " + from);
        mRobotMetaDataMap.remove(from);

        final MobileMessageConnection connection = mMessageConnectionMap.get(from);
        if (connection != null) {
            ConnectionStateListener listener = connection.getConnectionStateListener();
            if (listener != null) {
                listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
            }
            connection.removeConnectionStateListener();
            connection.removeMessageListener();
            mMessageConnectionMap.remove(from);
        }

        synchronized (mConnectionsReadyInfo) {
            Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
            while (iterator.hasNext()) {
                ConnectionReadyInfo info = iterator.next();
                if (from.equals(info.mPackageTo)) {
                    info.clear();
                    iterator.remove();
                }
            }
        }
    }

    private void receiveUpdateList(JSONObject jsonObject) throws Exception {
        JSONObject object = new JSONObject(jsonObject.getString(MessageUtil.UPDATE_LIST));
        mRobotMetaDataMap = MessageUtil.packJsonToMap(object);
        Logger.d(TAG, MessageUtil.UPDATE_LIST + " >>>>> " + mRobotMetaDataMap.toString());
        for (String robot : mConnectionInfo.getMetadata()) {
            if (mRobotMetaDataMap.containsKey(robot)
                    && mRobotMetaDataMap.get(robot).contains(mConnectionInfo.getPackageName())) {
                MobileMessageConnection messageConnection = new MobileMessageConnection(robot);
                mMessageConnectionMap.put(robot, messageConnection);
                if (mMessageConnectionListener != null) {
                    mMessageConnectionListener.onConnectionCreated(messageConnection);
                }
            }
        }
    }

    private void receiveRemoveDeadHost(JSONObject jsonObject) throws Exception {
        String name = jsonObject.getString(MessageUtil.REMOVE_DEAD);
        String hostMeta = mRobotMetaDataMap.get(name);
        Logger.d(TAG, MessageUtil.REMOVE_DEAD + " >>>>> " + name + ";" + hostMeta);
        if (hostMeta != null && hostMeta.contains(mConnectionInfo.getPackageName())
                && mConnectionInfo.getMetadata().contains(name)) {
            final MobileMessageConnection connection = mMessageConnectionMap.get(name);
            if (connection != null) {
                ConnectionStateListener listener = connection.getConnectionStateListener();
                if (listener != null) {
                    listener.onClosed(MessageError.ROBOT_APP_DISCONNECTED);
                }
                connection.removeMessageListener();
                connection.removeConnectionStateListener();
                mMessageConnectionMap.remove(name);
            }
        }

        synchronized (mConnectionsReadyInfo) {
            Iterator<ConnectionReadyInfo> iterator = mConnectionsReadyInfo.iterator();
            while (iterator.hasNext()) {
                ConnectionReadyInfo info = iterator.next();
                if (name.equals(info.mPackageTo)) {
                    iterator.remove();
                }
            }
        }
        mRobotMetaDataMap.remove(name);
    }

    private void receiveListenerReady(JSONObject jsonObject) throws Exception {
        String from = jsonObject.getString(MessageUtil.MSG_FROM);
        String to = jsonObject.getString(MessageUtil.MESSAGE_LISTENER_READY);
        Logger.d(TAG, MessageUtil.MESSAGE_LISTENER_READY + " >>>>> " + "from= " + from + "; to=" + to);
        synchronized (mConnectionsReadyInfo) {
            for (ConnectionReadyInfo info : mConnectionsReadyInfo) {
                if (info.mPackage.equals(to) && info.mPackageTo.equals(from) && info.mConnectionStateListener != null) {
                    info.mConnectionStateListener.onOpened();
                    Logger.d(TAG, "on opened = " + from);
                }
            }
        }
    }

    private void onByteMessageSent(Object message) throws Exception {
        ByteMessage byteMessage = new ByteMessage();
        byteMessage.unpackMessage((byte[]) message, byteMessage);
        String from = byteMessage.mFrom;
        String to = byteMessage.mTo;
        int id = byteMessage.mId;
        long timestamp = byteMessage.mTimestamp;
        byte[] messageByte = byteMessage.mMessage;
        Logger.v(TAG, "messageSent >>> BufferMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
        for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
            if (to.equals(connection.getName()) && connection.getMessageListener() != null) {
                BufferMessage bufferMessage = new BufferMessage(timestamp, messageByte, id);
                connection.getMessageListener().onMessageSent(bufferMessage);
            }
        }
    }

    private void onStringMessageSent(Object message) throws Exception {
        JSONObject json = new JSONObject(message.toString());
        if (checkMessageSent(json)) {    //to check message sent
            String to = json.getString(MessageUtil.MSG_TO);
            String from = json.getString(MessageUtil.MSG_FROM);
            String body = json.getString(MessageUtil.MSG_BODY);
            long timestamp = json.getLong(MessageUtil.TIMESTAMP);
            int id = json.getInt(MessageUtil.MSG_ID);
            Logger.v(TAG, "messageSent >>> StringMessage >>> id=" + id + ";timestamp=" + timestamp + ";from=" + from + ";to=" + to);
            for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
                if (to.equals(connection.getName()) && connection.getMessageListener() != null) {
                    StringMessage stringMessage = new StringMessage(timestamp, body, id);
                    connection.getMessageListener().onMessageSent(stringMessage);
                }
            }
        }
    }

    private void doIoSessionClosed(Throwable cause) {
        mIoSession = null;
        mSessionStateCallback.onSessionClosed();
        for (MobileMessageConnection connection : mMessageConnectionMap.values()) {
            ConnectionStateListener listener = connection.getConnectionStateListener();
            if (listener != null) {
                if (cause != null) {
                    listener.onClosed(cause.getMessage());
                } else {
                    listener.onClosed(MessageError.SESSION_CLOSED);
                }
            }
            connection.removeConnectionStateListener();
            connection.removeMessageListener();
        }
        mMessageConnectionMap.clear();
        synchronized (mConnectionsReadyInfo) {
            mConnectionsReadyInfo.clear();
        }
    }

    protected boolean isConnected() {
        return mIoSession != null;
    }

    protected synchronized void sendMessage(Object message, String to) throws MobileException {
        if (mIoSession == null) {
            throw MobileException.getNotConnectedException("Can't send messages to the disconnected device");
        }

        if (!mMessageConnectionMap.containsKey(to)) {
            throw MobileException.getMessageConnectionClosedException("Can't send messages to closed MessageConnection: " + to);
        }
        mIoSession.write(message);
    }

    protected void setSessionChangedCallback(SessionStateCallback callback) {
        mSessionStateCallback = callback;
    }

    protected void setMessageConnectionListener(MessageRouter.MessageConnectionListener listener) {
        mMessageConnectionListener = listener;
    }

    protected void removeMessageConnectionListener() {
        mMessageConnectionListener = null;
    }

    protected void setMetaData(String packageName, Set set) {
        Logger.d(TAG, "setMetaData: name=" + packageName + "; set =" + set.toString());
        mConnectionInfo.setPackageName(packageName);
        mConnectionInfo.setMetadata(set);
    }

    private String setSessionOpenMessage() {
        JSONObject json = new JSONObject();
        try {
            json.put(MessageUtil.MSG_FROM, mConnectionInfo.getPackageName());
            json.put(MessageUtil.SESSION_OPEN, MessageUtil.packSetToString(mConnectionInfo.getMetadata()));
        } catch (JSONException e) {
            Logger.e(TAG, "JSONException: ", e);
        }
        return json.toString();
    }

    protected void setListenersReady(String from, String to, ConnectionStateListener connectionStateListener, MessageListener messageListener) {
        ConnectionReadyInfo info = new ConnectionReadyInfo(from, to, connectionStateListener, messageListener);
        mConnectionsReadyInfo.add(info);
        sendListenerReadyInfo(from, to);
    }

    private void sendListenerReadyInfo(String packageName, String to) {
        if (isConnected()) {
            JSONObject object = new JSONObject();
            try {
                object.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
                object.put(MessageUtil.MSG_FROM, packageName);
                object.put(MessageUtil.MESSAGE_LISTENER_READY, to);
            } catch (JSONException e) {
                Logger.e(TAG, "Message is not json", e);
            }

            mConnectionInfo.getIoSession().write(object.toString());
        }
    }

    protected void sendUnbindInfo(String packageName) {
        if (isConnected()) {
            JSONObject object = new JSONObject();
            try {
                object.put(MessageUtil.TIMESTAMP, MessageUtil.getTimestamp());
                object.put(MessageUtil.MOBILE_UNBIND, packageName);
            } catch (JSONException e) {
                Logger.e(TAG, "Message is not json", e);
            }

            mConnectionInfo.getIoSession().write(object.toString());
        }
    }

    private boolean checkMessageSent(JSONObject json) {
        return !json.has(MOBILE_HEARTBEAT) && !json.has(MessageUtil.ADD_NEW)
                && !json.has(MessageUtil.UPDATE_LIST) && !json.has(MessageUtil.REMOVE_HOST)
                && !json.has(MessageUtil.REMOVE_DEAD) && !json.has(MessageUtil.MESSAGE_LISTENER_READY)
                && !json.has(MessageUtil.SESSION_OPEN) && !json.has(MessageUtil.MOBILE_UNBIND);
    }

    protected final class ConnectionReadyInfo {
        protected String mPackage;
        protected String mPackageTo;
        protected MessageListener mMessageListener = null;
        protected ConnectionStateListener mConnectionStateListener = null;

        private ConnectionReadyInfo(String packageName, String to, ConnectionStateListener listener, MessageListener iMessageListener) {
            mPackage = packageName;
            mPackageTo = to;
            mConnectionStateListener = listener;
            mMessageListener = iMessageListener;
        }

        void clear() {
            mMessageListener = null;
            mConnectionStateListener = null;
        }
    }
}