package com.segway.robot.mobile.sdk.connectivity;

import com.segway.robot.sdk.baseconnectivity.Message;
import com.segway.robot.sdk.baseconnectivity.MessageConnection;
import com.segway.robot.sdk.baseconnectivity.MessageUtil;

/**
 * This class provides methods to set the message listener and send messages to
 * the specific robot applications.
 */
public class MobileMessageConnection extends MessageConnection {
    private static final String TAG = "MobileMessageConnection";
    private String mName = null;
    private MessageListener mMessageListener = null;
    private ConnectionStateListener mConnectionStateListener = null;
    private MobileMessageRouter mMobileMessageRouter = null;

    protected MobileMessageConnection(String name) {
        mName = name;
        mMobileMessageRouter = MobileMessageRouter.getInstance();
    }

    /**
     * Get the package name of the specific robot application that is connected to.
     *
     * @return the package name of the specific robot application.
     */
    @Override
    public String getName() {
        return mName;
    }

    /**
     * Set the ConnectionStateListener for the connection state and MessageListener for the message state.
     *
     * @param connectionStateListener the ConnectionStateListener to listen to the connection state.
     * @param messageListener         the MessageListener to listen to the state of receiving and sending message.
     */
    @Override
    public void setListeners(ConnectionStateListener connectionStateListener, MessageListener messageListener) throws MobileException {
        if (connectionStateListener == null) {
            throw new IllegalArgumentException("connectionStateListener can't be null!");
        }

        if (messageListener == null) {
            throw new IllegalArgumentException("MessageListener can't be null!");
        }
        mConnectionStateListener = connectionStateListener;
        mMessageListener = messageListener;
        mMobileMessageRouter.setListenersReady(mName, mConnectionStateListener, mMessageListener);
    }

    /**
     * Get the MessageListener of MessageConnection.
     *
     * @return the MessageListener that has been set.
     */
    @Override
    public MessageListener getMessageListener() {
        return mMessageListener;
    }

    /**
     * Get the ConnectionStateListener of MessageConnection.
     *
     * @return the ConnectionStateListener that has been set.
     */
    @Override
    public ConnectionStateListener getConnectionStateListener() {
        return mConnectionStateListener;
    }

    /**
     * Remove the MessageListener of MessageConnection.
     */
    @Override
    public void removeMessageListener() {
        mMessageListener = null;
    }

    /**
     * Remove the ConnectionStateListener of MessageConnection.
     */
    @Override
    public void removeConnectionStateListener() {
        mConnectionStateListener = null;
    }

    /**
     * Send messages to the specific robot applications.
     *
     * @param message the message to be sent.
     * @throws MobileException If the input argument is illegal, throw IllegalArgumentException,
     *                         otherwise display the detailed {@link MobileException}.
     */
    @Override
    public void sendMessage(Message message) throws MobileException {
        if (message == null) {
            throw new IllegalArgumentException("The message can't be null!");
        }
        if (message.getContent() == null) {
            throw new IllegalArgumentException("The message content can't be null!");
        }
        if (message.getContent() instanceof byte[]) {
            sendBufferMessage((BufferMessage) message);
        } else {
            sendStringMessage((StringMessage) message);
        }
    }

    private void sendStringMessage(StringMessage message) throws MobileException {
        if (message.getContent().length() > MessageUtil.MAX_STRING_LENGTH) {
            throw MobileException.getMessageTooLongException("The message is too long!");
        }
        mMobileMessageRouter.sendStringMessage(mName, message);
    }

    private void sendBufferMessage(BufferMessage message) throws MobileException {
        if (message.getContent().length > MessageUtil.MAX_BUFFER_LENGTH) {
            throw MobileException.getMessageTooLongException("The message is too long!");
        }
        mMobileMessageRouter.sendBufferMessage(mName, message);
    }
}
